/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.service;

import org.kuali.kfs.module.ld.businessobject.LaborLedgerPendingEntry;
import org.kuali.kfs.module.ld.document.LaborLedgerPostingDocument;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * Defines methods that must be implemented by classes providing a LaborLedgerPendingEntryServiceImpl.
 */
public interface LaborLedgerPendingEntryService {

    /**
     * Does the given account have any labor ledger entries? It is necessary to check this before closing an account.
     *
     * @param chartOfAccountsCode
     * @param accountNumber
     * @return
     */
    boolean hasPendingLaborLedgerEntry(String chartOfAccountsCode, String accountNumber);

    /**
     * @param fieldValues the given search criteria
     * @return true if there is one or more pending entries that have not been processed for the given criteria;
     *         otherwise, false
     */
    boolean hasPendingLaborLedgerEntry(Map fieldValues);

    /**
     * This method generates labor ledger pending entries.
     *
     * @param document
     * @return
     */
    boolean generateLaborLedgerPendingEntries(LaborLedgerPostingDocument document);

    /**
     * @return all entries that have been approved but still in pending entry queue.
     */
    Iterator<LaborLedgerPendingEntry> findApprovedPendingLedgerEntries();

    /**
     * Delete the pending entries with the given financial document approved code
     *
     * @param financialDocumentApprovedCode
     */
    void deleteByFinancialDocumentApprovedCode(String financialDocumentApprovedCode);

    /**
     * This method checks for pending ledger entries that match the current balance inquiry
     *
     * @param fieldValues
     * @param isApproved
     * @return
     */
    Iterator findPendingLedgerEntriesForLedgerBalance(Map fieldValues, boolean isApproved);

    /**
     * Use fieldValues to create a query for matching records of {@link LaborLedgerPendingEntry} instances
     *
     * @param fieldValues properties to match against
     * @param isApproved  Retrieve approved or unapproved entries?
     */
    Collection findPendingEntries(Map fieldValues, boolean isApproved);

    /**
     * delete pending entries with the given document header id
     *
     * @param documentHeaderId
     */
    void delete(String documentHeaderId);
}
