/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.businessobject;

import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kim.api.identity.Person;

import java.math.BigDecimal;
import java.util.Objects;

/**
 * Labor business object for Employee Funding.
 */
public class EmployeeFunding extends LedgerBalance {

    private String name;
    private String csfDeleteCode;
    private String csfFundingStatusCode;
    private BigDecimal csfTimePercent;
    private BigDecimal csfFullTimeEmploymentQuantity;
    private KualiDecimal csfAmount;
    private KualiDecimal currentAmount;
    private KualiDecimal outstandingEncumbrance;
    private KualiDecimal totalAmount;

    /**
     * Although the title of this class is EmployeeFunding, it is really a representation of the
     * AccountStatusCurrentFunds business object, however it is generated using the fiscal year and employee ID.
     */
    public EmployeeFunding() {
        super();
        this.setMonth1Amount(KualiDecimal.ZERO);
        this.setCurrentAmount(KualiDecimal.ZERO);
        this.setOutstandingEncumbrance(KualiDecimal.ZERO);
        this.setTotalAmount(KualiDecimal.ZERO);
    }

    public String getName() {
        Person person = this.getLedgerPerson();
        if (person == null) {
            return LaborConstants.BalanceInquiries.UnknownPersonName;
        }

        return person.getName();
    }

    public void setName(String personName) {
        this.name = personName;
    }

    public KualiDecimal getCsfAmount() {
        return csfAmount;
    }

    public void setCsfAmount(KualiDecimal csfAmount) {
        this.csfAmount = csfAmount;
    }

    public String getCsfDeleteCode() {
        return csfDeleteCode;
    }

    public void setCsfDeleteCode(String csfDeleteCode) {
        this.csfDeleteCode = csfDeleteCode;
    }

    public String getCsfFundingStatusCode() {
        return csfFundingStatusCode;
    }

    public void setCsfFundingStatusCode(String csfFundingStatusCode) {
        this.csfFundingStatusCode = csfFundingStatusCode;
    }

    public BigDecimal getCsfTimePercent() {
        return csfTimePercent;
    }

    public void setCsfTimePercent(BigDecimal csfTimePercent) {
        this.csfTimePercent = csfTimePercent;
    }

    public KualiDecimal getCurrentAmount() {
        return currentAmount;
    }

    public void setCurrentAmount(KualiDecimal currentAmount) {
        this.currentAmount = currentAmount;
    }

    public KualiDecimal getOutstandingEncumbrance() {
        return outstandingEncumbrance;
    }

    public void setOutstandingEncumbrance(KualiDecimal outstandingEncumbrance) {
        this.outstandingEncumbrance = outstandingEncumbrance;
    }

    /**
     * @return a total amount based upon adding any outstanding encumbrance records to the annual balance amount.
     */
    public KualiDecimal getTotalAmount() {
        return this.currentAmount.add(this.outstandingEncumbrance);
    }

    public void setTotalAmount(KualiDecimal totalAmount) {
        this.totalAmount = totalAmount;
    }

    @Override
    public int hashCode() {
        final int PRIME = 31;
        int result = 1;
        result = PRIME * result + (getAccountNumber() == null ? 0 : getAccountNumber().hashCode());
        result = PRIME * result + (getChartOfAccountsCode() == null ? 0 : getChartOfAccountsCode().hashCode());
        result = PRIME * result + (getFinancialObjectCode() == null ? 0 : getFinancialObjectCode().hashCode());
        result = PRIME * result + (getFinancialSubObjectCode() == null ? 0 : getFinancialSubObjectCode().hashCode());
        result = PRIME * result + (getSubAccountNumber() == null ? 0 : getSubAccountNumber().hashCode());
        result = PRIME * result + (getUniversityFiscalYear() == null ? 0 : getUniversityFiscalYear().hashCode());
        result = PRIME * result + (getEmplid() == null ? 0 : getEmplid().hashCode());
        result = PRIME * result + (getPositionNumber() == null ? 0 : getPositionNumber().hashCode());

        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }

        final EmployeeFunding other = (EmployeeFunding) obj;

        if (!Objects.equals(getAccountNumber(), other.getAccountNumber())) {
            return false;
        } else if (!Objects.equals(getChartOfAccountsCode(), other.getChartOfAccountsCode())) {
            return false;
        } else if (!Objects.equals(getFinancialObjectCode(), other.getFinancialObjectCode())) {
            return false;
        } else if (!Objects.equals(getFinancialSubObjectCode(), other.getFinancialSubObjectCode())) {
            return false;
        } else if (!Objects.equals(getSubAccountNumber(), other.getSubAccountNumber())) {
            return false;
        } else if (!Objects.equals(getUniversityFiscalYear(), other.getUniversityFiscalYear())) {
            return false;
        } else if (!Objects.equals(getEmplid(), other.getEmplid())) {
            return false;
        } else {
            return Objects.equals(getPositionNumber(), other.getPositionNumber());
        }

    }

    public BigDecimal getCsfFullTimeEmploymentQuantity() {
        return csfFullTimeEmploymentQuantity;
    }

    public void setCsfFullTimeEmploymentQuantity(BigDecimal csfFullTimeEmploymentQuantity) {
        this.csfFullTimeEmploymentQuantity = csfFullTimeEmploymentQuantity;
    }
}

