/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.batch.service;

import org.kuali.kfs.module.purap.businessobject.AutoClosePurchaseOrderView;
import org.kuali.kfs.module.purap.document.PurchaseOrderDocument;

import java.util.List;

public interface AutoClosePurchaseOrderService {

    /**
     * This gets a list of Purchase Orders in Open status and checks to see if their line item encumbrances are all
     * fully disencumbered and if so then the Purchase Order is closed and notes added to indicate the change occurred
     * in batch.
     *
     * @return boolean true if the job is completed successfully and false otherwise.
     */
    boolean autoCloseFullyDisencumberedOrders();

    void autoClosePurchaseOrder(AutoClosePurchaseOrderView poAutoClose);

    /**
     * - PO status is OPEN
     * - Recurring payment type code is not null
     * - Vendor Choice is not Sub-Contract
     * - PO End Date <= parm date (comes from system parameter)
     * - Verify that the system parameter date entered is not greater than the current date minus three months.
     * If the date entered is invalid, the batch process will halt and an error will be generated.
     * - Close and disencumber all recurring PO's that have end dates less than the system parameter date.
     * - Set the system parameter date to mm/dd/yyyy after processing.
     * - Send email indicating that the job ran and which orders were closed.
     * Mail it to the AUTO_CLOSE_RECURRING_PO_EMAIL_ADDRESSES in system parameter.
     *
     * @return boolean true if the job is completed successfully and false otherwise.
     */
    boolean autoCloseRecurringOrders();

    /**
     * Creates and add a note to the purchase order document using the annotation String in the input parameter. This
     * method is used by the autoCloseRecurringOrders() and autoCloseFullyDisencumberedOrders to add a note to the
     * purchase order to indicate that the purchase order was closed by the batch job.
     *
     * @param purchaseOrderDocument The purchase order document that is being closed by the batch job.
     * @param annotation            The string to appear on the note to be attached to the purchase order.
     */
    void createNoteForAutoCloseOrders(PurchaseOrderDocument purchaseOrderDocument, String annotation);

    /**
     * This method gets all the PurchaseOrderView objects that relate to POs with no recurring payment type, status of
     * 'OPEN', and total encumbrance of 0 that do not have any of the excluded vendor choice codes.
     *
     * @return List of PurchaseOrderAutoClose objects
     */
    List<AutoClosePurchaseOrderView> getAllOpenPurchaseOrdersForAutoClose();
}
