/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.util.cxml;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.sys.batch.service.BatchInputFileService;
import org.kuali.kfs.sys.context.SpringContext;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

public final class B2BParserHelper {

    private static Logger LOG = LogManager.getLogger(B2BParserHelper.class);

    private DocumentBuilder builder;
    private static B2BParserHelper _this;

    protected BatchInputFileService batchInputFileService;
    protected PunchOutSetupResponseFileType punchOutSetupResponseFileType;
    protected B2BPunchOutOrderFileType b2BPunchOutOrderFileType;
    protected B2BPOResponseFileType b2BPOResponseFileType;

    private B2BParserHelper() {

        DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();
        builderFactory.setValidating(false); // It's not needed to validate here
        builderFactory.setIgnoringElementContentWhitespace(true);

        try {
            // This is a funky one. Without setting this "load-external-dtd" feature, even though we're
            // explicitly setting non-validating, the parser will still reach out and retrieve that DTD. If
            // the xml.cxml.org site happens to be down, it'll hang or fail on that dependency.
            //
            // http://xerces.apache.org/xerces2-j/features.html#nonvalidating.load-external-dtd
            builderFactory.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);

            builder = builderFactory.newDocumentBuilder(); // Create the parser
        } catch (ParserConfigurationException e) {
            throw new RuntimeException(e);
        }

    }

    public static B2BParserHelper getInstance() {
        if (_this == null) {
            _this = new B2BParserHelper();
        }
        return _this;
    }

    public synchronized B2BShoppingCart parseShoppingCartXML(String xmlChunk) {

        Document xmlDoc = null;
        try {
            xmlDoc = builder.parse(new ByteArrayInputStream(xmlChunk.getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }

        byte[] xmlDocAsBytes = addXMLNameSpace(xmlDoc, "http://www.kuali.org/kfs/purap/b2bPunchOutOrder");

        B2BPunchOutOrderFileType fileType = getB2BPunchOutOrderFileType();

        B2BShoppingCart cart = (B2BShoppingCart) getBatchInputFileService().parse(fileType, xmlDocAsBytes);

        return cart;

    }

    public synchronized PunchOutSetupResponse parsePunchOutSetupResponse(String xmlChunk) {

        Document xmlDoc = null;
        try {
            xmlDoc = builder.parse(new ByteArrayInputStream(xmlChunk.getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }

        byte[] xmlDocAsBytes = addXMLNameSpace(xmlDoc, "http://www.kuali.org/kfs/purap/b2bPunchOutResponse");

        PunchOutSetupResponseFileType fileType = getPunchOutSetupResponseFileType();

        PunchOutSetupResponse response = (PunchOutSetupResponse) getBatchInputFileService().parse(fileType, xmlDocAsBytes);

        return response;

    }

    public synchronized PurchaseOrderResponse parsePurchaseOrderResponse(String xmlChunk) {

        Document xmlDoc = null;
        try {
            xmlDoc = builder.parse(new ByteArrayInputStream(xmlChunk.getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }

        byte[] xmlDocAsBytes = addXMLNameSpace(xmlDoc, "http://www.kuali.org/kfs/purap/b2bPOResponse");

        B2BPOResponseFileType fileType = getB2BPOResponseFileType();

        PurchaseOrderResponse response = (PurchaseOrderResponse) getBatchInputFileService().parse(fileType, xmlDocAsBytes);

        return response;

    }

    private byte[] addXMLNameSpace(Document xmlDoc,
                                   String nameSpace) {

        Node node = xmlDoc.getDocumentElement();
        Element element = (Element) node;

        element.setAttribute("xmlns", nameSpace);

        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            TransformerFactory transformerFactory = TransformerFactory.newInstance();
            Transformer transformer = transformerFactory.newTransformer();
            Source source = new DOMSource(xmlDoc);
            Result result = new StreamResult(out);
            transformer.transform(source, result);
        } catch (TransformerException e) {
            LOG.fatal("Failed to serialize xml." + e);
            throw new RuntimeException(e);
        }

        return out.toByteArray();
    }

    protected BatchInputFileService getBatchInputFileService() {
        if (batchInputFileService == null) {
            batchInputFileService = SpringContext.getBean(BatchInputFileService.class);
        }
        return batchInputFileService;
    }

    protected void setBatchInputFileService(BatchInputFileService batchInputFileService) {
        this.batchInputFileService = batchInputFileService;
    }

    protected PunchOutSetupResponseFileType getPunchOutSetupResponseFileType() {
        if (punchOutSetupResponseFileType == null) {
            punchOutSetupResponseFileType = SpringContext.getBean(PunchOutSetupResponseFileType.class);
        }
        return punchOutSetupResponseFileType;
    }

    protected void setPunchOutSetupResponseFileType(PunchOutSetupResponseFileType punchOutSetupResponseFileType) {
        this.punchOutSetupResponseFileType = punchOutSetupResponseFileType;
    }

    protected B2BPunchOutOrderFileType getB2BPunchOutOrderFileType() {
        if (b2BPunchOutOrderFileType == null) {
            b2BPunchOutOrderFileType = SpringContext.getBean(B2BPunchOutOrderFileType.class);
        }
        return b2BPunchOutOrderFileType;
    }

    protected void setB2BPunchOutOrderFileType(B2BPunchOutOrderFileType b2BPunchOutOrderFileType) {
        this.b2BPunchOutOrderFileType = b2BPunchOutOrderFileType;
    }

    protected B2BPOResponseFileType getB2BPOResponseFileType() {
        if (b2BPOResponseFileType == null) {
            b2BPOResponseFileType = SpringContext.getBean(B2BPOResponseFileType.class);
        }
        return b2BPOResponseFileType;
    }

    protected void setB2BPOResponseFileType(B2BPOResponseFileType b2BPOResponseFileType) {
        this.b2BPOResponseFileType = b2BPOResponseFileType;
    }
}
