/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document.authorization;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.PurapPropertyConstants;
import org.kuali.kfs.module.purap.businessobject.RequisitionItem;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.document.AccountingDocument;
import org.kuali.rice.kew.api.WorkflowDocument;
import org.kuali.rice.kim.api.identity.Person;

import java.util.Map;
import java.util.Set;

/**
 * Accounting line authorizer for Requisition document which allows adding accounting lines at specified nodes
 */
public class RequisitionAccountingLineAuthorizer extends PurapAccountingLineAuthorizer {

    protected static final String INITIATOR_NODE = "Initiator";
    protected static final String CONTENT_REVIEW_NODE = "Organization";

    /**
     * Allow new lines to be rendered at Initiator node
     */
    @Override
    public boolean renderNewLine(AccountingDocument accountingDocument, String accountingGroupProperty) {
        WorkflowDocument workflowDocument = accountingDocument.getFinancialSystemDocumentHeader().getWorkflowDocument();

        Set<String> currentNodes = workflowDocument.getCurrentNodeNames();
        if (CollectionUtils.isNotEmpty(currentNodes)
                && (currentNodes.contains(RequisitionAccountingLineAuthorizer.INITIATOR_NODE)
                    || currentNodes.contains(RequisitionAccountingLineAuthorizer.CONTENT_REVIEW_NODE))) {
            return true;
        }
        return super.renderNewLine(accountingDocument, accountingGroupProperty);
    }

    @Override
    public Set<String> getUnviewableBlocks(AccountingDocument accountingDocument, AccountingLine accountingLine,
            boolean newLine, Person currentUser) {
        Set<String> unviewableBlocks = super.getUnviewableBlocks(accountingDocument, accountingLine, newLine,
                currentUser);
        unviewableBlocks.remove(KFSPropertyConstants.PERCENT);
        unviewableBlocks.remove(KFSPropertyConstants.AMOUNT);

        return unviewableBlocks;
    }

    /**
     * Overridden to add commodity code to the role qualifiers if the document is at the Commodity Code review node
     * because this qualifier is needed to check if the user should have permission to edit the object code.
     */
    @Override
    protected Map<String, String> getRoleQualifiers(AccountingDocument accountingDocument,
            AccountingLine accountingLine) {
        Map<String, String> roleQualifiers = super.getRoleQualifiers(accountingDocument, accountingLine);

        if (ObjectUtils.isNotNull(accountingLine) && ObjectUtils.isNotNull(accountingDocument.getDocumentHeader())
                && ObjectUtils.isNotNull(accountingDocument.getDocumentHeader().getWorkflowDocument())) {
            final Set<String> currentNodes = accountingDocument.getDocumentHeader().getWorkflowDocument()
                    .getCurrentNodeNames();

            if (currentNodes != null
                    && currentNodes.contains(PurapConstants.RequisitionStatuses.NODE_COMMODITY_CODE_REVIEW)) {
                RequisitionItem purApItem = (RequisitionItem) findTheItemForAccountingLine(accountingDocument,
                        accountingLine);
                if (ObjectUtils.isNotNull(purApItem)) {
                    final String purchasingCommodityCode = purApItem.getPurchasingCommodityCode();
                    if (StringUtils.isNotBlank(purchasingCommodityCode)) {
                        roleQualifiers.put(PurapPropertyConstants.ITEM_COMMODITY_CODE, purchasingCommodityCode);
                    }
                }
            }
        }

        return roleQualifiers;
    }

}
