/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document.web.struts;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.kuali.kfs.kns.document.authorization.DocumentAuthorizer;
import org.kuali.kfs.kns.service.DocumentHelperService;
import org.kuali.kfs.kns.web.struts.action.KualiAction;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.purap.businessobject.PurchaseOrderVendorQuote;
import org.kuali.kfs.module.purap.document.PurchaseOrderDocument;
import org.kuali.kfs.module.purap.document.authorization.DocumentInitiationException;
import org.kuali.kfs.module.purap.document.service.PurchaseOrderService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.context.SpringContext;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;

/**
 * Struts Action for printing Purap documents outside of a document action
 */
public class PrintAction extends KualiAction {

    @Override
    public ActionForward execute(ActionMapping mapping, ActionForm form, HttpServletRequest request,
            HttpServletResponse response) throws Exception {
        String poDocNumber = request.getParameter("poDocNumber");
        Integer vendorQuoteId = new Integer(request.getParameter("vendorQuoteId"));
        if (StringUtils.isEmpty(poDocNumber) || StringUtils.isEmpty(poDocNumber)) {
            throw new RuntimeException();
        }

        PurchaseOrderDocument po = (PurchaseOrderDocument) SpringContext.getBean(DocumentService.class)
                .getByDocumentHeaderId(poDocNumber);
        DocumentAuthorizer documentAuthorizer = SpringContext.getBean(DocumentHelperService.class)
                .getDocumentAuthorizer(po);

        if (!(documentAuthorizer.canInitiate(KFSConstants.FinancialDocumentTypeCodes.PURCHASE_ORDER,
                GlobalVariables.getUserSession().getPerson()) ||
            documentAuthorizer.canInitiate(KFSConstants.FinancialDocumentTypeCodes.CONTRACT_MANAGER_ASSIGNMENT,
                    GlobalVariables.getUserSession().getPerson()))) {
            throw new DocumentInitiationException(KFSKeyConstants.AUTHORIZATION_ERROR_DOCUMENT, new String[]{
                    GlobalVariables.getUserSession().getPerson().getPrincipalName(), "print", "Purchase Order"});
        }

        // get the vendor quote
        PurchaseOrderVendorQuote poVendorQuote = null;
        for (PurchaseOrderVendorQuote vendorQuote : po.getPurchaseOrderVendorQuotes()) {
            if (vendorQuote.getPurchaseOrderVendorQuoteIdentifier().equals(vendorQuoteId)) {
                poVendorQuote = vendorQuote;
                break;
            }
        }

        if (poVendorQuote == null) {
            throw new RuntimeException();
        }

        ByteArrayOutputStream baosPDF = new ByteArrayOutputStream();
        poVendorQuote.setTransmitPrintDisplayed(false);
        try {
            boolean success = SpringContext.getBean(PurchaseOrderService.class)
                    .printPurchaseOrderQuotePDF(po, poVendorQuote, baosPDF);

            if (!success) {
                poVendorQuote.setTransmitPrintDisplayed(true);
                if (baosPDF != null) {
                    baosPDF.reset();
                }
                return mapping.findForward(KFSConstants.MAPPING_BASIC);
            }
            response.setHeader("Cache-Control", "max-age=30");
            response.setContentType("application/pdf");

            String filename = "PURAP_PO_QUOTE_" + po.getPurapDocumentIdentifier() + "_" +
                    System.currentTimeMillis() + ".pdf";
            String contentDispValue = "attachment" + "; filename=" + filename;
            response.setHeader("Content-disposition", contentDispValue);

            response.setContentLength(baosPDF.size());

            ServletOutputStream sos;
            sos = response.getOutputStream();

            baosPDF.writeTo(sos);

            sos.flush();
        } finally {
            if (baosPDF != null) {
                baosPDF.reset();
            }
        }

        return null;
    }

}

