/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/*
 * Created on Mar 9, 2005
 *
 */
package org.kuali.kfs.module.purap.businessobject;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.kuali.kfs.module.purap.service.ElectronicInvoiceMappingService;
import org.kuali.kfs.module.purap.util.ElectronicInvoiceUtils;
import org.kuali.kfs.module.purap.util.cxml.CxmlHeader;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

public class ElectronicInvoice {
    private static BigDecimal zero = new BigDecimal(0.00);

    public static String INVOICE_AMOUNT_TYPE_CODE_ITEM = "ITEM";
    public static String INVOICE_AMOUNT_TYPE_CODE_TAX = "TAX";
    public static String INVOICE_AMOUNT_TYPE_CODE_SPECIAL_HANDLING = "SPHD";
    public static String INVOICE_AMOUNT_TYPE_CODE_SHIPPING = "SHIP";
    public static String INVOICE_AMOUNT_TYPE_CODE_DISCOUNT = "DISC";
    public static String INVOICE_AMOUNT_TYPE_CODE_DEPOSIT = "DPST";
    public static String INVOICE_AMOUNT_TYPE_CODE_DUE = "DUE";
    public static String INVOICE_AMOUNT_TYPE_CODE_EXMT = "EXMT";

    public static boolean FILE_REJECTED = true;
    public static boolean FILE_NOT_REJECTED = false;
    public static boolean FILE_DOES_CONTAIN_REJECTS = true;
    public static boolean FILE_DOES_NOT_CONTAIN_REJECTS = false;

    private CxmlHeader cxmlHeader;
    private String customerNumber;

    private String fileName;
    private String dunsNumber;
    private Integer vendorHeaderID = null;
    private Integer vendorDetailID = null;
    private String vendorName = null;
    private boolean fileRejected = FILE_NOT_REJECTED;
    private boolean containsRejects = FILE_DOES_NOT_CONTAIN_REJECTS;
    private List fileRejectReasons = new ArrayList();

    private ElectronicInvoiceDetailRequestHeader invoiceDetailRequestHeader;
    private List<ElectronicInvoiceOrder> invoiceDetailOrders = new ArrayList<>();
    private ElectronicInvoiceDetailRequestSummary invoiceDetailRequestSummary;

    private String version = "1.2.019";
    private String timestamp;
    private String payloadID;
    private String locale;
    private String deploymentMode = "production";

    public ElectronicInvoice() {
        super();
    }

    private boolean containsLineLevelAmounts() {
        return invoiceDetailRequestHeader.isShippingInLine() || invoiceDetailRequestHeader.isSpecialHandlingInLine()
                || invoiceDetailRequestHeader.isTaxInLine() || invoiceDetailRequestHeader.isDiscountInLine();
    }

    public void addFileRejectReasonToList(ElectronicInvoiceRejectReason reason) {
        this.fileRejectReasons.add(reason);
    }

    public ElectronicInvoicePostalAddress getCxmlPostalAddress(ElectronicInvoiceOrder eio, String roleID,
            String addressName) {
        if (this.invoiceDetailRequestHeader.isShippingInLine()) {
            return eio.getCxmlPostalAddressByRoleID(roleID, addressName);
        } else {
            return this.invoiceDetailRequestHeader.getCxmlPostalAddressByRoleID(roleID, addressName);
        }
    }

    public ElectronicInvoiceContact getCxmlContact(ElectronicInvoiceOrder eio, String roleID) {
        if (this.invoiceDetailRequestHeader.isShippingInLine()) {
            return eio.getCxmlContactByRoleID(roleID);
        } else {
            return this.invoiceDetailRequestHeader.getCxmlContactByRoleID(roleID);
        }
    }

    public String getShippingDateDisplayText(ElectronicInvoiceOrder eio) {
        Date date;
        String dateString;
        if (this.invoiceDetailRequestHeader.isShippingInLine()) {
            date = eio.getInvoiceShippingDate();
            dateString = eio.getInvoiceShippingDateString();
        } else {
            date = this.invoiceDetailRequestHeader.getShippingDate();
            dateString = this.invoiceDetailRequestHeader.getShippingDateString();
        }
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return dateString;
        }
    }

    public String getMasterAgreementIDInfoDateDisplayText(ElectronicInvoiceOrder eio) {
        Date date = eio.getMasterAgreementIDInfoDate();
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return eio.getMasterAgreementIDInfoDateString();
        }
    }

    public String getMasterAgreementReferenceDateDisplayText(ElectronicInvoiceOrder eio) {
        Date date = eio.getMasterAgreementReferenceDate();
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return eio.getMasterAgreementReferenceDateString();
        }
    }

    public String getOrderIDInfoDateDisplayText(ElectronicInvoiceOrder eio) {
        Date date = eio.getOrderIDInfoDate();
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return eio.getOrderIDInfoDateString();
        }
    }

    public String getInvoiceDateDisplayText() {
        Date date = this.invoiceDetailRequestHeader.getInvoiceDate();
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return this.invoiceDetailRequestHeader.getInvoiceDateString();
        }
    }

    public String getInvoiceShippingDescription(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isShippingInLine()) {
            return eio.getInvoiceShippingDescription();
        } else {
            return invoiceDetailRequestSummary.getShippingDescription();
        }
    }

    public String getInvoiceTaxDescription(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isTaxInLine()) {
            return eio.getInvoiceTaxDescription();
        } else {
            return invoiceDetailRequestSummary.getTaxDescription();
        }
    }

    public String getInvoiceSpecialHandlingDescription(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isSpecialHandlingInLine()) {
            return eio.getInvoiceSpecialHandlingDescription();

        } else {
            return invoiceDetailRequestSummary.getSpecialHandlingAmountDescription();
        }
    }

    public BigDecimal getFileTotalAmountForInLineItems(String invoiceLineItemType) {
        BigDecimal total = zero;
        for (ElectronicInvoiceOrder invoiceDetailOrder : this.invoiceDetailOrders) {
            if (INVOICE_AMOUNT_TYPE_CODE_TAX.equalsIgnoreCase(invoiceLineItemType)) {
                total = total.add(invoiceDetailOrder.getInvoiceTaxAmount());
            } else if (INVOICE_AMOUNT_TYPE_CODE_SPECIAL_HANDLING.equalsIgnoreCase(invoiceLineItemType)) {
                total = total.add(invoiceDetailOrder.getInvoiceSpecialHandlingAmount());
            } else if (INVOICE_AMOUNT_TYPE_CODE_SHIPPING.equalsIgnoreCase(invoiceLineItemType)) {
                total = total.add(invoiceDetailOrder.getInvoiceShippingAmount());
            } else if (INVOICE_AMOUNT_TYPE_CODE_DISCOUNT.equalsIgnoreCase(invoiceLineItemType)) {
                total = total.add(invoiceDetailOrder.getInvoiceDiscountAmount());
            }
        }
        return total;
    }

    public BigDecimal getInvoiceSubTotalAmount(ElectronicInvoiceOrder eio) {
        if (this.containsLineLevelAmounts()) {
            return eio.getInvoiceSubTotalAmount();
        } else {
            return invoiceDetailRequestSummary.getInvoiceSubTotalAmount();
        }
    }

    public BigDecimal getInvoiceTaxAmount(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isTaxInLine()) {
            return eio.getInvoiceTaxAmount();
        } else {
            return invoiceDetailRequestSummary.getInvoiceTaxAmount();
        }
    }

    public BigDecimal getInvoiceSpecialHandlingAmount(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isSpecialHandlingInLine()) {
            return eio.getInvoiceSpecialHandlingAmount();
        } else {
            return invoiceDetailRequestSummary.getInvoiceSpecialHandlingAmount();
        }
    }

    public BigDecimal getInvoiceShippingAmount(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isShippingInLine()) {
            return eio.getInvoiceShippingAmount();
        } else {
            return invoiceDetailRequestSummary.getInvoiceShippingAmount();
        }
    }

    public BigDecimal getInvoiceGrossAmount(ElectronicInvoiceOrder eio) {
        if (this.containsLineLevelAmounts()) {
            return eio.getInvoiceGrossAmount();
        } else {
            return invoiceDetailRequestSummary.getInvoiceGrossAmount();
        }
    }

    public BigDecimal getInvoiceDiscountAmount(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isDiscountInLine()) {
            return eio.getInvoiceDiscountAmount();
        } else {
            return invoiceDetailRequestSummary.getInvoiceDiscountAmount();
        }
    }

    public BigDecimal getInvoiceNetAmount(ElectronicInvoiceOrder eio) {
        if (this.containsLineLevelAmounts()) {
            return eio.getInvoiceNetAmount();
        } else {
            return invoiceDetailRequestSummary.getInvoiceNetAmount();
        }
    }

    public BigDecimal getInvoiceDepositAmount() {
        return invoiceDetailRequestSummary.getInvoiceDepositAmount();
    }

    public BigDecimal getInvoiceDueAmount() {
        return invoiceDetailRequestSummary.getInvoiceDueAmount();
    }

    public String getCodeOfLineItemThatContainsInvalidCurrency(String invoiceLineItemType) {
        for (ElectronicInvoiceOrder invoiceDetailOrder : this.invoiceDetailOrders) {
            for (ElectronicInvoiceItem invoiceItem : invoiceDetailOrder.getInvoiceItems()) {
                if (INVOICE_AMOUNT_TYPE_CODE_TAX.equalsIgnoreCase(invoiceLineItemType)) {
                    if (!this.isCodeValidCurrency(invoiceItem.getTaxAmountCurrency())) {
                        return invoiceItem.getTaxAmountCurrency();
                    }
                } else if (INVOICE_AMOUNT_TYPE_CODE_SPECIAL_HANDLING.equalsIgnoreCase(invoiceLineItemType)) {
                    if (!this.isCodeValidCurrency(invoiceItem.getInvoiceLineSpecialHandlingAmountCurrency())) {
                        return invoiceItem.getInvoiceLineSpecialHandlingAmountCurrency();
                    }
                } else if (INVOICE_AMOUNT_TYPE_CODE_SHIPPING.equalsIgnoreCase(invoiceLineItemType)) {
                    if (!this.isCodeValidCurrency(invoiceItem.getInvoiceLineShippingAmountCurrency())) {
                        return invoiceItem.getInvoiceLineShippingAmountCurrency();
                    }
                } else if (INVOICE_AMOUNT_TYPE_CODE_DISCOUNT.equalsIgnoreCase(invoiceLineItemType)) {
                    if (!this.isCodeValidCurrency(invoiceItem.getInvoiceLineDiscountAmountCurrency())) {
                        return invoiceItem.getInvoiceLineDiscountAmountCurrency();
                    }
                }
            }
        }
        return null;
    }

    /**
     * This method contains the mapping check for valid Currency Code(s)
     */
    public String checkCodeForValidCurrency(String code) {
        if (!this.isCodeValidCurrency(code)) {
            return code;
        } else {
            return null;
        }
    }

    /**
     * This method contains the mapping check for valid Currency Code(s)
     */
    public boolean isCodeValidCurrency(String code) {
        if (code != null) {
            for (int i = 0; i < ElectronicInvoiceMappingService.CXML_VALID_CURRENCY_CODES.length; i++) {
                String validCode = ElectronicInvoiceMappingService.CXML_VALID_CURRENCY_CODES[i];
                if (code.equalsIgnoreCase(validCode)) {
                    return true;
                }
            }
        }
        return false;
    }

    public String getInvoiceSubTotalCurrencyIfNotValid(ElectronicInvoiceOrder eio) {
        if (this.containsLineLevelAmounts()) {
            for (ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                String currentCode = this.checkCodeForValidCurrency(eii.getSubTotalAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return this.checkCodeForValidCurrency(invoiceDetailRequestSummary.getSubTotalAmountCurrency());
        }
    }

    public String getInvoiceTaxCurrencyIfNotValid(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isTaxInLine()) {
            for (ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                String currentCode = this.checkCodeForValidCurrency(eii.getTaxAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return this.checkCodeForValidCurrency(invoiceDetailRequestSummary.getTaxAmountCurrency());
        }
    }

    public String getInvoiceSpecialHandlingCurrencyIfNotValid(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isSpecialHandlingInLine()) {
            for (ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                String currentCode = this.checkCodeForValidCurrency(eii.getInvoiceLineSpecialHandlingAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return this.checkCodeForValidCurrency(invoiceDetailRequestSummary.getSpecialHandlingAmountCurrency());
        }
    }

    public String getInvoiceShippingCurrencyIfNotValid(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isShippingInLine()) {
            for (ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                String currentCode = this.checkCodeForValidCurrency(eii.getInvoiceLineShippingAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return this.checkCodeForValidCurrency(invoiceDetailRequestSummary.getShippingAmountCurrency());
        }
    }

    public String getInvoiceGrossCurrencyIfNotValid(ElectronicInvoiceOrder eio) {
        if (this.containsLineLevelAmounts()) {
            for (ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                String currentCode = this.checkCodeForValidCurrency(eii.getInvoiceLineGrossAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return this.checkCodeForValidCurrency(invoiceDetailRequestSummary.getGrossAmountCurrency());
        }
    }

    public String getInvoiceDiscountCurrencyIfNotValid(ElectronicInvoiceOrder eio) {
        if (this.invoiceDetailRequestHeader.isDiscountInLine()) {
            for (ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                String currentCode = this.checkCodeForValidCurrency(eii.getInvoiceLineDiscountAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return this.checkCodeForValidCurrency(invoiceDetailRequestSummary.getDiscountAmountCurrency());
        }
    }

    public String getInvoiceNetCurrencyIfNotValid(ElectronicInvoiceOrder eio) {
        if (this.containsLineLevelAmounts()) {
            for (ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                String currentCode = this.checkCodeForValidCurrency(eii.getInvoiceLineNetAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return this.checkCodeForValidCurrency(invoiceDetailRequestSummary.getNetAmountCurrency());
        }
    }

    public String getInvoiceDepositCurrencyIfNotValid() {
        return this.checkCodeForValidCurrency(invoiceDetailRequestSummary.getDepositAmountCurrency());
    }

    public String getInvoiceDueCurrencyIfNotValid() {
        return this.checkCodeForValidCurrency(invoiceDetailRequestSummary.getDueAmountCurrency());
    }

    public boolean isContainsRejects() {
        return containsRejects;
    }

    public void setContainsRejects(boolean containsRejects) {
        this.containsRejects = containsRejects;
    }

    public String getCustomerNumber() {
        return customerNumber;
    }

    public void setCustomerNumber(String customerNumber) {
        this.customerNumber = customerNumber;
    }

    public CxmlHeader getCxmlHeader() {
        return cxmlHeader;
    }

    public void setCxmlHeader(CxmlHeader cxmlHeader) {
        this.cxmlHeader = cxmlHeader;
    }

    public String getDunsNumber() {
        return dunsNumber;
    }

    public void setDunsNumber(String dunsNumber) {
        this.dunsNumber = dunsNumber;
    }

    public String getFileName() {
        return fileName;
    }

    public void setFileName(String fileName) {
        this.fileName = fileName;
    }

    public boolean isFileRejected() {
        return fileRejected;
    }

    public void setFileRejected(boolean fileRejected) {
        this.fileRejected = fileRejected;
    }

    public List<ElectronicInvoiceRejectReason> getFileRejectReasons() {
        return fileRejectReasons;
    }

    public void setFileRejectReasons(List<ElectronicInvoiceRejectReason> fileRejectReasons) {
        this.fileRejectReasons = fileRejectReasons;
    }

    public List<ElectronicInvoiceOrder> getInvoiceDetailOrders() {
        return invoiceDetailOrders;
    }

    public void setInvoiceDetailOrders(List<ElectronicInvoiceOrder> invoiceDetailOrders) {
        this.invoiceDetailOrders = invoiceDetailOrders;
    }

    public ElectronicInvoiceDetailRequestHeader getInvoiceDetailRequestHeader() {
        return invoiceDetailRequestHeader;
    }

    public void setInvoiceDetailRequestHeader(ElectronicInvoiceDetailRequestHeader invoiceDetailRequestHeader) {
        this.invoiceDetailRequestHeader = invoiceDetailRequestHeader;
    }

    public ElectronicInvoiceDetailRequestSummary getInvoiceDetailRequestSummary() {
        return invoiceDetailRequestSummary;
    }

    public void setInvoiceDetailRequestSummary(ElectronicInvoiceDetailRequestSummary invoiceDetailRequestSummary) {
        this.invoiceDetailRequestSummary = invoiceDetailRequestSummary;
    }

    public Integer getVendorDetailID() {
        return vendorDetailID;
    }

    public void setVendorDetailID(Integer vendorDetailID) {
        this.vendorDetailID = vendorDetailID;
    }

    public Integer getVendorHeaderID() {
        return vendorHeaderID;
    }

    public void setVendorHeaderID(Integer vendorHeaderID) {
        this.vendorHeaderID = vendorHeaderID;
    }

    public String getVendorName() {
        return vendorName;
    }

    public void setVendorName(String vendorName) {
        this.vendorName = vendorName;
    }

    public String getPayloadID() {
        return payloadID;
    }

    public void setPayloadID(String payloadID) {
        this.payloadID = payloadID;
    }

    public String getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(String timestamp) {
        this.timestamp = timestamp;
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(String version) {
        this.version = version;
    }

    public String getLocale() {
        return locale;
    }

    public void setLocale(String locale) {
        this.locale = locale;
    }

    public void addInvoiceDetailOrder(ElectronicInvoiceOrder electronicInvoiceOrder) {
        invoiceDetailOrders.add(electronicInvoiceOrder);
    }

    public ElectronicInvoiceOrder[] getInvoiceDetailOrdersAsArray() {
        if (invoiceDetailOrders.size() > 0) {
            ElectronicInvoiceOrder[] tempOrders = new ElectronicInvoiceOrder[invoiceDetailOrders.size()];
            invoiceDetailOrders.toArray(tempOrders);
            return tempOrders;
        }
        return null;
    }

    public String getDeploymentMode() {
        return deploymentMode;
    }

    public void setDeploymentMode(String deploymentMode) {
        this.deploymentMode = deploymentMode;
    }

    public String toString() {
        ToStringBuilder toString = new ToStringBuilder(this);
        toString.append("version", getVersion());
        toString.append("timestamp", getTimestamp());
        toString.append("payloadID", getPayloadID());
        toString.append("locale", getLocale());
        toString.append("customerNumber", getCustomerNumber());
        toString.append("fileName", getFileName());
        toString.append("deploymentMode", getDeploymentMode());

        toString.append("dunsNumber", getDunsNumber());
        toString.append("vendorHeaderID", getVendorHeaderID());
        toString.append("vendorDetailID", getVendorDetailID());
        toString.append("vendorName", getVendorName());
        toString.append("cxmlHeader", getCxmlHeader());
        toString.append("invoiceDetailRequestHeader", getInvoiceDetailRequestHeader());
        toString.append("invoiceDetailOrders", getInvoiceDetailOrders());
        toString.append("invoiceDetailRequestSummary", getInvoiceDetailRequestSummary());

        return toString.toString();
    }

}
