/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.kew.api.document.WorkflowDocumentService;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;
import org.kuali.kfs.kim.api.identity.Person;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.rules.rule.event.KualiDocumentEvent;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.purap.CreditMemoStatuses;
import org.kuali.kfs.module.purap.PaymentRequestStatuses;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.PurapConstants.CREDIT_MEMO_TYPE_LABELS;
import org.kuali.kfs.module.purap.PurapConstants.PurapDocTypeCodes;
import org.kuali.kfs.module.purap.PurapParameterConstants;
import org.kuali.kfs.module.purap.PurapPropertyConstants;
import org.kuali.kfs.module.purap.PurapWorkflowConstants;
import org.kuali.kfs.module.purap.businessobject.CreditMemoItem;
import org.kuali.kfs.module.purap.businessobject.CreditMemoItemUseTax;
import org.kuali.kfs.module.purap.document.service.AccountsPayableDocumentSpecificService;
import org.kuali.kfs.module.purap.document.service.AccountsPayableService;
import org.kuali.kfs.module.purap.document.service.CreditMemoService;
import org.kuali.kfs.module.purap.document.service.PaymentRequestService;
import org.kuali.kfs.module.purap.document.service.PurapService;
import org.kuali.kfs.module.purap.document.validation.event.AttributedContinuePurapEvent;
import org.kuali.kfs.module.purap.service.PurapGeneralLedgerService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySourceDetail;
import org.kuali.kfs.sys.context.SpringContext;

import java.sql.Date;
import java.sql.Timestamp;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * Credit Memo Document Business Object. Contains the fields associated with the main document table.
 */
public class VendorCreditMemoDocument extends AccountsPayableDocumentBase {

    private static final Logger LOG = LogManager.getLogger();

    protected Integer paymentRequestIdentifier;
    protected String creditMemoNumber;
    protected Date creditMemoDate;
    protected KualiDecimal creditMemoAmount;
    protected Timestamp creditMemoPaidTimestamp;
    protected String itemMiscellaneousCreditDescription;
    protected Date purchaseOrderEndDate;
    protected String vendorAttentionName;

    protected PaymentRequestDocument paymentRequestDocument;

    public VendorCreditMemoDocument() {
        super();
    }

    public boolean isSourceDocumentPaymentRequest() {
        return getPaymentRequestIdentifier() != null;
    }

    public boolean isSourceDocumentPurchaseOrder() {
        return !isSourceDocumentPaymentRequest() && getPurchaseOrderIdentifier() != null;
    }

    public boolean isSourceVendor() {
        return !isSourceDocumentPaymentRequest() && !isSourceDocumentPurchaseOrder();
    }

    /**
     * Overrides the method in PurchasingAccountsPayableDocumentBase to add the criteria specific to Credit Memo
     * Document.
     */
    @Override
    public boolean isInquiryRendered() {
        return !isPostingYearPrior()
               || !getApplicationDocumentStatus().equals(CreditMemoStatuses.APPDOC_COMPLETE)
                 && !getApplicationDocumentStatus().equals(
                           PaymentRequestStatuses.APPDOC_CANCELLED_POST_AP_APPROVE)
                 && !getApplicationDocumentStatus().equals(
                           PaymentRequestStatuses.APPDOC_CANCELLED_IN_PROCESS);
    }

    /**
     * Initializes the values for a new document.
     */
    public void initiateDocument() {
        LOG.debug("initiateDocument() started");
        updateAndSaveAppDocStatus(CreditMemoStatuses.APPDOC_INITIATE);

        Person currentUser = GlobalVariables.getUserSession().getPerson();
        setAccountsPayableProcessorIdentifier(currentUser.getPrincipalId());
        setProcessingCampusCode(currentUser.getCampusCode());
    }

    /**
     * Clear out the initially populated fields.
     */
    public void clearInitFields() {
        LOG.debug("clearDocument() started");

        // Clearing document overview fields
        getDocumentHeader().setDocumentDescription(null);
        getDocumentHeader().setExplanation(null);
        getFinancialSystemDocumentHeader().setFinancialDocumentTotalAmount(null);
        getDocumentHeader().setOrganizationDocumentNumber(null);

        // Clearing document Init fields
        setPurchaseOrderIdentifier(null);
        setCreditMemoNumber(null);
        setCreditMemoDate(null);
        setCreditMemoAmount(null);
        setVendorNumber(null);
        setPaymentRequestIdentifier(null);
    }

    /**
     * @return the type of the Credit Memo that was selected on the init screen. It is based on them entering the
     *         Vendor, PO or PREQ #.
     */
    public String getCreditMemoType() {
        String type = CREDIT_MEMO_TYPE_LABELS.TYPE_VENDOR;
        if (isSourceDocumentPaymentRequest()) {
            type = CREDIT_MEMO_TYPE_LABELS.TYPE_PREQ;
        } else if (isSourceDocumentPurchaseOrder()) {
            type = CREDIT_MEMO_TYPE_LABELS.TYPE_PO;
        }
        return type;
    }

    public boolean isBoNotesSupport() {
        return true;
    }

    /**
     * @return true if po has notes, false if po does not have notes
     */
    public boolean getPurchaseOrderNotes() {
        boolean hasNotes = false;

        if (this.getNotes().size() > 0) {
            hasNotes = true;
        }

        return hasNotes;
    }

    /**
     * @return the indicator text that will appear in the workflow document title
     */
    protected String getTitleIndicator() {
        if (isHoldIndicator()) {
            return PurapConstants.PaymentRequestIndicatorText.HOLD;
        } else {
            return "";
        }
    }

    @Override
    public void doRouteStatusChange(DocumentRouteStatusChange statusChangeEvent) {
        LOG.debug("doRouteStatusChange() started");
        super.doRouteStatusChange(statusChangeEvent);

        try {
            // DOCUMENT PROCESSED
            if (this.getFinancialSystemDocumentHeader().getWorkflowDocument().isProcessed()) {
                updateAndSaveAppDocStatus(CreditMemoStatuses.APPDOC_COMPLETE);
            } else if (this.getFinancialSystemDocumentHeader().getWorkflowDocument().isDisapproved()) {
                // DOCUMENT DISAPPROVED
                String nodeName = SpringContext.getBean(WorkflowDocumentService.class).getCurrentRouteLevelName(
                        this.getFinancialSystemDocumentHeader().getWorkflowDocument());

                String disapprovalStatus = CreditMemoStatuses.getCreditMemoAppDocDisapproveStatuses().get(nodeName);

                if (StringUtils.isBlank(disapprovalStatus)
                        && (CreditMemoStatuses.APPDOC_INITIATE.equals(getApplicationDocumentStatus())
                            || CreditMemoStatuses.APPDOC_IN_PROCESS.equals(getApplicationDocumentStatus()))) {
                    disapprovalStatus = CreditMemoStatuses.APPDOC_CANCELLED_IN_PROCESS;
                    updateAndSaveAppDocStatus(disapprovalStatus);
                }
                if (StringUtils.isNotBlank(disapprovalStatus)) {
                    SpringContext.getBean(AccountsPayableService.class).cancelAccountsPayableDocument(this,
                            nodeName);
                } else {
                    logAndThrowRuntimeException("No status found to set for document being disapproved in node '" +
                            nodeName + "'");
                }
            } else if (this.getFinancialSystemDocumentHeader().getWorkflowDocument().isCanceled()) {
                // DOCUMENT CANCELED
                Set<String> currentNodes = this.getFinancialSystemDocumentHeader().getWorkflowDocument()
                        .getCurrentNodeNames();
                if (CollectionUtils.isNotEmpty(currentNodes)) {
                    String currentNodeName = currentNodes.iterator().next();
                    SpringContext.getBean(AccountsPayableService.class).cancelAccountsPayableDocument(this,
                            currentNodeName);
                }
            }
        } catch (Exception e) {
            logAndThrowRuntimeException("Error saving routing data while saving document with id " +
                    getDocumentNumber(), e);
        }
    }

    /**
     * Hook point for performing actions that occur after a route level change, in this case; Performs logic necessary
     * after full entry has been completed when past Adhoc Review, or sets the AP approval date when past AP review.
     */
    @Override
    public boolean processNodeChange(String newNodeName, String oldNodeName) {
        if (CreditMemoStatuses.NODE_ADHOC_REVIEW.equals(oldNodeName)) {
            SpringContext.getBean(AccountsPayableService.class).performLogicForFullEntryCompleted(this);
        } else if (CreditMemoStatuses.NODE_ACCOUNT_REVIEW.equals(newNodeName) && this.isReopenPurchaseOrderIndicator()) {
            // if we've hit Account node then reopen po
            SpringContext.getBean(PurapService.class).performLogicForCloseReopenPO(this);
        }
        return true;
    }

    @Override
    public String getDocumentTitle() {
        if (SpringContext.getBean(ParameterService.class).getParameterValueAsBoolean(VendorCreditMemoDocument.class,
                PurapParameterConstants.PURAP_OVERRIDE_CM_DOC_TITLE)) {
            return getCustomDocumentTitle();
        }
        return super.getDocumentTitle();
    }

    /**
     * Returns a custom document title based on the workflow document title.
     * Depending on the document status, the PO, vendor, amount, etc may be added to the documents title.
     *
     * @return Customized document title text dependent upon route level.
     */
    protected String getCustomDocumentTitle() {
        String popreq = "";
        if (this.isSourceDocumentPurchaseOrder() || this.isSourceDocumentPaymentRequest()) {
            String poNumber = getPurchaseOrderIdentifier().toString();
            popreq = "PO: " + poNumber;
        }

        String vendorName = StringUtils.trimToEmpty(getVendorName());
        String cmAmount = getGrandTotal().toString();
        String indicator = getTitleIndicator();
        return popreq + " Vendor: " + vendorName + " Amount: " + cmAmount + " " + indicator;
    }

    @Override
    public void saveDocumentFromPostProcessing() {
        SpringContext.getBean(PurapService.class).saveDocumentNoValidation(this);
    }

    @Override
    public Class<CreditMemoItem> getItemClass() {
        return CreditMemoItem.class;
    }

    @Override
    public Class getItemUseTaxClass() {
        return CreditMemoItemUseTax.class;
    }

    @Override
    public PurchasingAccountsPayableDocument getPurApSourceDocumentIfPossible() {
        PurchasingAccountsPayableDocument sourceDocument = null;
        if (isSourceDocumentPaymentRequest()) {
            sourceDocument = getPaymentRequestDocument();
        } else if (isSourceDocumentPurchaseOrder()) {
            sourceDocument = getPurchaseOrderDocument();
        }
        return sourceDocument;
    }

    @Override
    public String getPurApSourceDocumentLabelIfPossible() {
        PurchasingAccountsPayableDocument document = getPurApSourceDocumentIfPossible();
        if (ObjectUtils.isNotNull(document)) {
            return SpringContext.getBean(DataDictionaryService.class).getDocumentLabelByClass(document.getClass());
        }
        return null;
    }

    /**
     * @return the pretax total of the above the line items
     */
    public KualiDecimal getLineItemPreTaxTotal() {
        KualiDecimal lineItemPreTaxTotal = KualiDecimal.ZERO;

        for (CreditMemoItem item : (List<CreditMemoItem>) getItems()) {
            item.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

            if (item.getItemType().isLineItemIndicator() && item.getExtendedPrice() != null) {
                lineItemPreTaxTotal = lineItemPreTaxTotal.add(item.getExtendedPrice());
            }
        }

        return lineItemPreTaxTotal;
    }

    /**
     * @return the total of the above the line items
     */
    public KualiDecimal getLineItemTotal() {
        KualiDecimal lineItemTotal = KualiDecimal.ZERO;

        for (CreditMemoItem item : (List<CreditMemoItem>) getItems()) {
            item.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

            if (item.getItemType().isLineItemIndicator() && item.getTotalAmount() != null) {
                lineItemTotal = lineItemTotal.add(item.getTotalAmount());
            }
        }

        return lineItemTotal;
    }

    /**
     * @return the credit memo total: Sum of above the line - restocking fees + misc amount
     */
    @Override
    public KualiDecimal getGrandTotal() {
        KualiDecimal grandTotal = KualiDecimal.ZERO;

        for (CreditMemoItem item : (List<CreditMemoItem>) getItems()) {
            item.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

            if (item.getTotalAmount() != null) {
                // make sure restocking fee is negative
                if (StringUtils.equals(PurapConstants.ItemTypeCodes.ITEM_TYPE_RESTCK_FEE_CODE,
                        item.getItemTypeCode())) {
                    if (ObjectUtils.isNotNull(item.getExtendedPrice())) {
                        item.setExtendedPrice(item.getExtendedPrice().abs().negated());
                    } else {
                        item.setExtendedPrice(KualiDecimal.ZERO);
                    }
                }
                grandTotal = grandTotal.add(item.getTotalAmount());
            }
        }

        return grandTotal;
    }

    /**
     * @return the credit memo pretax total: Sum of above the line - restocking fees + misc amount
     */
    public KualiDecimal getGrandPreTaxTotal() {
        KualiDecimal grandTotal = KualiDecimal.ZERO;

        for (CreditMemoItem item : (List<CreditMemoItem>) getItems()) {
            item.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

            if (item.getExtendedPrice() != null) {
                // make sure restocking fee is negative
                if (StringUtils.equals(PurapConstants.ItemTypeCodes.ITEM_TYPE_RESTCK_FEE_CODE,
                        item.getItemTypeCode())) {
                    item.setExtendedPrice(item.getExtendedPrice().abs().negated());
                }
                grandTotal = grandTotal.add(item.getExtendedPrice());
            }
        }

        return grandTotal;
    }

    /**
     * @return the credit memo tax amount
     */
    public KualiDecimal getGrandTaxAmount() {
        KualiDecimal grandTotal = KualiDecimal.ZERO;

        for (CreditMemoItem item : (List<CreditMemoItem>) getItems()) {
            item.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

            if (item.getItemTaxAmount() != null) {
                // make sure restocking fee is negative
                if (StringUtils.equals(PurapConstants.ItemTypeCodes.ITEM_TYPE_RESTCK_FEE_CODE,
                        item.getItemTypeCode())) {
                    item.setExtendedPrice(item.getItemTaxAmount().abs().negated());
                }
                grandTotal = grandTotal.add(item.getItemTaxAmount());
            }
        }

        return grandTotal;
    }

    public KualiDecimal getGrandPreTaxTotalExcludingRestockingFee() {
        String[] restockingFeeCode = new String[]{PurapConstants.ItemTypeCodes.ITEM_TYPE_RESTCK_FEE_CODE};
        return this.getTotalPreTaxDollarAmountWithExclusions(restockingFeeCode, true);
    }

    public KualiDecimal getGrandTotalExcludingRestockingFee() {
        String[] restockingFeeCode = new String[]{PurapConstants.ItemTypeCodes.ITEM_TYPE_RESTCK_FEE_CODE};
        return this.getTotalDollarAmountWithExclusions(restockingFeeCode, true);
    }

    public Integer getPaymentRequestIdentifier() {
        return paymentRequestIdentifier;
    }

    public void setPaymentRequestIdentifier(Integer paymentRequestIdentifier) {
        this.paymentRequestIdentifier = paymentRequestIdentifier;
    }

    public String getCreditMemoNumber() {
        return creditMemoNumber;
    }

    public void setCreditMemoNumber(String creditMemoNumber) {
        if (creditMemoNumber != null) {
            creditMemoNumber = creditMemoNumber.toUpperCase(Locale.US);
        }

        this.creditMemoNumber = creditMemoNumber;
    }

    public Date getCreditMemoDate() {
        return creditMemoDate;
    }

    public void setCreditMemoDate(Date creditMemoDate) {
        this.creditMemoDate = creditMemoDate;
    }

    public KualiDecimal getCreditMemoAmount() {
        return creditMemoAmount;
    }

    public void setCreditMemoAmount(KualiDecimal creditMemoAmount) {
        this.creditMemoAmount = creditMemoAmount;
    }

    public String getItemMiscellaneousCreditDescription() {
        return itemMiscellaneousCreditDescription;
    }

    public void setItemMiscellaneousCreditDescription(String itemMiscellaneousCreditDescription) {
        this.itemMiscellaneousCreditDescription = itemMiscellaneousCreditDescription;
    }

    public Timestamp getCreditMemoPaidTimestamp() {
        return creditMemoPaidTimestamp;
    }

    public void setCreditMemoPaidTimestamp(Timestamp creditMemoPaidTimestamp) {
        this.creditMemoPaidTimestamp = creditMemoPaidTimestamp;
    }

    public PaymentRequestDocument getPaymentRequestDocument() {
        if (ObjectUtils.isNull(paymentRequestDocument)
                && ObjectUtils.isNotNull(getPaymentRequestIdentifier())) {
            setPaymentRequestDocument(SpringContext.getBean(PaymentRequestService.class)
                    .getPaymentRequestById(getPaymentRequestIdentifier()));
        }
        return this.paymentRequestDocument;
    }

    public void setPaymentRequestDocument(PaymentRequestDocument paymentRequestDocument) {
        if (ObjectUtils.isNull(paymentRequestDocument)) {
            this.paymentRequestDocument = null;
        } else {
            setPaymentRequestIdentifier(paymentRequestDocument.getPurapDocumentIdentifier());
            this.paymentRequestDocument = paymentRequestDocument;
        }
    }

    /**
     * @deprecated use {@link #getPaymentRequestDocument()}
     */
    @Deprecated
    public PaymentRequestDocument getPaymentRequest() {
        return getPaymentRequestDocument();
    }

    /**
     * @deprecated use {@link #setPaymentRequestDocument(PaymentRequestDocument)}
     */
    @Deprecated
    public void setPaymentRequest(PaymentRequestDocument paymentRequest) {
        setPaymentRequestDocument(paymentRequest);
    }

    /**
     * @deprecated use {@link #getPurchaseOrderDocument()}
     */
    @Deprecated
    public PurchaseOrderDocument getPurchaseOrder() {
        return getPurchaseOrderDocument();
    }

    /**
     * @deprecated use {@link #setPurchaseOrderDocument(PurchaseOrderDocument)}
     */
    @Deprecated
    public void setPurchaseOrder(PurchaseOrderDocument purchaseOrder) {
        setPurchaseOrderDocument(purchaseOrder);
    }

    public Date getPurchaseOrderEndDate() {
        return purchaseOrderEndDate;
    }

    public void setPurchaseOrderEndDate(Date purchaseOrderEndDate) {
        this.purchaseOrderEndDate = purchaseOrderEndDate;
    }

    @Override
    public String getPoDocumentTypeForAccountsPayableDocumentCancel() {
        return PurapDocTypeCodes.PURCHASE_ORDER_CLOSE_DOCUMENT;
    }

    @Override
    public KualiDecimal getInitialAmount() {
        return this.getCreditMemoAmount();
    }

    /**
     * Credit Memo document is first populated on Continue AP Event, and then prepareForSave continues.
     */
    @Override
    public void prepareForSave(KualiDocumentEvent event) {
        // first populate, then call super
        if (event instanceof AttributedContinuePurapEvent) {
            SpringContext.getBean(CreditMemoService.class).populateDocumentAfterInit(this);
        }

        super.prepareForSave(event);
    }

    @Override
    protected boolean isAttachmentRequired() {
        return StringUtils.equalsIgnoreCase("Y", SpringContext.getBean(ParameterService.class)
                .getParameterValueAsString(VendorCreditMemoDocument.class,
                        PurapParameterConstants.PURAP_CM_REQUIRE_ATTACHMENT));
    }

    @Override
    public AccountsPayableDocumentSpecificService getDocumentSpecificService() {
        return SpringContext.getBean(CreditMemoService.class);
    }

    /**
     * Forces GL entries to be approved before document final approval.
     */
    @Override
    public void customizeExplicitGeneralLedgerPendingEntry(GeneralLedgerPendingEntrySourceDetail postable,
            GeneralLedgerPendingEntry explicitEntry) {
        super.customizeExplicitGeneralLedgerPendingEntry(postable, explicitEntry);

        SpringContext.getBean(PurapGeneralLedgerService.class).customizeGeneralLedgerPendingEntry(this,
                (AccountingLine) postable, explicitEntry, getPurchaseOrderIdentifier(),
                getDebitCreditCodeForGLEntries(), PurapDocTypeCodes.CREDIT_MEMO_DOCUMENT,
                isGenerateEncumbranceEntries());

        // CMs do not wait for document final approval to post GL entries; here we are forcing them to be APPROVED
        explicitEntry.setFinancialDocumentApprovedCode(KFSConstants.PENDING_ENTRY_APPROVED_STATUS_CODE.APPROVED);
    }

    @Override
    public Date getTransactionTaxDate() {
        return getCreditMemoDate();
    }

    @Override
    public String getVendorAttentionName() {
        return vendorAttentionName;
    }

    @Override
    public void setVendorAttentionName(String vendorAttentionName) {
        this.vendorAttentionName = vendorAttentionName;
    }

    /**
     * Provides answers to the following splits:
     * RequiresInvoiceAttachment
     */
    @Override
    public boolean answerSplitNodeQuestion(String nodeName) throws UnsupportedOperationException {
        if (nodeName.equals(PurapWorkflowConstants.REQUIRES_IMAGE_ATTACHMENT)) {
            return requiresAccountsPayableReviewRouting();
        }
        throw new UnsupportedOperationException("Cannot answer split question for this node you call \"" + nodeName +
                "\"");
    }

    public String getPaidIndicatorForResult() {
        return getCreditMemoPaidTimestamp() != null ? "Yes" : "No";
    }

    /**
     * Checks all documents notes for attachments.
     *
     * @return true if document does not have an image attached, false otherwise
     */
    @Override
    public boolean documentHasNoImagesAttached() {
        List boNotes = this.getNotes();
        if (ObjectUtils.isNotNull(boNotes)) {
            for (Object obj : boNotes) {
                Note note = (Note) obj;

                note.refreshReferenceObject("attachment");
                if (ObjectUtils.isNotNull(note.getAttachment())
                        && PurapConstants.AttachmentTypeCodes.ATTACHMENT_TYPE_CM_IMAGE.equals(note.getAttachment()
                        .getAttachmentTypeCode())) {
                    return false;
                }
            }
        }
        return true;
    }

}

