/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document.service.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.api.document.WorkflowDocumentService;
import org.kuali.kfs.krad.bo.AdHocRoutePerson;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.PurapKeyConstants;
import org.kuali.kfs.module.purap.PurapParameterConstants;
import org.kuali.kfs.module.purap.PurchaseOrderStatuses;
import org.kuali.kfs.module.purap.businessobject.CorrectionReceivingItem;
import org.kuali.kfs.module.purap.businessobject.ItemType;
import org.kuali.kfs.module.purap.businessobject.LineItemReceivingItem;
import org.kuali.kfs.module.purap.businessobject.LineItemReceivingView;
import org.kuali.kfs.module.purap.businessobject.PurApAccountingLine;
import org.kuali.kfs.module.purap.businessobject.PurchaseOrderItem;
import org.kuali.kfs.module.purap.businessobject.ReceivingItem;
import org.kuali.kfs.module.purap.document.CorrectionReceivingDocument;
import org.kuali.kfs.module.purap.document.LineItemReceivingDocument;
import org.kuali.kfs.module.purap.document.PurchaseOrderAmendmentDocument;
import org.kuali.kfs.module.purap.document.PurchaseOrderDocument;
import org.kuali.kfs.module.purap.document.ReceivingDocument;
import org.kuali.kfs.module.purap.document.dataaccess.ReceivingDao;
import org.kuali.kfs.module.purap.document.service.LogicContainer;
import org.kuali.kfs.module.purap.document.service.PurapService;
import org.kuali.kfs.module.purap.document.service.PurchaseOrderService;
import org.kuali.kfs.module.purap.document.service.ReceivingService;
import org.kuali.kfs.module.purap.document.validation.event.AttributedContinuePurapEvent;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.document.service.FinancialSystemDocumentService;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

@Transactional
public class ReceivingServiceImpl implements ReceivingService {

    private static final Logger LOG = LogManager.getLogger();

    protected PurchaseOrderService purchaseOrderService;
    protected ReceivingDao receivingDao;
    protected DocumentService documentService;
    protected WorkflowDocumentService workflowDocumentService;
    protected ConfigurationService configurationService;
    protected PurapService purapService;
    protected NoteService noteService;
    protected FinancialSystemDocumentService financialSystemDocumentService;
    private ParameterService parameterService;

    @Override
    public void populateReceivingLineFromPurchaseOrder(LineItemReceivingDocument rlDoc) {
        if (rlDoc == null) {
            rlDoc = new LineItemReceivingDocument();
        }

        //retrieve po by doc id
        PurchaseOrderDocument poDoc = purchaseOrderService.getCurrentPurchaseOrder(rlDoc.getPurchaseOrderIdentifier());

        if (poDoc != null) {
            rlDoc.populateReceivingLineFromPurchaseOrder(poDoc);
        }
    }

    @Override
    public void populateCorrectionReceivingFromReceivingLine(CorrectionReceivingDocument rcDoc) {
        if (rcDoc == null) {
            rcDoc = new CorrectionReceivingDocument();
        }

        //retrieve receiving line by doc id
        LineItemReceivingDocument rlDoc = rcDoc.getLineItemReceivingDocument();

        if (rlDoc != null) {
            rcDoc.populateCorrectionReceivingFromReceivingLine(rlDoc);
        }
    }

    @Override
    public void populateAndSaveLineItemReceivingDocument(LineItemReceivingDocument rlDoc) {
        documentService.saveDocument(rlDoc, AttributedContinuePurapEvent.class);
    }

    @Override
    public void populateCorrectionReceivingDocument(CorrectionReceivingDocument rcDoc) {
        populateCorrectionReceivingFromReceivingLine(rcDoc);
    }

    @Override
    public boolean canCreateLineItemReceivingDocument(Integer poId, String receivingDocumentNumber) throws RuntimeException {
        PurchaseOrderDocument po = purchaseOrderService.getCurrentPurchaseOrder(poId);
        return canCreateLineItemReceivingDocument(po, receivingDocumentNumber);
    }

    @Override
    public boolean canCreateLineItemReceivingDocument(PurchaseOrderDocument po) throws RuntimeException {
        return canCreateLineItemReceivingDocument(po, null);
    }

    protected boolean canCreateLineItemReceivingDocument(PurchaseOrderDocument po, String receivingDocumentNumber) {
        boolean canCreate = false;

        if (isPurchaseOrderValidForLineItemReceivingDocumentCreation(po)
                && !isLineItemReceivingDocumentInProcessForPurchaseOrder(po.getPurapDocumentIdentifier(),
                    receivingDocumentNumber)
                && !isCorrectionReceivingDocumentInProcessForPurchaseOrder(po.getPurapDocumentIdentifier(), null)) {
            canCreate = true;
        }

        return canCreate;
    }

    @Override
    public boolean isPurchaseOrderActiveForLineItemReceivingDocumentCreation(Integer poId) {
        PurchaseOrderDocument po = purchaseOrderService.getCurrentPurchaseOrder(poId);
        return isPurchaseOrderValidForLineItemReceivingDocumentCreation(po);
    }

    protected boolean isPurchaseOrderValidForLineItemReceivingDocumentCreation(PurchaseOrderDocument po) {
        return po != null && ObjectUtils.isNotNull(po.getPurapDocumentIdentifier())
                && po.isPurchaseOrderCurrentIndicator()
                && (PurchaseOrderStatuses.APPDOC_OPEN.equals(po.getApplicationDocumentStatus())
                || PurchaseOrderStatuses.APPDOC_CLOSED.equals(po.getApplicationDocumentStatus())
                || PurchaseOrderStatuses.APPDOC_PAYMENT_HOLD.equals(po.getApplicationDocumentStatus()));
    }

    @Override
    public boolean canCreateCorrectionReceivingDocument(LineItemReceivingDocument rl) throws RuntimeException {
        return canCreateCorrectionReceivingDocument(rl, null);
    }

    @Override
    public boolean canCreateCorrectionReceivingDocument(LineItemReceivingDocument rl,
            String receivingCorrectionDocNumber) throws RuntimeException {
        boolean canCreate = false;
        WorkflowDocument workflowDocument;

        workflowDocument = workflowDocumentService.loadWorkflowDocument(rl.getDocumentNumber(),
                GlobalVariables.getUserSession().getPerson());

        if (workflowDocument.isFinal()
                && !isCorrectionReceivingDocumentInProcessForReceivingLine(rl.getDocumentNumber(),
                    receivingCorrectionDocNumber)) {
            canCreate = true;
        }

        return canCreate;
    }

    protected boolean isLineItemReceivingDocumentInProcessForPurchaseOrder(Integer poId,
            String receivingDocumentNumber) throws RuntimeException {
        return !getLineItemReceivingDocumentNumbersInProcessForPurchaseOrder(poId, receivingDocumentNumber).isEmpty();
    }

    @Override
    public List<String> getLineItemReceivingDocumentNumbersInProcessForPurchaseOrder(Integer poId,
            String receivingDocumentNumber) {
        List<String> inProcessDocNumbers = new ArrayList<>();
        List<String> docNumbers = receivingDao.getDocumentNumbersByPurchaseOrderId(poId);
        WorkflowDocument workflowDocument;

        for (String docNumber : docNumbers) {
            workflowDocument = workflowDocumentService.loadWorkflowDocument(docNumber,
                    GlobalVariables.getUserSession().getPerson());

            if (!(workflowDocument.isCanceled() || workflowDocument.isException() || workflowDocument.isFinal())
                    && !docNumber.equals(receivingDocumentNumber)) {
                inProcessDocNumbers.add(docNumber);
            }
        }

        return inProcessDocNumbers;
    }

    protected boolean isCorrectionReceivingDocumentInProcessForPurchaseOrder(Integer poId,
            String receivingDocumentNumber) throws RuntimeException {
        return !getCorrectionReceivingDocumentNumbersInProcessForPurchaseOrder(poId, receivingDocumentNumber).isEmpty();
    }

    @Override
    public List<String> getCorrectionReceivingDocumentNumbersInProcessForPurchaseOrder(Integer poId,
            String receivingDocumentNumber) {
        List<String> inProcessDocNumbers = new ArrayList<>();
        List<String> docNumbers = receivingDao.getCorrectionReceivingDocumentNumbersByPurchaseOrderId(poId);
        WorkflowDocument workflowDocument;

        for (String docNumber : docNumbers) {
            workflowDocument = workflowDocumentService.loadWorkflowDocument(docNumber,
                    GlobalVariables.getUserSession().getPerson());

            if (!(workflowDocument.isCanceled() || workflowDocument.isException() || workflowDocument.isFinal())
                    && !docNumber.equals(receivingDocumentNumber)) {
                inProcessDocNumbers.add(docNumber);
            }
        }

        return inProcessDocNumbers;
    }

    protected boolean isCorrectionReceivingDocumentInProcessForReceivingLine(String receivingDocumentNumber,
            String receivingCorrectionDocNumber) throws RuntimeException {
        boolean isInProcess = false;

        List<String> docNumbers = receivingDao.getCorrectionReceivingDocumentNumbersByReceivingLineNumber(
                receivingDocumentNumber);
        WorkflowDocument workflowDocument;

        for (String docNumber : docNumbers) {
            workflowDocument = workflowDocumentService.loadWorkflowDocument(docNumber,
                    GlobalVariables.getUserSession().getPerson());

            if (!(workflowDocument.isCanceled() || workflowDocument.isException() || workflowDocument.isFinal())
                    && !docNumber.equals(receivingCorrectionDocNumber)) {
                isInProcess = true;
                break;
            }
        }

        return isInProcess;
    }

    @Override
    public HashMap<String, String> receivingLineDuplicateMessages(LineItemReceivingDocument rlDoc) {
        HashMap<String, String> msgs;
        msgs = new HashMap<>();
        Integer poId = rlDoc.getPurchaseOrderIdentifier();
        StringBuffer currentMessage = new StringBuffer();
        List<String> docNumbers;

        //check vendor date for duplicates
        if (rlDoc.getShipmentReceivedDate() != null) {
            docNumbers = receivingDao.duplicateVendorDate(poId, rlDoc.getShipmentReceivedDate());
            if (hasDuplicateEntry(docNumbers)) {
                appendDuplicateMessage(currentMessage, PurapKeyConstants.MESSAGE_DUPLICATE_RECEIVING_LINE_VENDOR_DATE,
                        rlDoc.getPurchaseOrderIdentifier());
            }
        }

        //check packing slip number for duplicates
        if (StringUtils.isNotEmpty(rlDoc.getShipmentPackingSlipNumber())) {
            docNumbers = receivingDao.duplicatePackingSlipNumber(poId, rlDoc.getShipmentPackingSlipNumber());
            if (hasDuplicateEntry(docNumbers)) {
                appendDuplicateMessage(currentMessage,
                        PurapKeyConstants.MESSAGE_DUPLICATE_RECEIVING_LINE_PACKING_SLIP_NUMBER,
                        rlDoc.getPurchaseOrderIdentifier());
            }
        }

        //check bill of lading number for duplicates
        if (StringUtils.isNotEmpty(rlDoc.getShipmentBillOfLadingNumber())) {
            docNumbers = receivingDao.duplicateBillOfLadingNumber(poId, rlDoc.getShipmentBillOfLadingNumber());
            if (hasDuplicateEntry(docNumbers)) {
                appendDuplicateMessage(currentMessage,
                        PurapKeyConstants.MESSAGE_DUPLICATE_RECEIVING_LINE_BILL_OF_LADING_NUMBER,
                        rlDoc.getPurchaseOrderIdentifier());
            }
        }

        //add message if one exists
        if (currentMessage.length() > 0) {
            //add suffix
            appendDuplicateMessage(currentMessage, PurapKeyConstants.MESSAGE_DUPLICATE_RECEIVING_LINE_SUFFIX,
                    rlDoc.getPurchaseOrderIdentifier());

            //add msg to map
            msgs.put(PurapConstants.LineItemReceivingDocumentStrings.DUPLICATE_RECEIVING_LINE_QUESTION,
                    currentMessage.toString());
        }

        return msgs;
    }

    /**
     * Looks at a list of doc numbers, but only considers an entry duplicate if the document is in a Final status.
     *
     * @param docNumbers
     * @return
     */
    protected boolean hasDuplicateEntry(List<String> docNumbers) {
        boolean isDuplicate = false;
        WorkflowDocument workflowDocument;

        for (String docNumber : docNumbers) {
            workflowDocument = workflowDocumentService.loadWorkflowDocument(docNumber,
                    GlobalVariables.getUserSession().getPerson());

            //if the doc number exists, and is in final status, consider this a dupe and return
            if (workflowDocument.isFinal()) {
                isDuplicate = true;
                break;
            }
        }

        return isDuplicate;
    }

    protected void appendDuplicateMessage(StringBuffer currentMessage, String duplicateMessageKey, Integer poId) {
        //append prefix if this is first call
        if (currentMessage.length() == 0) {
            String messageText = configurationService.getPropertyValueAsString(
                    PurapKeyConstants.MESSAGE_DUPLICATE_RECEIVING_LINE_PREFIX);
            String prefix = MessageFormat.format(messageText, poId.toString());

            currentMessage.append(prefix);
        }

        currentMessage.append(configurationService.getPropertyValueAsString(duplicateMessageKey));
    }

    @Override
    public void completeCorrectionReceivingDocument(ReceivingDocument correctionDocument) {
        ReceivingDocument receivingDoc =
                ((CorrectionReceivingDocument) correctionDocument).getLineItemReceivingDocument();

        for (CorrectionReceivingItem correctionItem : (List<CorrectionReceivingItem>) correctionDocument.getItems()) {
            if (!StringUtils.equalsIgnoreCase(correctionItem.getItemType().getItemTypeCode(),
                    PurapConstants.ItemTypeCodes.ITEM_TYPE_UNORDERED_ITEM_CODE)) {
                LineItemReceivingItem recItem = receivingDoc.getItem(correctionItem.getItemLineNumber() - 1);
                List<PurchaseOrderItem> purchaseOrderItems = receivingDoc.getPurchaseOrderDocument().getItems();

                if (ObjectUtils.isNotNull(recItem)) {
                    recItem.setItemReceivedTotalQuantity(correctionItem.getItemReceivedTotalQuantity());
                    recItem.setItemReturnedTotalQuantity(correctionItem.getItemReturnedTotalQuantity());
                    recItem.setItemDamagedTotalQuantity(correctionItem.getItemDamagedTotalQuantity());
                }
            }
        }
    }

    /**
     * This method deletes unneeded items and updates the totals on the po and does any additional processing based
     * on items i.e. FYI etc
     *
     * @param receivingDocument receiving document
     */
    @Override
    public void completeReceivingDocument(ReceivingDocument receivingDocument) {
        PurchaseOrderDocument poDoc = null;

        if (receivingDocument instanceof LineItemReceivingDocument) {
            purapService.deleteUnenteredItems(receivingDocument);
            poDoc = receivingDocument.getPurchaseOrderDocument();
        } else if (receivingDocument instanceof CorrectionReceivingDocument) {
            CorrectionReceivingDocument correctionDocument = (CorrectionReceivingDocument) receivingDocument;
            poDoc = purchaseOrderService.getCurrentPurchaseOrder(correctionDocument.getLineItemReceivingDocument()
                    .getPurchaseOrderIdentifier());
        }

        updateReceivingTotalsOnPurchaseOrder(receivingDocument, poDoc);

        //TODO: custom doc specific service hook here for correction to do it's receiving doc update

        purapService.saveDocumentNoValidation(poDoc);

        sendFyiForItems(receivingDocument);

        spawnPoAmendmentForUnorderedItems(receivingDocument, poDoc);

        purapService.saveDocumentNoValidation(receivingDocument);
    }

    @Override
    public void createNoteForReturnedAndDamagedItems(ReceivingDocument recDoc) {
        for (ReceivingItem item : (List<ReceivingItem>) recDoc.getItems()) {
            if (!StringUtils.equalsIgnoreCase(item.getItemType().getItemTypeCode(),
                    PurapConstants.ItemTypeCodes.ITEM_TYPE_UNORDERED_ITEM_CODE)) {
                if (item.getItemReturnedTotalQuantity() != null
                        && item.getItemReturnedTotalQuantity().isGreaterThan(KualiDecimal.ZERO)) {
                    try {
                        String noteString = configurationService.getPropertyValueAsString(
                                PurapKeyConstants.MESSAGE_RECEIVING_LINEITEM_RETURN_NOTE_TEXT);
                        noteString = item.getItemReturnedTotalQuantity().intValue() + " " + noteString + " " +
                                item.getItemLineNumber();
                        addNoteToReceivingDocument(recDoc, noteString);
                    } catch (Exception e) {
                        String errorMsg = "Note Service Exception caught: " + e.getLocalizedMessage();
                        throw new RuntimeException(errorMsg, e);
                    }
                }

                if (item.getItemDamagedTotalQuantity() != null
                        && item.getItemDamagedTotalQuantity().isGreaterThan(KualiDecimal.ZERO)) {
                    try {
                        String noteString = configurationService.getPropertyValueAsString(
                                PurapKeyConstants.MESSAGE_RECEIVING_LINEITEM_DAMAGE_NOTE_TEXT);
                        noteString = item.getItemDamagedTotalQuantity().intValue() + " " + noteString + " " +
                                item.getItemLineNumber();
                        addNoteToReceivingDocument(recDoc, noteString);
                    } catch (Exception e) {
                        String errorMsg = "Note Service Exception caught: " + e.getLocalizedMessage();
                        throw new RuntimeException(errorMsg, e);
                    }
                }
            }
        }
    }

    protected void updateReceivingTotalsOnPurchaseOrder(ReceivingDocument receivingDocument,
            PurchaseOrderDocument poDoc) {
        for (ReceivingItem receivingItem : (List<ReceivingItem>) receivingDocument.getItems()) {
            ItemType itemType = receivingItem.getItemType();
            if (!StringUtils.equalsIgnoreCase(itemType.getItemTypeCode(),
                    PurapConstants.ItemTypeCodes.ITEM_TYPE_UNORDERED_ITEM_CODE)) {
                // TODO: Chris - this method of getting the line out of po should be turned into a method that can get
                // an item based on a combo or itemType and line
                PurchaseOrderItem poItem = (PurchaseOrderItem) poDoc.getItemByLineNumber(
                        receivingItem.getItemLineNumber());

                if (ObjectUtils.isNotNull(poItem)) {
                    KualiDecimal poItemReceivedTotal = poItem.getItemReceivedTotalQuantity();

                    KualiDecimal receivingItemReceivedOriginal = receivingItem.getItemOriginalReceivedTotalQuantity();
                    /*
                     * FIXME: It's coming as null although we set the default value in the LineItemReceivingItem
                     * constructor - mpv
                     */
                    if (ObjectUtils.isNull(receivingItemReceivedOriginal)) {
                        receivingItemReceivedOriginal = KualiDecimal.ZERO;
                    }
                    KualiDecimal receivingItemReceived = receivingItem.getItemReceivedTotalQuantity();
                    KualiDecimal receivingItemTotalReceivedAdjusted = receivingItemReceived.subtract(
                            receivingItemReceivedOriginal);

                    if (ObjectUtils.isNull(poItemReceivedTotal)) {
                        poItemReceivedTotal = KualiDecimal.ZERO;
                    }
                    KualiDecimal poItemReceivedTotalAdjusted = poItemReceivedTotal.add(
                            receivingItemTotalReceivedAdjusted);

                    KualiDecimal receivingItemReturnedOriginal = receivingItem.getItemOriginalReturnedTotalQuantity();
                    if (ObjectUtils.isNull(receivingItemReturnedOriginal)) {
                        receivingItemReturnedOriginal = KualiDecimal.ZERO;
                    }

                    KualiDecimal receivingItemReturned = receivingItem.getItemReturnedTotalQuantity();
                    if (ObjectUtils.isNull(receivingItemReturned)) {
                        receivingItemReturned = KualiDecimal.ZERO;
                    }

                    KualiDecimal receivingItemTotalReturnedAdjusted = receivingItemReturned.subtract(
                            receivingItemReturnedOriginal);

                    poItemReceivedTotalAdjusted = poItemReceivedTotalAdjusted.subtract(
                            receivingItemTotalReturnedAdjusted);

                    poItem.setItemReceivedTotalQuantity(poItemReceivedTotalAdjusted);

                    KualiDecimal poTotalDamaged = poItem.getItemDamagedTotalQuantity();
                    if (ObjectUtils.isNull(poTotalDamaged)) {
                        poTotalDamaged = KualiDecimal.ZERO;
                    }

                    KualiDecimal receivingItemTotalDamagedOriginal = receivingItem
                            .getItemOriginalDamagedTotalQuantity();
                    if (ObjectUtils.isNull(receivingItemTotalDamagedOriginal)) {
                        receivingItemTotalDamagedOriginal = KualiDecimal.ZERO;
                    }

                    KualiDecimal receivingItemTotalDamaged = receivingItem.getItemDamagedTotalQuantity();
                    if (ObjectUtils.isNull(receivingItemTotalDamaged)) {
                        receivingItemTotalDamaged = KualiDecimal.ZERO;
                    }

                    KualiDecimal receivingItemTotalDamagedAdjusted = receivingItemTotalDamaged.subtract(
                            receivingItemTotalDamagedOriginal);

                    poItem.setItemDamagedTotalQuantity(poTotalDamaged.add(receivingItemTotalDamagedAdjusted));

                }
            }
        }
    }

    /**
     * Spawns PO amendments for new unordered items on a receiving document.
     *
     * @param receivingDocument
     * @param po
     */
    protected void spawnPoAmendmentForUnorderedItems(ReceivingDocument receivingDocument, PurchaseOrderDocument po) {
        if (receivingDocument instanceof LineItemReceivingDocument) {
            LineItemReceivingDocument rlDoc = (LineItemReceivingDocument) receivingDocument;

            //if a new item has been added spawn a purchase order amendment
            if (hasNewUnorderedItem((LineItemReceivingDocument) receivingDocument)) {
                String newSessionUserId = KFSConstants.SYSTEM_USER;
                LogicContainer logicToRun = objects -> {
                    LineItemReceivingDocument rlDoc1 = (LineItemReceivingDocument) objects[0];
                    String poDocNumber = (String) objects[1];

                    PurchaseOrderAmendmentDocument amendmentPo =
                            (PurchaseOrderAmendmentDocument) purchaseOrderService
                                    .createAndSavePotentialChangeDocument(poDocNumber,
                                            PurapConstants.PurapDocTypeCodes.PURCHASE_ORDER_AMENDMENT_DOCUMENT,
                                            PurchaseOrderStatuses.APPDOC_AMENDMENT);

                    addUnorderedItemsToAmendment(amendmentPo, rlDoc1);
                    documentService.routeDocument(amendmentPo, null, null);

                    //add note to amendment po document
                    String note = "Purchase Order Amendment " + amendmentPo.getPurapDocumentIdentifier() +
                            " (document id " + amendmentPo.getDocumentNumber() + ") created for new unordered " +
                            "line items due to Receiving (document id " + rlDoc1.getDocumentNumber() + ")";

                    Note noteObj = documentService.createNoteFromDocument(amendmentPo, note);
                    amendmentPo.addNote(noteObj);
                    noteService.save(noteObj);

                    return null;
                };

                purapService.performLogicWithFakedUserSession(newSessionUserId, logicToRun, rlDoc,
                        po.getDocumentNumber());
            }
        }
    }

    /**
     * Checks the item list for newly added items.
     *
     * @param rlDoc
     * @return
     */
    @Override
    public boolean hasNewUnorderedItem(LineItemReceivingDocument rlDoc) {
        boolean itemAdded = false;

        for (LineItemReceivingItem rlItem : (List<LineItemReceivingItem>) rlDoc.getItems()) {
            if (PurapConstants.ItemTypeCodes.ITEM_TYPE_UNORDERED_ITEM_CODE.equals(rlItem.getItemTypeCode())
                    && StringUtils.isNotEmpty(rlItem.getItemReasonAddedCode())) {
                itemAdded = true;
                break;
            }
        }

        return itemAdded;
    }

    /**
     * Adds an unordered item to a po amendment document.
     *
     * @param amendment
     * @param rlDoc
     */
    protected void addUnorderedItemsToAmendment(PurchaseOrderAmendmentDocument amendment,
            LineItemReceivingDocument rlDoc) {
        PurchaseOrderItem poi;

        for (LineItemReceivingItem rlItem : (List<LineItemReceivingItem>) rlDoc.getItems()) {
            if (PurapConstants.ItemTypeCodes.ITEM_TYPE_UNORDERED_ITEM_CODE.equals(rlItem.getItemTypeCode())
                    && StringUtils.isNotEmpty(rlItem.getItemReasonAddedCode())) {
                poi = createPoItemFromReceivingLine(rlItem);
                poi.setDocumentNumber(amendment.getDocumentNumber());

                // add default commodity code from parameter, if commodity code is required on PO and not specified in
                // the unordered item. Note: if we don't add logic to populate commodity code in
                // LineItemReceivingItem, then at this point this field is always empty for unordered item
                if (purchaseOrderService.isCommodityCodeRequiredOnPurchaseOrder()
                        && StringUtils.isEmpty(poi.getPurchasingCommodityCode())) {
                    String defaultCommodityCode = parameterService
                            .getParameterValueAsString(PurchaseOrderAmendmentDocument.class,
                                    PurapParameterConstants.UNORDERED_ITEM_DEFAULT_COMMODITY_CODE);
                    poi.setPurchasingCommodityCode(defaultCommodityCode);
                }

                poi.refreshNonUpdateableReferences();
                amendment.addItem(poi);
            }
        }
    }

    /**
     * Creates a PO item from a receiving line item.
     *
     * @param rlItem
     * @return
     */
    protected PurchaseOrderItem createPoItemFromReceivingLine(LineItemReceivingItem rlItem) {
        PurchaseOrderItem poi = new PurchaseOrderItem();

        poi.setItemActiveIndicator(true);
        poi.setItemTypeCode(rlItem.getItemTypeCode());
        poi.setItemLineNumber(rlItem.getItemLineNumber());
        poi.setItemCatalogNumber(rlItem.getItemCatalogNumber());
        poi.setItemDescription(rlItem.getItemDescription());

        if (rlItem.getItemReturnedTotalQuantity() == null) {
            poi.setItemQuantity(rlItem.getItemReceivedTotalQuantity());
        } else {
            poi.setItemQuantity(rlItem.getItemReceivedTotalQuantity().subtract(rlItem.getItemReturnedTotalQuantity()));
        }

        poi.setItemUnitOfMeasureCode(rlItem.getItemUnitOfMeasureCode());
        poi.setItemUnitPrice(new BigDecimal(0));

        poi.setItemDamagedTotalQuantity(rlItem.getItemDamagedTotalQuantity());
        poi.setItemReceivedTotalQuantity(rlItem.getItemReceivedTotalQuantity());

        return poi;
    }

    /**
     * Creates a list of fiscal officers for new unordered items added to a purchase order.
     *
     * @param recDoc
     * @return
     */
    protected List<AdHocRoutePerson> createFyiFiscalOfficerList(ReceivingDocument recDoc) {
        PurchaseOrderDocument po = recDoc.getPurchaseOrderDocument();
        List<AdHocRoutePerson> adHocRoutePersons = new ArrayList<>();
        Map fiscalOfficers = new HashMap();
        AdHocRoutePerson adHocRoutePerson;

        for (ReceivingItem recItem : (List<ReceivingItem>) recDoc.getItems()) {
            //if this item has an item line number then it is coming from the po
            if (ObjectUtils.isNotNull(recItem.getItemLineNumber())) {
                PurchaseOrderItem poItem = (PurchaseOrderItem) po.getItemByLineNumber(recItem.getItemLineNumber());

                if (poItem.getItemQuantity().isLessThan(poItem.getItemReceivedTotalQuantity()) ||
                    recItem.getItemDamagedTotalQuantity().isGreaterThan(KualiDecimal.ZERO)) {

                    // loop through accounts and pull off fiscal officer
                    for (PurApAccountingLine account : poItem.getSourceAccountingLines()) {

                        //check for dupes of fiscal officer
                        if (!fiscalOfficers.containsKey(
                                account.getAccount().getAccountFiscalOfficerUser().getPrincipalName())) {

                            //add fiscal officer to list
                            fiscalOfficers.put(account.getAccount().getAccountFiscalOfficerUser().getPrincipalName(),
                                    account.getAccount().getAccountFiscalOfficerUser().getPrincipalName());

                            //create AdHocRoutePerson object and add to list
                            adHocRoutePerson = new AdHocRoutePerson();
                            adHocRoutePerson.setId(account.getAccount().getAccountFiscalOfficerUser()
                                    .getPrincipalName());
                            adHocRoutePerson.setActionRequested(KFSConstants.WORKFLOW_FYI_REQUEST);
                            adHocRoutePersons.add(adHocRoutePerson);
                        }
                    }
                }
            }
        }

        return adHocRoutePersons;
    }

    /**
     * Sends an FYI to fiscal officers for new unordered items.
     *
     * @param recDoc
     */
    protected void sendFyiForItems(ReceivingDocument recDoc) {
        List<AdHocRoutePerson> fyiList = createFyiFiscalOfficerList(recDoc);
        String annotation = "Notification of Item exceeded Quantity or Damaged" + "(document id " +
                recDoc.getDocumentNumber() + ")";
        String responsibilityNote = "Please Review";

        for (AdHocRoutePerson adHocPerson : fyiList) {
            recDoc.appSpecificRouteDocumentToUser(recDoc.getDocumentHeader().getWorkflowDocument(),
                adHocPerson.getPerson().getPrincipalId(), annotation, responsibilityNote);
        }
    }

    @Override
    public void addNoteToReceivingDocument(ReceivingDocument receivingDocument, String note) throws Exception {
        Note noteObj = documentService.createNoteFromDocument(receivingDocument, note);
        receivingDocument.addNote(noteObj);
        noteService.save(noteObj);
    }

    @Override
    public String getReceivingDeliveryCampusCode(PurchaseOrderDocument po) {
        String deliveryCampusCode = "";
        String latestDocumentNumber = "";

        List<LineItemReceivingView> rViews = null;
        WorkflowDocument workflowDocument;
        DateTime latestCreateDate = null;

        //get related views
        if (ObjectUtils.isNotNull(po.getRelatedViews())) {
            rViews = po.getRelatedViews().getRelatedLineItemReceivingViews();
        }

        //if not empty, then grab the latest receiving view
        if (ObjectUtils.isNotNull(rViews) && !rViews.isEmpty()) {
            for (LineItemReceivingView rView : rViews) {
                workflowDocument = workflowDocumentService.loadWorkflowDocument(rView.getDocumentNumber(),
                        GlobalVariables.getUserSession().getPerson());

                //if latest create date is null or the latest is before the current, current is newer
                if (ObjectUtils.isNull(latestCreateDate)
                        || latestCreateDate.isBefore(workflowDocument.getDateCreated())) {
                    latestCreateDate = workflowDocument.getDateCreated();
                    latestDocumentNumber = workflowDocument.getDocumentId();
                }
            }

            //if there is a create date, a latest workflow doc was found
            if (ObjectUtils.isNotNull(latestCreateDate)) {
                LineItemReceivingDocument rlDoc =
                        (LineItemReceivingDocument) documentService.getByDocumentHeaderId(latestDocumentNumber);
                deliveryCampusCode = rlDoc.getDeliveryCampusCode();
            }
        }

        return deliveryCampusCode;
    }

    @Override
    public void approveReceivingDocsForPOAmendment() {
        List<LineItemReceivingDocument> docs = getDocumentsAwaitingPurchaseOrderOpenStatus();
        if (docs != null) {
            for (LineItemReceivingDocument receivingDoc : docs) {
                Set<String> currentNodes = receivingDoc.getDocumentHeader().getWorkflowDocument()
                        .getCurrentNodeNames();
                if (CollectionUtils.isNotEmpty(currentNodes) && currentNodes.contains(
                        PurapConstants.LineItemReceivingDocumentStrings.AWAITING_PO_OPEN_STATUS)) {
                    approveReceivingDoc(receivingDoc);
                }
            }
        }
    }

    protected void approveReceivingDoc(LineItemReceivingDocument receivingDoc) {
        PurchaseOrderDocument poDoc = receivingDoc.getPurchaseOrderDocument();
        if (purchaseOrderService.isPurchaseOrderOpenForProcessing(poDoc)) {
            documentService.approveDocument(receivingDoc, "Approved by the batch job", null);
        }
    }

    /**
     * Gets a list of strings of receiving line item document numbers where
     * applicationDocumentStatus = 'Awaiting Purchase Order Open Status'
     * If there are documents then the document number is added to the list
     *
     * @return list of documentNumbers to retrieve line item receiving documents.
     */
    @Deprecated
    protected List<String> getDocumentsNumbersAwaitingPurchaseOrderOpenStatus() {
        List<String> receivingDocumentNumbers = new ArrayList<>();
        List<LineItemReceivingDocument> receivingDocuments = getDocumentsAwaitingPurchaseOrderOpenStatus();
        for (LineItemReceivingDocument document : receivingDocuments) {
            receivingDocumentNumbers.add(document.getDocumentNumber());
        }
        return receivingDocumentNumbers;
    }

    /**
     * Gets a list of receiving line item documents where
     * applicationDocumentStatus = 'Awaiting Purchase Order Open Status'
     *
     * @return list of line item receiving documents.
     */
    protected List<LineItemReceivingDocument> getDocumentsAwaitingPurchaseOrderOpenStatus() {
        return (List<LineItemReceivingDocument>) financialSystemDocumentService
                .findByApplicationDocumentStatus(LineItemReceivingDocument.class,
                        PurapConstants.LineItemReceivingStatuses.APPDOC_AWAITING_PO_OPEN_STATUS);
    }

    public void setPurchaseOrderService(PurchaseOrderService purchaseOrderService) {
        this.purchaseOrderService = purchaseOrderService;
    }

    public void setReceivingDao(ReceivingDao receivingDao) {
        this.receivingDao = receivingDao;
    }

    public void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setWorkflowDocumentService(WorkflowDocumentService workflowDocumentService) {
        this.workflowDocumentService = workflowDocumentService;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    public void setPurapService(PurapService purapService) {
        this.purapService = purapService;
    }

    public void setNoteService(NoteService noteService) {
        this.noteService = noteService;
    }

    public void setFinancialSystemDocumentService(FinancialSystemDocumentService financialSystemDocumentService) {
        this.financialSystemDocumentService = financialSystemDocumentService;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }
}

