/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.util;

import java.beans.Beans;
import java.util.ArrayList;

/**
 * This class is setup to allow an argument to a created ArrayList (it could be a possible extension to the other
 * ArrayList
 */
public class PurApArrayList extends ArrayList {

    private final Class listObjectType;
    private final Class[] argumentClasses;
    private final Object[] arguments;

    /**
     * @param listObjectType the class
     */
    public PurApArrayList(Class listObjectType) {
        this(listObjectType, null, null);
    }

    /**
     * @param listObjectType  the object type
     * @param methodClasses   classes
     * @param methodArguments arguments
     */
    public PurApArrayList(Class listObjectType, Class[] methodClasses, Object[] methodArguments) {
        super();

        Class[] assignArgumentClasses = null;
        Object[] assignArguments = null;

        if (listObjectType == null) {
            throw new RuntimeException("class type for list is required.");
        }

        // attempt to get an instance of the class to check it has a visible default constructor
        if (methodClasses == null && methodArguments == null) {
            try {
                listObjectType.newInstance();
            } catch (InstantiationException | IllegalAccessException e) {
                throw new RuntimeException("unable to get instance of class" + listObjectType.getName());
            }
        } else {
            try {
                listObjectType.getConstructor(methodClasses).newInstance(methodArguments);
            } catch (ReflectiveOperationException | SecurityException | IllegalArgumentException e) {
                throw new RuntimeException("unable to get instance of class" + listObjectType.getName());
            }
            assignArgumentClasses = methodClasses;
            assignArguments = methodArguments;
        }

        this.listObjectType = listObjectType;
        this.argumentClasses = assignArgumentClasses;
        this.arguments = assignArguments;
    }

    public void add(int index, Object element) {
        checkType(element);
        super.add(index, element);
    }

    public boolean add(Object o) {
        checkType(o);
        return super.add(o);
    }

    public Object get(int index) {
        growArray(index);
        return super.get(index);
    }

    public Object set(int index, Object element) {
        growArray(index);
        return super.set(index, element);
    }

    /**
     * Adds new instances of type listObjectType to the arrayList until the size of the list is greater than the
     * index required.
     *
     * @param index the index to grow to
     */
    private void growArray(int index) {
        if (index < 0) {
            throw new IndexOutOfBoundsException("Index must be positive.");
        }

        while (size() <= index) {
            try {
                if (this.arguments == null && this.argumentClasses == null) {
                    super.add(listObjectType.newInstance());
                } else {
                    super.add(listObjectType.getConstructor(argumentClasses).newInstance(arguments));
                }
            } catch (ReflectiveOperationException | IllegalArgumentException | SecurityException e) {
                throw new RuntimeException("Cannot get new instance of class " + listObjectType.getName());
            }
        }
    }

    /**
     * Checks the type of an element matches the underlying list type.
     */
    private void checkType(Object element) {
        if (element != null) {
            if (!Beans.isInstanceOf(element, listObjectType)) {
                throw new RuntimeException("element is not of correct type.");
            }
        }
    }

    public Class getListObjectType() {
        return listObjectType;
    }

}
