/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/*
 * Created on Mar 9, 2005
 *
 */
package org.kuali.kfs.module.purap.businessobject;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlSchemaType;
import jakarta.xml.bind.annotation.XmlTransient;
import jakarta.xml.bind.annotation.XmlType;
import jakarta.xml.bind.annotation.adapters.CollapsedStringAdapter;
import jakarta.xml.bind.annotation.adapters.XmlJavaTypeAdapter;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.kuali.kfs.core.api.impex.xml.XmlConstants;
import org.kuali.kfs.module.purap.businessobject.jaxb.Header;
import org.kuali.kfs.module.purap.businessobject.jaxb.Request;
import org.kuali.kfs.module.purap.service.ElectronicInvoiceMappingService;
import org.kuali.kfs.module.purap.util.ElectronicInvoiceUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * <p>Java class for anonymous complex type.
 *
 * <p>The following schema fragment specifies the expected content contained within this class.
 *
 * <pre>
 * &lt;complexType&gt;
 *   &lt;complexContent&gt;
 *     &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *       &lt;sequence&gt;
 *         &lt;element name="Header"&gt;
 *           &lt;complexType&gt;
 *             &lt;complexContent&gt;
 *               &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                 &lt;sequence&gt;
 *                   &lt;element name="From"&gt;
 *                     &lt;complexType&gt;
 *                       &lt;complexContent&gt;
 *                         &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                           &lt;sequence&gt;
 *                             &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Credential"/&gt;
 *                           &lt;/sequence&gt;
 *                         &lt;/restriction&gt;
 *                       &lt;/complexContent&gt;
 *                     &lt;/complexType&gt;
 *                   &lt;/element&gt;
 *                   &lt;element name="To"&gt;
 *                     &lt;complexType&gt;
 *                       &lt;complexContent&gt;
 *                         &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                           &lt;sequence&gt;
 *                             &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Credential"/&gt;
 *                           &lt;/sequence&gt;
 *                         &lt;/restriction&gt;
 *                       &lt;/complexContent&gt;
 *                     &lt;/complexType&gt;
 *                   &lt;/element&gt;
 *                   &lt;element name="Sender"&gt;
 *                     &lt;complexType&gt;
 *                       &lt;complexContent&gt;
 *                         &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                           &lt;sequence&gt;
 *                             &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Credential"/&gt;
 *                             &lt;element name="UserAgent" type="{http://www.kuali.org/kfs/sys/types}zeroToFiftyFiveCharType"/&gt;
 *                           &lt;/sequence&gt;
 *                         &lt;/restriction&gt;
 *                       &lt;/complexContent&gt;
 *                     &lt;/complexType&gt;
 *                   &lt;/element&gt;
 *                 &lt;/sequence&gt;
 *               &lt;/restriction&gt;
 *             &lt;/complexContent&gt;
 *           &lt;/complexType&gt;
 *         &lt;/element&gt;
 *         &lt;element name="Request"&gt;
 *           &lt;complexType&gt;
 *             &lt;complexContent&gt;
 *               &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                 &lt;sequence&gt;
 *                   &lt;element name="InvoiceDetailRequest"&gt;
 *                     &lt;complexType&gt;
 *                       &lt;complexContent&gt;
 *                         &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                           &lt;sequence&gt;
 *                             &lt;element name="InvoiceDetailRequestHeader"&gt;
 *                               &lt;complexType&gt;
 *                                 &lt;complexContent&gt;
 *                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                     &lt;sequence&gt;
 *                                       &lt;element name="InvoiceDetailHeaderIndicator"&gt;
 *                                         &lt;complexType&gt;
 *                                           &lt;simpleContent&gt;
 *                                             &lt;extension base="&lt;http://www.kuali.org/kfs/purap/types&gt;emptyType"&gt;
 *                                               &lt;attribute name="isHeaderInvoice"&gt;
 *                                                 &lt;simpleType&gt;
 *                                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                                     &lt;enumeration value="yes"/&gt;
 *                                                   &lt;/restriction&gt;
 *                                                 &lt;/simpleType&gt;
 *                                               &lt;/attribute&gt;
 *                                               &lt;attribute name="isVatRecoverable"&gt;
 *                                                 &lt;simpleType&gt;
 *                                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                                     &lt;enumeration value="yes"/&gt;
 *                                                   &lt;/restriction&gt;
 *                                                 &lt;/simpleType&gt;
 *                                               &lt;/attribute&gt;
 *                                             &lt;/extension&gt;
 *                                           &lt;/simpleContent&gt;
 *                                         &lt;/complexType&gt;
 *                                       &lt;/element&gt;
 *                                       &lt;element name="InvoiceDetailLineIndicator"&gt;
 *                                         &lt;complexType&gt;
 *                                           &lt;simpleContent&gt;
 *                                             &lt;extension base="&lt;http://www.kuali.org/kfs/purap/types&gt;emptyType"&gt;
 *                                               &lt;attribute name="isTaxInLine"&gt;
 *                                                 &lt;simpleType&gt;
 *                                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                                     &lt;enumeration value="yes"/&gt;
 *                                                   &lt;/restriction&gt;
 *                                                 &lt;/simpleType&gt;
 *                                               &lt;/attribute&gt;
 *                                               &lt;attribute name="isSpecialHandlingInLine"&gt;
 *                                                 &lt;simpleType&gt;
 *                                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                                     &lt;enumeration value="yes"/&gt;
 *                                                   &lt;/restriction&gt;
 *                                                 &lt;/simpleType&gt;
 *                                               &lt;/attribute&gt;
 *                                               &lt;attribute name="isShippingInLine"&gt;
 *                                                 &lt;simpleType&gt;
 *                                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                                     &lt;enumeration value="yes"/&gt;
 *                                                   &lt;/restriction&gt;
 *                                                 &lt;/simpleType&gt;
 *                                               &lt;/attribute&gt;
 *                                               &lt;attribute name="isDiscountInLine"&gt;
 *                                                 &lt;simpleType&gt;
 *                                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                                     &lt;enumeration value="yes"/&gt;
 *                                                   &lt;/restriction&gt;
 *                                                 &lt;/simpleType&gt;
 *                                               &lt;/attribute&gt;
 *                                               &lt;attribute name="isAccountingInLine"&gt;
 *                                                 &lt;simpleType&gt;
 *                                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                                     &lt;enumeration value="yes"/&gt;
 *                                                   &lt;/restriction&gt;
 *                                                 &lt;/simpleType&gt;
 *                                               &lt;/attribute&gt;
 *                                             &lt;/extension&gt;
 *                                           &lt;/simpleContent&gt;
 *                                         &lt;/complexType&gt;
 *                                       &lt;/element&gt;
 *                                       &lt;element name="InvoicePartner" maxOccurs="unbounded" minOccurs="0"&gt;
 *                                         &lt;complexType&gt;
 *                                           &lt;complexContent&gt;
 *                                             &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                               &lt;sequence&gt;
 *                                                 &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Contact" maxOccurs="unbounded"/&gt;
 *                                                 &lt;element name="IdReference" minOccurs="0"&gt;
 *                                                   &lt;complexType&gt;
 *                                                     &lt;complexContent&gt;
 *                                                       &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                                         &lt;sequence&gt;
 *                                                           &lt;element name="Creator" type="{http://www.kuali.org/kfs/sys/types}zeroToTwentyCharType" minOccurs="0"/&gt;
 *                                                           &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Description" minOccurs="0"/&gt;
 *                                                         &lt;/sequence&gt;
 *                                                         &lt;attribute name="domain" use="required" type="{http://www.kuali.org/kfs/sys/types}zeroToTwentyCharType" /&gt;
 *                                                         &lt;attribute name="identifier" use="required" type="{http://www.kuali.org/kfs/sys/types}zeroToTwentyCharType" /&gt;
 *                                                       &lt;/restriction&gt;
 *                                                     &lt;/complexContent&gt;
 *                                                   &lt;/complexType&gt;
 *                                                 &lt;/element&gt;
 *                                               &lt;/sequence&gt;
 *                                             &lt;/restriction&gt;
 *                                           &lt;/complexContent&gt;
 *                                         &lt;/complexType&gt;
 *                                       &lt;/element&gt;
 *                                       &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}InvoiceDetailShipping" minOccurs="0"/&gt;
 *                                       &lt;choice maxOccurs="unbounded" minOccurs="0"&gt;
 *                                         &lt;element name="InvoiceDetailPaymentTerm"&gt;
 *                                           &lt;complexType&gt;
 *                                             &lt;complexContent&gt;
 *                                               &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                                 &lt;attribute name="payInNumberOfDays" use="required" type="{http://www.w3.org/2001/XMLSchema}unsignedInt" /&gt;
 *                                                 &lt;attribute name="percentageRate" use="required" type="{http://www.w3.org/2001/XMLSchema}decimal" /&gt;
 *                                               &lt;/restriction&gt;
 *                                             &lt;/complexContent&gt;
 *                                           &lt;/complexType&gt;
 *                                         &lt;/element&gt;
 *                                         &lt;element name="PaymentTerm"&gt;
 *                                           &lt;complexType&gt;
 *                                             &lt;complexContent&gt;
 *                                               &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                                 &lt;attribute name="payInNumberOfDays" use="required" type="{http://www.w3.org/2001/XMLSchema}unsignedInt" /&gt;
 *                                               &lt;/restriction&gt;
 *                                             &lt;/complexContent&gt;
 *                                           &lt;/complexType&gt;
 *                                         &lt;/element&gt;
 *                                       &lt;/choice&gt;
 *                                       &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Comments" minOccurs="0"/&gt;
 *                                       &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Extrinsic" maxOccurs="unbounded" minOccurs="0"/&gt;
 *                                     &lt;/sequence&gt;
 *                                     &lt;attribute name="purpose" default="standard"&gt;
 *                                       &lt;simpleType&gt;
 *                                         &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                           &lt;enumeration value="standard"/&gt;
 *                                         &lt;/restriction&gt;
 *                                       &lt;/simpleType&gt;
 *                                     &lt;/attribute&gt;
 *                                     &lt;attribute name="operation" default="new"&gt;
 *                                       &lt;simpleType&gt;
 *                                         &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                           &lt;enumeration value="new"/&gt;
 *                                         &lt;/restriction&gt;
 *                                       &lt;/simpleType&gt;
 *                                     &lt;/attribute&gt;
 *                                     &lt;attribute name="invoiceDate" use="required" type="{http://www.kuali.org/kfs/purap/types}dateStringType" /&gt;
 *                                     &lt;attribute name="invoiceID" use="required" type="{http://www.kuali.org/kfs/purap/types}idType" /&gt;
 *                                     &lt;attribute name="isInformationOnly"&gt;
 *                                       &lt;simpleType&gt;
 *                                         &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                                           &lt;enumeration value="yes"/&gt;
 *                                         &lt;/restriction&gt;
 *                                       &lt;/simpleType&gt;
 *                                     &lt;/attribute&gt;
 *                                   &lt;/restriction&gt;
 *                                 &lt;/complexContent&gt;
 *                               &lt;/complexType&gt;
 *                             &lt;/element&gt;
 *                             &lt;element name="InvoiceDetailOrder" maxOccurs="unbounded"&gt;
 *                               &lt;complexType&gt;
 *                                 &lt;complexContent&gt;
 *                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                     &lt;sequence&gt;
 *                                       &lt;element name="InvoiceDetailOrderInfo"&gt;
 *                                         &lt;complexType&gt;
 *                                           &lt;complexContent&gt;
 *                                             &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                               &lt;sequence&gt;
 *                                                 &lt;choice&gt;
 *                                                   &lt;sequence&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}OrderReference"/&gt;
 *                                                     &lt;choice minOccurs="0"&gt;
 *                                                       &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}MasterAgreementReference"/&gt;
 *                                                       &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}MasterAgreementIDInfo"/&gt;
 *                                                     &lt;/choice&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}SupplierOrderInfo" minOccurs="0"/&gt;
 *                                                   &lt;/sequence&gt;
 *                                                   &lt;sequence&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}MasterAgreementReference"/&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}OrderIDInfo" minOccurs="0"/&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}SupplierOrderInfo" minOccurs="0"/&gt;
 *                                                   &lt;/sequence&gt;
 *                                                   &lt;sequence&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}MasterAgreementIDInfo"/&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}OrderIDInfo" minOccurs="0"/&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}SupplierOrderInfo" minOccurs="0"/&gt;
 *                                                   &lt;/sequence&gt;
 *                                                   &lt;sequence&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}OrderIDInfo"/&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}SupplierOrderInfo" minOccurs="0"/&gt;
 *                                                   &lt;/sequence&gt;
 *                                                   &lt;sequence&gt;
 *                                                     &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}SupplierOrderInfo"/&gt;
 *                                                   &lt;/sequence&gt;
 *                                                 &lt;/choice&gt;
 *                                               &lt;/sequence&gt;
 *                                             &lt;/restriction&gt;
 *                                           &lt;/complexContent&gt;
 *                                         &lt;/complexType&gt;
 *                                       &lt;/element&gt;
 *                                       &lt;choice&gt;
 *                                         &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}InvoiceDetailItem" maxOccurs="unbounded"/&gt;
 *                                         &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}InvoiceDetailServiceItem" maxOccurs="unbounded"/&gt;
 *                                       &lt;/choice&gt;
 *                                     &lt;/sequence&gt;
 *                                   &lt;/restriction&gt;
 *                                 &lt;/complexContent&gt;
 *                               &lt;/complexType&gt;
 *                             &lt;/element&gt;
 *                             &lt;element name="InvoiceDetailSummary"&gt;
 *                               &lt;complexType&gt;
 *                                 &lt;complexContent&gt;
 *                                   &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                     &lt;sequence&gt;
 *                                       &lt;element name="SubtotalAmount"&gt;
 *                                         &lt;complexType&gt;
 *                                           &lt;complexContent&gt;
 *                                             &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                               &lt;sequence&gt;
 *                                                 &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Money"/&gt;
 *                                               &lt;/sequence&gt;
 *                                             &lt;/restriction&gt;
 *                                           &lt;/complexContent&gt;
 *                                         &lt;/complexType&gt;
 *                                       &lt;/element&gt;
 *                                       &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Tax"/&gt;
 *                                       &lt;element name="SpecialHandlingAmount" minOccurs="0"&gt;
 *                                         &lt;complexType&gt;
 *                                           &lt;complexContent&gt;
 *                                             &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                               &lt;sequence&gt;
 *                                                 &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Money"/&gt;
 *                                                 &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Description" minOccurs="0"/&gt;
 *                                               &lt;/sequence&gt;
 *                                             &lt;/restriction&gt;
 *                                           &lt;/complexContent&gt;
 *                                         &lt;/complexType&gt;
 *                                       &lt;/element&gt;
 *                                       &lt;element name="ShippingAmount" minOccurs="0"&gt;
 *                                         &lt;complexType&gt;
 *                                           &lt;complexContent&gt;
 *                                             &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                               &lt;sequence&gt;
 *                                                 &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Money"/&gt;
 *                                               &lt;/sequence&gt;
 *                                             &lt;/restriction&gt;
 *                                           &lt;/complexContent&gt;
 *                                         &lt;/complexType&gt;
 *                                       &lt;/element&gt;
 *                                       &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}GrossAmount" minOccurs="0"/&gt;
 *                                       &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}InvoiceDetailDiscount" minOccurs="0"/&gt;
 *                                       &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}NetAmount"/&gt;
 *                                       &lt;element name="DepositAmount" minOccurs="0"&gt;
 *                                         &lt;complexType&gt;
 *                                           &lt;complexContent&gt;
 *                                             &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                               &lt;sequence&gt;
 *                                                 &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Money"/&gt;
 *                                               &lt;/sequence&gt;
 *                                             &lt;/restriction&gt;
 *                                           &lt;/complexContent&gt;
 *                                         &lt;/complexType&gt;
 *                                       &lt;/element&gt;
 *                                       &lt;element name="DueAmount" minOccurs="0"&gt;
 *                                         &lt;complexType&gt;
 *                                           &lt;complexContent&gt;
 *                                             &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType"&gt;
 *                                               &lt;sequence&gt;
 *                                                 &lt;element ref="{http://www.kuali.org/kfs/purap/electronicInvoice}Money"/&gt;
 *                                               &lt;/sequence&gt;
 *                                             &lt;/restriction&gt;
 *                                           &lt;/complexContent&gt;
 *                                         &lt;/complexType&gt;
 *                                       &lt;/element&gt;
 *                                     &lt;/sequence&gt;
 *                                   &lt;/restriction&gt;
 *                                 &lt;/complexContent&gt;
 *                               &lt;/complexType&gt;
 *                             &lt;/element&gt;
 *                           &lt;/sequence&gt;
 *                         &lt;/restriction&gt;
 *                       &lt;/complexContent&gt;
 *                     &lt;/complexType&gt;
 *                   &lt;/element&gt;
 *                 &lt;/sequence&gt;
 *                 &lt;attribute name="deploymentMode" default="production"&gt;
 *                   &lt;simpleType&gt;
 *                     &lt;restriction base="{http://www.w3.org/2001/XMLSchema}string"&gt;
 *                       &lt;enumeration value="production"/&gt;
 *                     &lt;/restriction&gt;
 *                   &lt;/simpleType&gt;
 *                 &lt;/attribute&gt;
 *               &lt;/restriction&gt;
 *             &lt;/complexContent&gt;
 *           &lt;/complexType&gt;
 *         &lt;/element&gt;
 *       &lt;/sequence&gt;
 *       &lt;attribute name="version" type="{http://www.w3.org/2001/XMLSchema}string" default="1.2.019" /&gt;
 *       &lt;attribute ref="{http://www.w3.org/XML/1998/namespace}lang"/&gt;
 *       &lt;attribute name="timestamp" use="required" type="{http://www.w3.org/2001/XMLSchema}string" /&gt;
 *       &lt;attribute name="payloadID" use="required" type="{http://www.w3.org/2001/XMLSchema}string" /&gt;
 *     &lt;/restriction&gt;
 *   &lt;/complexContent&gt;
 * &lt;/complexType&gt;
 * </pre>
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "", propOrder = {"header", "request"})
@XmlRootElement(name = "cXML", namespace = XmlConstants.ELECTRONIC_INV_NAMESPACE)
public class ElectronicInvoice {
    private static final BigDecimal zero = new BigDecimal(0.00);

    public static String INVOICE_AMOUNT_TYPE_CODE_ITEM = "ITEM";
    public static String INVOICE_AMOUNT_TYPE_CODE_TAX = "TAX";
    public static String INVOICE_AMOUNT_TYPE_CODE_SPECIAL_HANDLING = "SPHD";
    public static String INVOICE_AMOUNT_TYPE_CODE_SHIPPING = "SHIP";
    public static String INVOICE_AMOUNT_TYPE_CODE_DISCOUNT = "DISC";
    public static String INVOICE_AMOUNT_TYPE_CODE_DEPOSIT = "DPST";
    public static String INVOICE_AMOUNT_TYPE_CODE_DUE = "DUE";
    public static String INVOICE_AMOUNT_TYPE_CODE_EXMT = "EXMT";

    public static boolean FILE_REJECTED = true;
    public static boolean FILE_NOT_REJECTED;
    public static boolean FILE_DOES_CONTAIN_REJECTS = true;
    public static boolean FILE_DOES_NOT_CONTAIN_REJECTS;

    @XmlElement(name = "Header", namespace = XmlConstants.ELECTRONIC_INV_NAMESPACE, required = true)
    private Header header;

    @XmlElement(name = "Request", namespace = XmlConstants.ELECTRONIC_INV_NAMESPACE, required = true)
    private Request request;

    @XmlAttribute(name = "lang", namespace = XmlConstants.LANG_NAMESPACE, required = true)
    @XmlJavaTypeAdapter(CollapsedStringAdapter.class)
    @XmlSchemaType(name = "language")
    private String lang;

    @XmlTransient
    private String customerNumber;
    @XmlTransient
    private String fileName;
    @XmlTransient
    private String dunsNumber;
    @XmlTransient
    private Integer vendorHeaderID;
    @XmlTransient
    private Integer vendorDetailID;
    @XmlTransient
    private String vendorName;
    @XmlTransient
    private boolean fileRejected = FILE_NOT_REJECTED;
    @XmlTransient
    private boolean containsRejects = FILE_DOES_NOT_CONTAIN_REJECTS;
    @XmlTransient
    private List fileRejectReasons = new ArrayList();

    @XmlTransient
    private ElectronicInvoiceDetailRequestHeader invoiceDetailRequestHeader;
    @XmlTransient
    private List<ElectronicInvoiceOrder> invoiceDetailOrders;
    @XmlTransient
    private ElectronicInvoiceDetailRequestSummary invoiceDetailRequestSummary;

    @XmlAttribute(name = "version")
    private String version = "1.2.019";

    @XmlAttribute(name = "timestamp", required = true)
    private String timestamp;

    @XmlAttribute(name = "payloadID", required = true)
    private String payloadID;

    @XmlTransient
    private String deploymentMode = "production";

    public ElectronicInvoice() {
    }

    private boolean containsLineLevelAmounts() {
        return getInvoiceDetailRequestHeader().isShippingInLine() || getInvoiceDetailRequestHeader().isSpecialHandlingInLine()
                || getInvoiceDetailRequestHeader().isTaxInLine() || getInvoiceDetailRequestHeader().isDiscountInLine();
    }

    public void addFileRejectReasonToList(final ElectronicInvoiceRejectReason reason) {
        fileRejectReasons.add(reason);
    }

    public ElectronicInvoicePostalAddress getCxmlPostalAddress(
            final ElectronicInvoiceOrder eio, final String roleID,
            final String addressName) {
        if (getInvoiceDetailRequestHeader().isShippingInLine()) {
            return eio.getCxmlPostalAddressByRoleID(roleID, addressName);
        } else {
            return getInvoiceDetailRequestHeader().getCxmlPostalAddressByRoleID(roleID, addressName);
        }
    }

    public ElectronicInvoiceContact getCxmlContact(final ElectronicInvoiceOrder eio, final String roleID) {
        if (getInvoiceDetailRequestHeader().isShippingInLine()) {
            return eio.getCxmlContactByRoleID(roleID);
        } else {
            return getInvoiceDetailRequestHeader().getCxmlContactByRoleID(roleID);
        }
    }

    public String getShippingDateDisplayText(final ElectronicInvoiceOrder eio) {
        final Date date;
        final String dateString;
        if (getInvoiceDetailRequestHeader().isShippingInLine()) {
            date = eio.getInvoiceShippingDate();
            dateString = eio.getInvoiceShippingDateString();
        } else {
            date = getInvoiceDetailRequestHeader().getShippingDate();
            dateString = getInvoiceDetailRequestHeader().getShippingDateString();
        }
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return dateString;
        }
    }

    public String getMasterAgreementIDInfoDateDisplayText(final ElectronicInvoiceOrder eio) {
        final Date date = eio.getMasterAgreementIDInfoDate();
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return eio.getMasterAgreementIDInfoDateString();
        }
    }

    public String getMasterAgreementReferenceDateDisplayText(final ElectronicInvoiceOrder eio) {
        final Date date = eio.getMasterAgreementReferenceDate();
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return eio.getMasterAgreementReferenceDateString();
        }
    }

    public String getOrderIDInfoDateDisplayText(final ElectronicInvoiceOrder eio) {
        final Date date = eio.getOrderIDInfoDate();
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return eio.getOrderIDInfoDateString();
        }
    }

    public String getInvoiceDateDisplayText() {
        final Date date = getInvoiceDetailRequestHeader().getInvoiceDate();
        if (date != null) {
            return ElectronicInvoiceUtils.getDateDisplayText(date);
        } else {
            return getInvoiceDetailRequestHeader().getInvoiceDateString();
        }
    }

    public String getInvoiceShippingDescription(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isShippingInLine()) {
            return eio.getInvoiceShippingDescription();
        } else {
            return getInvoiceDetailRequestSummary().getShippingDescription();
        }
    }

    public String getInvoiceTaxDescription(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isTaxInLine()) {
            return eio.getInvoiceTaxDescription();
        } else {
            return getInvoiceDetailRequestSummary().getTaxDescription();
        }
    }

    public String getInvoiceSpecialHandlingDescription(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isSpecialHandlingInLine()) {
            return eio.getInvoiceSpecialHandlingDescription();

        } else {
            return getInvoiceDetailRequestSummary().getSpecialHandlingAmountDescription();
        }
    }

    public BigDecimal getFileTotalAmountForInLineItems(final String invoiceLineItemType) {
        BigDecimal total = zero;
        for (final ElectronicInvoiceOrder invoiceDetailOrder : invoiceDetailOrders) {
            if (INVOICE_AMOUNT_TYPE_CODE_TAX.equalsIgnoreCase(invoiceLineItemType)) {
                total = total.add(invoiceDetailOrder.getInvoiceTaxAmount());
            } else if (INVOICE_AMOUNT_TYPE_CODE_SPECIAL_HANDLING.equalsIgnoreCase(invoiceLineItemType)) {
                total = total.add(invoiceDetailOrder.getInvoiceSpecialHandlingAmount());
            } else if (INVOICE_AMOUNT_TYPE_CODE_SHIPPING.equalsIgnoreCase(invoiceLineItemType)) {
                total = total.add(invoiceDetailOrder.getInvoiceShippingAmount());
            } else if (INVOICE_AMOUNT_TYPE_CODE_DISCOUNT.equalsIgnoreCase(invoiceLineItemType)) {
                total = total.add(invoiceDetailOrder.getInvoiceDiscountAmount());
            }
        }
        return total;
    }

    public BigDecimal getInvoiceSubTotalAmount(final ElectronicInvoiceOrder eio) {
        if (containsLineLevelAmounts()) {
            return eio.getInvoiceSubTotalAmount();
        } else {
            return getInvoiceDetailRequestSummary().getInvoiceSubTotalAmount();
        }
    }

    public BigDecimal getInvoiceTaxAmount(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isTaxInLine()) {
            return eio.getInvoiceTaxAmount();
        } else {
            return getInvoiceDetailRequestSummary().getInvoiceTaxAmount();
        }
    }

    public BigDecimal getInvoiceSpecialHandlingAmount(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isSpecialHandlingInLine()) {
            return eio.getInvoiceSpecialHandlingAmount();
        } else {
            return getInvoiceDetailRequestSummary().getInvoiceSpecialHandlingAmount();
        }
    }

    public BigDecimal getInvoiceShippingAmount(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isShippingInLine()) {
            return eio.getInvoiceShippingAmount();
        } else {
            return getInvoiceDetailRequestSummary().getInvoiceShippingAmount();
        }
    }

    public BigDecimal getInvoiceGrossAmount(final ElectronicInvoiceOrder eio) {
        if (containsLineLevelAmounts()) {
            return eio.getInvoiceGrossAmount();
        } else {
            return getInvoiceDetailRequestSummary().getInvoiceGrossAmount();
        }
    }

    public BigDecimal getInvoiceDiscountAmount(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isDiscountInLine()) {
            return eio.getInvoiceDiscountAmount();
        } else {
            return getInvoiceDetailRequestSummary().getInvoiceDiscountAmount();
        }
    }

    public BigDecimal getInvoiceNetAmount(final ElectronicInvoiceOrder eio) {
        if (containsLineLevelAmounts()) {
            return eio.getInvoiceNetAmount();
        } else {
            return getInvoiceDetailRequestSummary().getInvoiceNetAmount();
        }
    }

    public BigDecimal getInvoiceDepositAmount() {
        return getInvoiceDetailRequestSummary().getInvoiceDepositAmount();
    }

    public BigDecimal getInvoiceDueAmount() {
        return getInvoiceDetailRequestSummary().getInvoiceDueAmount();
    }

    public String getCodeOfLineItemThatContainsInvalidCurrency(final String invoiceLineItemType) {
        for (final ElectronicInvoiceOrder invoiceDetailOrder : invoiceDetailOrders) {
            for (final ElectronicInvoiceItem invoiceItem : invoiceDetailOrder.getInvoiceItems()) {
                if (INVOICE_AMOUNT_TYPE_CODE_TAX.equalsIgnoreCase(invoiceLineItemType)) {
                    if (!isCodeValidCurrency(invoiceItem.getTaxAmountCurrency())) {
                        return invoiceItem.getTaxAmountCurrency();
                    }
                } else if (INVOICE_AMOUNT_TYPE_CODE_SPECIAL_HANDLING.equalsIgnoreCase(invoiceLineItemType)) {
                    if (!isCodeValidCurrency(invoiceItem.getInvoiceLineSpecialHandlingAmountCurrency())) {
                        return invoiceItem.getInvoiceLineSpecialHandlingAmountCurrency();
                    }
                } else if (INVOICE_AMOUNT_TYPE_CODE_SHIPPING.equalsIgnoreCase(invoiceLineItemType)) {
                    if (!isCodeValidCurrency(invoiceItem.getInvoiceLineShippingAmountCurrency())) {
                        return invoiceItem.getInvoiceLineShippingAmountCurrency();
                    }
                } else if (INVOICE_AMOUNT_TYPE_CODE_DISCOUNT.equalsIgnoreCase(invoiceLineItemType)) {
                    if (!isCodeValidCurrency(invoiceItem.getInvoiceLineDiscountAmountCurrency())) {
                        return invoiceItem.getInvoiceLineDiscountAmountCurrency();
                    }
                }
            }
        }
        return null;
    }

    /**
     * This method contains the mapping check for valid Currency Code(s)
     */
    public String checkCodeForValidCurrency(final String code) {
        if (!isCodeValidCurrency(code)) {
            return code;
        } else {
            return null;
        }
    }

    /**
     * This method contains the mapping check for valid Currency Code(s)
     */
    public boolean isCodeValidCurrency(final String code) {
        if (code != null) {
            for (int i = 0; i < ElectronicInvoiceMappingService.CXML_VALID_CURRENCY_CODES.length; i++) {
                final String validCode = ElectronicInvoiceMappingService.CXML_VALID_CURRENCY_CODES[i];
                if (code.equalsIgnoreCase(validCode)) {
                    return true;
                }
            }
        }
        return false;
    }

    public String getInvoiceSubTotalCurrencyIfNotValid(final ElectronicInvoiceOrder eio) {
        if (containsLineLevelAmounts()) {
            for (final ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                final String currentCode = checkCodeForValidCurrency(eii.getSubTotalAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return checkCodeForValidCurrency(getInvoiceDetailRequestSummary().getSubTotalAmountCurrency());
        }
    }

    public String getInvoiceTaxCurrencyIfNotValid(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isTaxInLine()) {
            for (final ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                final String currentCode = checkCodeForValidCurrency(eii.getTaxAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return checkCodeForValidCurrency(getInvoiceDetailRequestSummary().getTaxAmountCurrency());
        }
    }

    public String getInvoiceSpecialHandlingCurrencyIfNotValid(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isSpecialHandlingInLine()) {
            for (final ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                final String currentCode = checkCodeForValidCurrency(eii.getInvoiceLineSpecialHandlingAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return checkCodeForValidCurrency(getInvoiceDetailRequestSummary().getSpecialHandlingAmountCurrency());
        }
    }

    public String getInvoiceShippingCurrencyIfNotValid(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isShippingInLine()) {
            for (final ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                final String currentCode = checkCodeForValidCurrency(eii.getInvoiceLineShippingAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return checkCodeForValidCurrency(getInvoiceDetailRequestSummary().getShippingAmountCurrency());
        }
    }

    public String getInvoiceGrossCurrencyIfNotValid(final ElectronicInvoiceOrder eio) {
        if (containsLineLevelAmounts()) {
            for (final ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                final String currentCode = checkCodeForValidCurrency(eii.getInvoiceLineGrossAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return checkCodeForValidCurrency(getInvoiceDetailRequestSummary().getGrossAmountCurrency());
        }
    }

    public String getInvoiceDiscountCurrencyIfNotValid(final ElectronicInvoiceOrder eio) {
        if (getInvoiceDetailRequestHeader().isDiscountInLine()) {
            for (final ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                final String currentCode = checkCodeForValidCurrency(eii.getInvoiceLineDiscountAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return checkCodeForValidCurrency(getInvoiceDetailRequestSummary().getDiscountAmountCurrency());
        }
    }

    public String getInvoiceNetCurrencyIfNotValid(final ElectronicInvoiceOrder eio) {
        if (containsLineLevelAmounts()) {
            for (final ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
                final String currentCode = checkCodeForValidCurrency(eii.getInvoiceLineNetAmountCurrency());
                if (currentCode != null) {
                    return currentCode;
                }
            }
            return null;
        } else {
            return checkCodeForValidCurrency(getInvoiceDetailRequestSummary().getNetAmountCurrency());
        }
    }

    public String getInvoiceDepositCurrencyIfNotValid() {
        return checkCodeForValidCurrency(getInvoiceDetailRequestSummary().getDepositAmountCurrency());
    }

    public String getInvoiceDueCurrencyIfNotValid() {
        return checkCodeForValidCurrency(getInvoiceDetailRequestSummary().getDueAmountCurrency());
    }

    public boolean isContainsRejects() {
        return containsRejects;
    }

    public void setContainsRejects(final boolean containsRejects) {
        this.containsRejects = containsRejects;
    }

    public String getCustomerNumber() {
        return customerNumber;
    }

    public void setCustomerNumber(final String customerNumber) {
        this.customerNumber = customerNumber;
    }

    public Header getHeader() {
        return header;
    }

    public void setHeader(final Header header) {
        this.header = header;
    }

    public String getDunsNumber() {
        return dunsNumber;
    }

    public void setDunsNumber(final String dunsNumber) {
        this.dunsNumber = dunsNumber;
    }

    public String getFileName() {
        return fileName;
    }

    public void setFileName(final String fileName) {
        this.fileName = fileName;
    }

    public boolean isFileRejected() {
        return fileRejected;
    }

    public void setFileRejected(final boolean fileRejected) {
        this.fileRejected = fileRejected;
    }

    public List<ElectronicInvoiceRejectReason> getFileRejectReasons() {
        return fileRejectReasons;
    }

    public void setFileRejectReasons(final List<ElectronicInvoiceRejectReason> fileRejectReasons) {
        this.fileRejectReasons = fileRejectReasons;
    }

    public List<ElectronicInvoiceOrder> getInvoiceDetailOrders() {
        if (invoiceDetailOrders == null) {
            invoiceDetailOrders = request.getInvoiceDetailRequest().getElectronicInvoiceOrders();
        }
        return invoiceDetailOrders;
    }

    public void setInvoiceDetailOrders(final List<ElectronicInvoiceOrder> invoiceDetailOrders) {
        this.invoiceDetailOrders = invoiceDetailOrders;
    }

    public ElectronicInvoiceDetailRequestHeader getInvoiceDetailRequestHeader() {
        if (invoiceDetailRequestHeader == null) {
            invoiceDetailRequestHeader = request.getInvoiceDetailRequest()
                    .getInvoiceDetailRequestHeader()
                    .toElectronicInvoiceDetailRequestHeader();
        }
        return invoiceDetailRequestHeader;
    }

    public ElectronicInvoiceDetailRequestSummary getInvoiceDetailRequestSummary() {
        if (invoiceDetailRequestSummary == null) {
            invoiceDetailRequestSummary = request.getInvoiceDetailRequest()
                    .getInvoiceDetailSummary()
                    .toElectronicInvoiceDetailRequestSummary();
        }
        return invoiceDetailRequestSummary;
    }

    public void setInvoiceDetailRequestSummary(final ElectronicInvoiceDetailRequestSummary invoiceDetailRequestSummary) {
        this.invoiceDetailRequestSummary = invoiceDetailRequestSummary;
    }

    public Integer getVendorDetailID() {
        return vendorDetailID;
    }

    public void setVendorDetailID(final Integer vendorDetailID) {
        this.vendorDetailID = vendorDetailID;
    }

    public Integer getVendorHeaderID() {
        return vendorHeaderID;
    }

    public void setVendorHeaderID(final Integer vendorHeaderID) {
        this.vendorHeaderID = vendorHeaderID;
    }

    public String getVendorName() {
        return vendorName;
    }

    public void setVendorName(final String vendorName) {
        this.vendorName = vendorName;
    }

    public String getPayloadID() {
        return payloadID;
    }

    public void setPayloadID(final String payloadID) {
        this.payloadID = payloadID;
    }

    public String getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(final String timestamp) {
        this.timestamp = timestamp;
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(final String version) {
        this.version = version;
    }

    public String getLang() {
        return lang;
    }

    public void setLang(final String lang) {
        this.lang = lang;
    }

    public void addInvoiceDetailOrder(final ElectronicInvoiceOrder electronicInvoiceOrder) {
        getInvoiceDetailOrders().add(electronicInvoiceOrder);
    }

    public ElectronicInvoiceOrder[] getInvoiceDetailOrdersAsArray() {
        if (getInvoiceDetailOrders().size() > 0) {
            final ElectronicInvoiceOrder[] tempOrders = new ElectronicInvoiceOrder[getInvoiceDetailOrders().size()];
            getInvoiceDetailOrders().toArray(tempOrders);
            return tempOrders;
        }
        return null;
    }

    public String getDeploymentMode() {
        return deploymentMode;
    }

    public void setDeploymentMode(final String deploymentMode) {
        this.deploymentMode = deploymentMode;
    }

    @Override
    public String toString() {
        final ToStringBuilder toString = new ToStringBuilder(this);
        toString.append("version", version);
        toString.append("timestamp", timestamp);
        toString.append("payloadID", payloadID);
        toString.append("locale", lang);
        toString.append("customerNumber", customerNumber);
        toString.append("fileName", fileName);
        toString.append("deploymentMode", deploymentMode);

        toString.append("dunsNumber", dunsNumber);
        toString.append("vendorHeaderID", vendorHeaderID);
        toString.append("vendorDetailID", vendorDetailID);
        toString.append("vendorName", vendorName);
        toString.append("header", header);
        toString.append("invoiceDetailRequestHeader", getInvoiceDetailRequestHeader());
        toString.append("invoiceDetailOrders", getInvoiceDetailOrders());
        toString.append("invoiceDetailRequestSummary", getInvoiceDetailRequestSummary());

        return toString.toString();
    }

}
