/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.document.SessionDocument;
import org.kuali.kfs.krad.util.NoteType;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.purap.PurapParameterConstants;
import org.kuali.kfs.module.purap.businessobject.ElectronicInvoice;
import org.kuali.kfs.module.purap.businessobject.ElectronicInvoiceContact;
import org.kuali.kfs.module.purap.businessobject.ElectronicInvoiceItem;
import org.kuali.kfs.module.purap.businessobject.ElectronicInvoiceLoadSummary;
import org.kuali.kfs.module.purap.businessobject.ElectronicInvoiceOrder;
import org.kuali.kfs.module.purap.businessobject.ElectronicInvoicePostalAddress;
import org.kuali.kfs.module.purap.businessobject.ElectronicInvoiceRejectItem;
import org.kuali.kfs.module.purap.businessobject.ElectronicInvoiceRejectReason;
import org.kuali.kfs.module.purap.businessobject.PurchaseOrderView;
import org.kuali.kfs.module.purap.document.service.PurchaseOrderService;
import org.kuali.kfs.module.purap.service.ElectronicInvoiceMappingService;
import org.kuali.kfs.module.purap.util.ElectronicInvoiceUtils;
import org.kuali.kfs.module.purap.util.PurApRelatedViews;
import org.kuali.kfs.module.purap.util.PurapSearchUtils;
import org.kuali.kfs.sys.KFSConstants.AdHocPaymentIndicator;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.FinancialSystemTransactionalDocumentBase;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.kfs.vnd.businessobject.CampusParameter;
import org.kuali.kfs.vnd.businessobject.VendorDetail;
import org.kuali.kfs.vnd.document.service.VendorService;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

public class ElectronicInvoiceRejectDocument extends FinancialSystemTransactionalDocumentBase implements
        SessionDocument {

    private static final Logger LOG = LogManager.getLogger();
    protected static BigDecimal zero = new BigDecimal(0);

    // NOT NULL FIELDS
    protected Integer purapDocumentIdentifier;
    protected Integer accountsPayablePurchasingDocumentLinkIdentifier;
    protected Integer invoiceLoadSummaryIdentifier;
    protected Timestamp invoiceProcessTimestamp;
    protected Boolean invoiceFileHeaderTypeIndicator = Boolean.FALSE;
    protected Boolean invoiceFileInformationOnlyIndicator = Boolean.FALSE;
    protected Boolean invoiceFileTaxInLineIndicator = Boolean.FALSE;
    protected Boolean invoiceFileSpecialHandlingInLineIndicator = Boolean.FALSE;
    protected Boolean invoiceFileShippingInLineIndicator = Boolean.FALSE;
    protected Boolean invoiceFileDiscountInLineIndicator = Boolean.FALSE;

    protected String invoiceFileName;
    protected String vendorDunsNumber;
    protected Integer vendorHeaderGeneratedIdentifier;
    protected Integer vendorDetailAssignedIdentifier;
    protected String invoiceFileDate;
    protected String invoiceFileNumber;
    protected String invoiceFilePurposeIdentifier;
    protected String invoiceFileOperationIdentifier;
    protected String invoiceFileDeploymentModeValue;
    protected String invoiceOrderReferenceOrderIdentifier;
    protected String invoiceOrderReferenceDocumentReferencePayloadIdentifier;
    protected String invoiceOrderReferenceDocumentReferenceText;
    protected String invoiceOrderMasterAgreementReferenceIdentifier;
    protected String invoiceOrderMasterAgreementReferenceDate;
    protected String invoiceOrderMasterAgreementInformationIdentifier;
    protected String invoiceOrderMasterAgreementInformationDate;
    protected String invoiceOrderPurchaseOrderIdentifier;
    protected String invoiceOrderPurchaseOrderDate;
    protected String invoiceOrderSupplierOrderInformationIdentifier;
    protected String invoiceShipDate;
    protected String invoiceShipToAddressName;
    protected String invoiceShipToAddressType;
    protected String invoiceShipToAddressLine1;
    protected String invoiceShipToAddressLine2;
    protected String invoiceShipToAddressLine3;
    protected String invoiceShipToAddressCityName;
    protected String invoiceShipToAddressStateCode;
    protected String invoiceShipToAddressPostalCode;
    protected String invoiceShipToAddressCountryCode;
    protected String invoiceShipToAddressCountryName;
    protected String invoiceBillToAddressName;
    protected String invoiceBillToAddressType;
    protected String invoiceBillToAddressLine1;
    protected String invoiceBillToAddressLine2;
    protected String invoiceBillToAddressLine3;
    protected String invoiceBillToAddressCityName;
    protected String invoiceBillToAddressStateCode;
    protected String invoiceBillToAddressPostalCode;
    protected String invoiceBillToAddressCountryCode;
    protected String invoiceBillToAddressCountryName;
    protected String invoiceRemitToAddressName;
    protected String invoiceRemitToAddressType;
    protected String invoiceRemitToAddressLine1;
    protected String invoiceRemitToAddressLine2;
    protected String invoiceRemitToAddressLine3;
    protected String invoiceRemitToAddressCityName;
    protected String invoiceRemitToAddressStateCode;
    protected String invoiceRemitToAddressPostalCode;
    protected String invoiceRemitToAddressCountryCode;
    protected String invoiceRemitToAddressCountryName;

    protected Integer paymentRequestIdentifier;

    protected String invoiceCustomerNumber;
    protected String invoicePurchaseOrderNumber;
    protected Integer purchaseOrderIdentifier;
    protected String purchaseOrderDeliveryCampusCode;

    protected String invoiceItemSubTotalCurrencyCode;
    protected String invoiceItemSpecialHandlingCurrencyCode;
    protected String invoiceItemSpecialHandlingDescription;
    protected String invoiceItemShippingCurrencyCode;
    protected String invoiceItemShippingDescription;
    protected String invoiceItemTaxCurrencyCode;
    protected String invoiceItemTaxDescription;
    protected String invoiceItemGrossCurrencyCode;
    protected String invoiceItemDiscountCurrencyCode;
    protected String invoiceItemNetCurrencyCode;

    protected BigDecimal invoiceItemSubTotalAmount;
    protected BigDecimal invoiceItemSpecialHandlingAmount;
    protected BigDecimal invoiceItemShippingAmount;
    protected BigDecimal invoiceItemTaxAmount;
    protected BigDecimal invoiceItemGrossAmount;
    protected BigDecimal invoiceItemDiscountAmount;
    protected BigDecimal invoiceItemNetAmount;

    protected boolean invoiceNumberAcceptIndicator = false;
    protected boolean invoiceResearchIndicator = false;
    protected Timestamp invoiceFileTimeStampForSearch;

    protected Timestamp accountsPayableApprovalTimestamp;

    protected transient PurApRelatedViews relatedViews;
    protected PurchaseOrderDocument currentPurchaseOrderDocument;

    protected VendorDetail vendorDetail;
    protected ElectronicInvoiceLoadSummary invoiceLoadSummary;
    protected List<ElectronicInvoiceRejectItem> invoiceRejectItems = new ArrayList<>();
    protected List<ElectronicInvoiceRejectReason> invoiceRejectReasons = new ArrayList<>();

    protected boolean isDocumentCreationInProgress = false;

    protected String vendorNumber;

    protected CampusParameter purchaseOrderDeliveryCampus;

    public ElectronicInvoiceRejectDocument() {
        super();
    }

    public void setFileLevelData(ElectronicInvoice ei) {
        DateTimeService dateTimeService = SpringContext.getBean(DateTimeService.class);
        this.invoiceProcessTimestamp = dateTimeService.getCurrentTimestamp();
        this.invoiceFileHeaderTypeIndicator = ei.getInvoiceDetailRequestHeader().isHeaderInvoiceIndicator();
        this.invoiceFileInformationOnlyIndicator = ei.getInvoiceDetailRequestHeader().isInformationOnly();
        this.invoiceFileTaxInLineIndicator = ei.getInvoiceDetailRequestHeader().isTaxInLine();
        this.invoiceFileSpecialHandlingInLineIndicator = ei.getInvoiceDetailRequestHeader().isSpecialHandlingInLine();
        this.invoiceFileShippingInLineIndicator = ei.getInvoiceDetailRequestHeader().isShippingInLine();
        this.invoiceFileDiscountInLineIndicator = ei.getInvoiceDetailRequestHeader().isDiscountInLine();

        this.invoiceFileName = ei.getFileName();
        this.vendorDunsNumber = ei.getDunsNumber();
        this.vendorHeaderGeneratedIdentifier = ei.getVendorHeaderID();
        this.vendorDetailAssignedIdentifier = ei.getVendorDetailID();
        this.invoiceFileDate = ei.getInvoiceDateDisplayText();
        this.invoiceFileNumber = ei.getInvoiceDetailRequestHeader().getInvoiceId();
        this.invoiceFilePurposeIdentifier = ei.getInvoiceDetailRequestHeader().getPurpose();
        this.invoiceFileOperationIdentifier = ei.getInvoiceDetailRequestHeader().getOperation();
        this.invoiceFileDeploymentModeValue = ei.getDeploymentMode();
        this.invoiceCustomerNumber = ei.getCustomerNumber();

        for (ElectronicInvoiceRejectReason eirr : ei.getFileRejectReasons()) {
            ElectronicInvoiceRejectReason newReason = new ElectronicInvoiceRejectReason();
            ObjectUtil.buildObject(newReason, eirr);
            this.invoiceRejectReasons.add(newReason);
        }
    }

    public void setInvoiceOrderLevelData(ElectronicInvoice ei, ElectronicInvoiceOrder eio) {
        this.invoiceOrderReferenceOrderIdentifier = eio.getOrderReferenceOrderID();
        this.invoiceOrderReferenceDocumentReferencePayloadIdentifier = eio.getOrderReferenceDocumentRefPayloadID();
        this.invoiceOrderReferenceDocumentReferenceText = eio.getOrderReferenceDocumentRef();
        this.invoiceOrderMasterAgreementReferenceIdentifier = eio.getMasterAgreementReferenceID();
        this.invoiceOrderMasterAgreementReferenceDate = ei.getMasterAgreementReferenceDateDisplayText(eio);
        this.invoiceOrderMasterAgreementInformationIdentifier = eio.getMasterAgreementIDInfoID();
        this.invoiceOrderMasterAgreementInformationDate = ei.getMasterAgreementIDInfoDateDisplayText(eio);
        this.invoiceOrderPurchaseOrderIdentifier = eio.getOrderIDInfoID();
        this.invoiceOrderPurchaseOrderDate = ei.getOrderIDInfoDateDisplayText(eio);
        this.invoiceOrderSupplierOrderInformationIdentifier = eio.getSupplierOrderInfoID();
        this.invoiceShipDate = ei.getShippingDateDisplayText(eio);

        ElectronicInvoiceContact shipToContact = ei.getCxmlContact(eio,
                ElectronicInvoiceMappingService.CXML_ADDRESS_SHIP_TO_ROLE_ID);
        ElectronicInvoicePostalAddress shipToAddress = this.getCxmlPostalAddressByAddressName(shipToContact,
                ElectronicInvoiceMappingService.CXML_ADDRESS_SHIP_TO_NAME);
        if (shipToAddress != null) {
            this.invoiceShipToAddressName = shipToContact.getName();
            this.invoiceShipToAddressType = shipToAddress.getType();
            this.invoiceShipToAddressLine1 = shipToAddress.getLine1();
            this.invoiceShipToAddressLine2 = shipToAddress.getLine2();
            this.invoiceShipToAddressLine3 = shipToAddress.getLine3();
            this.invoiceShipToAddressCityName = shipToAddress.getCityName();
            this.invoiceShipToAddressStateCode = shipToAddress.getStateCode();
            this.invoiceShipToAddressPostalCode = shipToAddress.getPostalCode();
            this.invoiceShipToAddressCountryCode = shipToAddress.getCountryCode();
            this.invoiceShipToAddressCountryName = shipToAddress.getCountryName();
        }

        ElectronicInvoiceContact billToContact = ei.getCxmlContact(eio,
                ElectronicInvoiceMappingService.CXML_ADDRESS_BILL_TO_ROLE_ID);
        ElectronicInvoicePostalAddress billToAddress = this.getCxmlPostalAddressByAddressName(billToContact,
                ElectronicInvoiceMappingService.CXML_ADDRESS_BILL_TO_NAME);
        if (billToAddress != null) {
            this.invoiceBillToAddressName = billToContact.getName();
            this.invoiceBillToAddressType = billToAddress.getType();
            this.invoiceBillToAddressLine1 = billToAddress.getLine1();
            this.invoiceBillToAddressLine2 = billToAddress.getLine2();
            this.invoiceBillToAddressLine3 = billToAddress.getLine3();
            this.invoiceBillToAddressCityName = billToAddress.getCityName();
            this.invoiceBillToAddressStateCode = billToAddress.getStateCode();
            this.invoiceBillToAddressPostalCode = billToAddress.getPostalCode();
            this.invoiceBillToAddressCountryCode = billToAddress.getCountryCode();
            this.invoiceBillToAddressCountryName = billToAddress.getCountryName();
        }

        ElectronicInvoiceContact remitToContact = ei.getCxmlContact(eio,
                ElectronicInvoiceMappingService.CXML_ADDRESS_REMIT_TO_ROLE_ID);
        ElectronicInvoicePostalAddress remitToAddress = this.getCxmlPostalAddressByAddressName(remitToContact,
                ElectronicInvoiceMappingService.CXML_ADDRESS_REMIT_TO_NAME);
        if (remitToAddress != null) {
            this.invoiceRemitToAddressName = remitToContact.getName();
            this.invoiceRemitToAddressType = remitToAddress.getType();
            this.invoiceRemitToAddressLine1 = remitToAddress.getLine1();
            this.invoiceRemitToAddressLine2 = remitToAddress.getLine2();
            this.invoiceRemitToAddressLine3 = remitToAddress.getLine3();
            this.invoiceRemitToAddressCityName = remitToAddress.getCityName();
            this.invoiceRemitToAddressStateCode = remitToAddress.getStateCode();
            this.invoiceRemitToAddressPostalCode = remitToAddress.getPostalCode();
            this.invoiceRemitToAddressCountryCode = remitToAddress.getCountryCode();
            this.invoiceRemitToAddressCountryName = remitToAddress.getCountryName();
        }
        this.invoicePurchaseOrderNumber = eio.getInvoicePurchaseOrderID();
        this.purchaseOrderIdentifier = eio.getPurchaseOrderID();
        this.purchaseOrderDeliveryCampusCode = eio.getPurchaseOrderCampusCode();

        try {
            this.invoiceItemSubTotalAmount = ei.getInvoiceSubTotalAmount(eio);
            this.invoiceItemSubTotalCurrencyCode = ei.getInvoiceSubTotalCurrencyIfNotValid(eio);
        } catch (Exception e) {
            this.invoiceItemSubTotalAmount = null;
            this.invoiceItemSubTotalCurrencyCode = "INVALID Amount";
        }
        try {
            this.invoiceItemSpecialHandlingAmount = ei.getInvoiceSpecialHandlingAmount(eio);
            this.invoiceItemSpecialHandlingCurrencyCode = ei.getInvoiceSpecialHandlingCurrencyIfNotValid(eio);
        } catch (Exception e) {
            this.invoiceItemSpecialHandlingAmount = null;
            this.invoiceItemSpecialHandlingCurrencyCode = "INVALID AMOUNT";
        }
        this.invoiceItemSpecialHandlingDescription = ei.getInvoiceSpecialHandlingDescription(eio);
        try {
            this.invoiceItemShippingAmount = ei.getInvoiceShippingAmount(eio);
            this.invoiceItemShippingCurrencyCode = ei.getInvoiceShippingCurrencyIfNotValid(eio);
        } catch (Exception e) {
            this.invoiceItemShippingAmount = null;
            this.invoiceItemShippingCurrencyCode = "INVALID AMOUNT";
        }
        this.invoiceItemShippingDescription = ei.getInvoiceShippingDescription(eio);
        try {
            this.invoiceItemTaxAmount = ei.getInvoiceTaxAmount(eio);
            this.invoiceItemTaxCurrencyCode = ei.getInvoiceTaxCurrencyIfNotValid(eio);
        } catch (Exception e) {
            this.invoiceItemTaxAmount = null;
            this.invoiceItemTaxCurrencyCode = "INVALID AMOUNT";
        }
        this.invoiceItemTaxDescription = ei.getInvoiceTaxDescription(eio);
        try {
            this.invoiceItemGrossAmount = ei.getInvoiceGrossAmount(eio);
            this.invoiceItemGrossCurrencyCode = ei.getInvoiceGrossCurrencyIfNotValid(eio);
        } catch (Exception e) {
            this.invoiceItemGrossAmount = null;
            this.invoiceItemGrossCurrencyCode = "INVALID AMOUNT";
        }
        try {
            this.invoiceItemDiscountAmount = ei.getInvoiceDiscountAmount(eio);
            this.invoiceItemDiscountCurrencyCode = ei.getInvoiceDiscountCurrencyIfNotValid(eio);
        } catch (Exception e) {
            this.invoiceItemDiscountAmount = null;
            this.invoiceItemDiscountCurrencyCode = "INVALID AMOUNT";
        }
        try {
            this.invoiceItemNetAmount = ei.getInvoiceNetAmount(eio);
            this.invoiceItemNetCurrencyCode = ei.getInvoiceNetCurrencyIfNotValid(eio);
        } catch (Exception e) {
            this.invoiceItemNetAmount = null;
            this.invoiceItemNetCurrencyCode = "INVALID AMOUNT";
        }

        for (ElectronicInvoiceItem eii : eio.getInvoiceItems()) {
            ElectronicInvoiceRejectItem eiri = new ElectronicInvoiceRejectItem(this, eii);
            this.invoiceRejectItems.add(eiri);
        }

        for (ElectronicInvoiceRejectReason eirr : eio.getOrderRejectReasons()) {
            eirr.setElectronicInvoiceRejectDocument(this);
            this.invoiceRejectReasons.add(eirr);
        }
    }

    @Override
    public String getDocumentTitle() {
        if (SpringContext.getBean(ParameterService.class).getParameterValueAsBoolean(
                ElectronicInvoiceRejectDocument.class, PurapParameterConstants.PURAP_OVERRIDE_EIRT_DOC_TITLE)) {
            return getCustomDocumentTitle();
        }

        return this.buildDocumentTitle(super.getDocumentTitle());
    }

    /**
     * @return a custom document title based on the properties of current document.
     */
    protected String getCustomDocumentTitle() {
        String poID = StringUtils.isEmpty(getInvoicePurchaseOrderNumber()) ? "UNKNOWN" :
                getInvoicePurchaseOrderNumber();
        String researchInd = isInvoiceResearchIndicator() ? "Y" : "N";

        // if this method is called when EIRT doc is just created, vendorDetail is not set yet, need to retrieve from
        // DB
        VendorDetail vendorDetail = getVendorDetail();
        Integer headerId = getVendorHeaderGeneratedIdentifier();
        Integer detailId = getVendorDetailAssignedIdentifier();
        if (vendorDetail == null && headerId != null && detailId != null) {
            vendorDetail = SpringContext.getBean(VendorService.class).getVendorDetail(headerId, detailId);
        }
        String vendorName = vendorDetail == null || StringUtils.isEmpty(vendorDetail.getVendorName()) ? "UNKNOWN" :
                vendorDetail.getVendorName();

        //set title to: Vendor: <Vendor's Name> PO: <PO#> <Research Indicator>
        return "Vendor: " + vendorName + " PO: " + poID + " " + researchInd;
    }

    /**
     * Builds document title based on the properties of current document.
     *
     * @param title the default document title
     * @return the combine information of the given title and additional payment indicators
     */
    protected String buildDocumentTitle(String title) {
        if (this.getVendorDetail() == null) {
            return title;
        }

        Integer vendorHeaderGeneratedIdentifier = this.getVendorDetail().getVendorHeaderGeneratedIdentifier();
        if (vendorHeaderGeneratedIdentifier != null) {
            VendorService vendorService = SpringContext.getBean(VendorService.class);

            Object[] indicators = new String[2];

            boolean isEmployeeVendor = vendorService.isVendorInstitutionEmployee(vendorHeaderGeneratedIdentifier);
            indicators[0] = isEmployeeVendor ? AdHocPaymentIndicator.EMPLOYEE_VENDOR : AdHocPaymentIndicator.OTHER;

            boolean isVendorForeign = vendorService.isVendorForeign(vendorHeaderGeneratedIdentifier);
            indicators[1] = isVendorForeign ? AdHocPaymentIndicator.NONRESIDENT_VENDOR : AdHocPaymentIndicator.OTHER;

            for (Object indicator : indicators) {
                if (!AdHocPaymentIndicator.OTHER.equals(indicator)) {
                    String titlePattern = title + " [{0}:{1}]";
                    return MessageFormat.format(titlePattern, indicators);
                }
            }
        }

        return title;
    }

    protected ElectronicInvoicePostalAddress getCxmlPostalAddressByAddressName(ElectronicInvoiceContact contact,
            String addressName) {
        if (contact != null) {
            for (ElectronicInvoicePostalAddress cpa : contact.getPostalAddresses()) {
                if (addressName == null) {
                    return cpa;
                } else {
                    if (addressName.equalsIgnoreCase(cpa.getName())) {
                        return cpa;
                    }
                }
            }
        }
        return null;
    }

    public KualiDecimal getTotalAmount() {
        KualiDecimal returnValue = new KualiDecimal(zero);
        try {
            for (ElectronicInvoiceRejectItem eiri : this.invoiceRejectItems) {
                BigDecimal toAddAmount1 = eiri.getInvoiceItemSubTotalAmount();
                KualiDecimal toAddAmount = KualiDecimal.ZERO;
                if (toAddAmount1 != null) {
                    toAddAmount = new KualiDecimal(toAddAmount1.setScale(KualiDecimal.SCALE,
                            KualiDecimal.ROUND_BEHAVIOR));
                }
                LOG.debug(
                        "getTotalAmount() setting returnValue with arithmetic => '{}' + '{}'",
                        returnValue::doubleValue,
                        toAddAmount::doubleValue
                );
                returnValue = returnValue.add(toAddAmount);
            }
            LOG.debug("getTotalAmount() returning amount {}", returnValue::doubleValue);
            return returnValue;
        } catch (NumberFormatException n) {
            // do nothing this is already rejected
            LOG.error(
                    "getTotalAmount() Error attempting to calculate total amount for invoice with filename {}",
                    this.invoiceFileName
            );
            return new KualiDecimal(zero);
        }
    }

    public void setTotalAmount(KualiDecimal amount) {
        // ignore this
    }

    public KualiDecimal getGrandTotalAmount() {
        KualiDecimal returnValue = new KualiDecimal(zero);
        try {
            for (ElectronicInvoiceRejectItem eiri : this.invoiceRejectItems) {
                KualiDecimal toAddAmount = new KualiDecimal(eiri.getInvoiceItemNetAmount());
                LOG.debug(
                        "getTotalAmount() setting returnValue with arithmetic => '{}' + '{}'",
                        returnValue::doubleValue,
                        toAddAmount::doubleValue
                );
                returnValue = returnValue.add(toAddAmount);
            }
            LOG.debug("getTotalAmount() returning amount {}", returnValue::doubleValue);

            if (this.getInvoiceItemSpecialHandlingAmount() != null
                    && zero.compareTo(this.getInvoiceItemSpecialHandlingAmount()) != 0) {
                returnValue = returnValue.add(new KualiDecimal(this.getInvoiceItemSpecialHandlingAmount()));
            }
            if (this.getInvoiceItemShippingAmount() != null
                    && zero.compareTo(this.getInvoiceItemShippingAmount()) != 0) {
                returnValue = returnValue.add(new KualiDecimal(this.getInvoiceItemShippingAmount()));
            }
            if (this.getInvoiceItemDiscountAmount() != null
                    && zero.compareTo(this.getInvoiceItemDiscountAmount()) != 0) {
                returnValue = returnValue.subtract(new KualiDecimal(this.getInvoiceItemDiscountAmount()));
            }
            LOG.debug("getGrandTotalAmount() returning amount {}", returnValue::doubleValue);
            return returnValue;
        } catch (NumberFormatException n) {
            // do nothing this is already rejected
            LOG.error(
                    "getTotalAmount() Error attempting to calculate total amount for invoice with filename {}",
                    this.invoiceFileName
            );
            return new KualiDecimal(zero);
        }
    }

    public void setGrandTotalAmount(KualiDecimal amount) {
        // ignore this
    }

    public Integer getInvoiceLoadSummaryIdentifier() {
        return invoiceLoadSummaryIdentifier;
    }

    public void setInvoiceLoadSummaryIdentifier(Integer loadSummaryId) {
        this.invoiceLoadSummaryIdentifier = loadSummaryId;
    }

    public PurchaseOrderDocument getCurrentPurchaseOrderDocument() {

        if (StringUtils.isEmpty(getInvoicePurchaseOrderNumber()) ||
            !NumberUtils.isDigits(getInvoicePurchaseOrderNumber())) {
            currentPurchaseOrderDocument = null;
        } else if (currentPurchaseOrderDocument == null) {
            currentPurchaseOrderDocument = SpringContext.getBean(PurchaseOrderService.class)
                    .getCurrentPurchaseOrder(new Integer(getInvoicePurchaseOrderNumber()));
        } else if (!StringUtils.equals(getInvoicePurchaseOrderNumber(),
                currentPurchaseOrderDocument.getPurapDocumentIdentifier().toString())) {
            currentPurchaseOrderDocument = SpringContext.getBean(PurchaseOrderService.class)
                    .getCurrentPurchaseOrder(new Integer(getInvoicePurchaseOrderNumber()));
        }

        return currentPurchaseOrderDocument;
    }

    public VendorDetail getVendorDetail() {
        return vendorDetail;
    }

    public void setVendorDetail(VendorDetail vendorDetail) {
        this.vendorDetail = vendorDetail;
    }

    public ElectronicInvoiceLoadSummary getInvoiceLoadSummary() {
        return invoiceLoadSummary;
    }

    public void setInvoiceLoadSummary(ElectronicInvoiceLoadSummary electronicInvoiceLoadSummary) {
        this.invoiceLoadSummary = electronicInvoiceLoadSummary;
        this.invoiceLoadSummaryIdentifier = electronicInvoiceLoadSummary.getInvoiceLoadSummaryIdentifier();
    }

    public String getInvoiceBillToAddressCityName() {
        return invoiceBillToAddressCityName;
    }

    public void setInvoiceBillToAddressCityName(String billToAddressCityName) {
        this.invoiceBillToAddressCityName = billToAddressCityName;
    }

    public String getInvoiceBillToAddressCountryCode() {
        return invoiceBillToAddressCountryCode;
    }

    public void setInvoiceBillToAddressCountryCode(String billToAddressCountryCode) {
        this.invoiceBillToAddressCountryCode = billToAddressCountryCode;
    }

    public String getInvoiceBillToAddressCountryName() {
        return invoiceBillToAddressCountryName;
    }

    public void setInvoiceBillToAddressCountryName(String billToAddressCountryName) {
        this.invoiceBillToAddressCountryName = billToAddressCountryName;
    }

    public String getInvoiceBillToAddressLine1() {
        return invoiceBillToAddressLine1;
    }

    public void setInvoiceBillToAddressLine1(String billToAddressLine1) {
        this.invoiceBillToAddressLine1 = billToAddressLine1;
    }

    public String getInvoiceBillToAddressLine2() {
        return invoiceBillToAddressLine2;
    }

    public void setInvoiceBillToAddressLine2(String billToAddressLine2) {
        this.invoiceBillToAddressLine2 = billToAddressLine2;
    }

    public String getInvoiceBillToAddressLine3() {
        return invoiceBillToAddressLine3;
    }

    public void setInvoiceBillToAddressLine3(String billToAddressLine3) {
        this.invoiceBillToAddressLine3 = billToAddressLine3;
    }

    public String getInvoiceBillToAddressName() {
        return invoiceBillToAddressName;
    }

    public void setInvoiceBillToAddressName(String billToAddressName) {
        this.invoiceBillToAddressName = billToAddressName;
    }

    public String getInvoiceBillToAddressPostalCode() {
        return invoiceBillToAddressPostalCode;
    }

    public void setInvoiceBillToAddressPostalCode(String billToAddressPostalCode) {
        this.invoiceBillToAddressPostalCode = billToAddressPostalCode;
    }

    public String getInvoiceBillToAddressStateCode() {
        return invoiceBillToAddressStateCode;
    }

    public void setInvoiceBillToAddressStateCode(String billToAddressStateCode) {
        this.invoiceBillToAddressStateCode = billToAddressStateCode;
    }

    public String getInvoiceBillToAddressType() {
        return invoiceBillToAddressType;
    }

    public void setInvoiceBillToAddressType(String billToAddressType) {
        this.invoiceBillToAddressType = billToAddressType;
    }

    public String getInvoiceFileDeploymentModeValue() {
        return invoiceFileDeploymentModeValue;
    }

    public void setInvoiceFileDeploymentModeValue(String deploymentMode) {
        this.invoiceFileDeploymentModeValue = deploymentMode;
    }

    public List<ElectronicInvoiceRejectItem> getInvoiceRejectItems() {
        return invoiceRejectItems;
    }

    public void setInvoiceRejectItems(List<ElectronicInvoiceRejectItem> electronicInvoiceRejectItems) {
        this.invoiceRejectItems = electronicInvoiceRejectItems;
    }

    public ElectronicInvoiceRejectItem getInvoiceRejectItem(int index) {
        while (getInvoiceRejectItems().size() <= index) {
            getInvoiceRejectItems().add(new ElectronicInvoiceRejectItem());
        }
        return getInvoiceRejectItems().get(index);
    }

    public List<ElectronicInvoiceRejectReason> getInvoiceRejectReasons() {
        return invoiceRejectReasons;
    }

    public void setInvoiceRejectReasons(List<ElectronicInvoiceRejectReason> electronicInvoiceRejectReasons) {
        this.invoiceRejectReasons = electronicInvoiceRejectReasons;
    }

    public ElectronicInvoiceRejectReason getInvoiceRejectReason(int index) {
        while (getInvoiceRejectReasons().size() <= index) {
            getInvoiceRejectReasons().add(new ElectronicInvoiceRejectReason());
        }
        return getInvoiceRejectReasons().get(index);
    }

    public Integer getPaymentRequestIdentifier() {
        return paymentRequestIdentifier;
    }

    public void setPaymentRequestIdentifier(Integer paymentRequestIdentifier) {
        this.paymentRequestIdentifier = paymentRequestIdentifier;
    }

    public String getPurchaseOrderDeliveryCampusCode() {
        return purchaseOrderDeliveryCampusCode;
    }

    public void setPurchaseOrderDeliveryCampusCode(String poDeliveryCampusCode) {
        this.purchaseOrderDeliveryCampusCode = poDeliveryCampusCode;
    }

    public Integer getPurchaseOrderIdentifier() {
        return purchaseOrderIdentifier;
    }

    public void setPurchaseOrderIdentifier(Integer purchaseOrderId) {
        this.purchaseOrderIdentifier = purchaseOrderId;
    }

    public Boolean getInvoiceFileDiscountInLineIndicator() {
        return invoiceFileDiscountInLineIndicator;
    }

    public Boolean isInvoiceFileDiscountInLineIndicator() {
        return invoiceFileDiscountInLineIndicator;
    }

    public void setInvoiceFileDiscountInLineIndicator(Boolean fileDiscountInLineIndicator) {
        this.invoiceFileDiscountInLineIndicator = fileDiscountInLineIndicator;
    }

    public Boolean getInvoiceFileHeaderTypeIndicator() {
        return invoiceFileHeaderTypeIndicator;
    }

    public Boolean isInvoiceFileHeaderTypeIndicator() {
        return invoiceFileHeaderTypeIndicator;
    }

    public void setInvoiceFileHeaderTypeIndicator(Boolean fileHeaderTypeIndicator) {
        this.invoiceFileHeaderTypeIndicator = fileHeaderTypeIndicator;
    }

    public Boolean getInvoiceFileInformationOnlyIndicator() {
        return invoiceFileInformationOnlyIndicator;
    }

    public Boolean isInvoiceFileInformationOnlyIndicator() {
        return invoiceFileInformationOnlyIndicator;
    }

    public void setInvoiceFileInformationOnlyIndicator(Boolean fileInformationOnlyIndicator) {
        this.invoiceFileInformationOnlyIndicator = fileInformationOnlyIndicator;
    }

    public String getInvoiceFileOperationIdentifier() {
        return invoiceFileOperationIdentifier;
    }

    public void setInvoiceFileOperationIdentifier(String fileOperationId) {
        this.invoiceFileOperationIdentifier = fileOperationId;
    }

    public String getInvoiceFilePurposeIdentifier() {
        return invoiceFilePurposeIdentifier;
    }

    public void setInvoiceFilePurposeIdentifier(String filePurposeId) {
        this.invoiceFilePurposeIdentifier = filePurposeId;
    }

    public Boolean getInvoiceFileShippingInLineIndicator() {
        return invoiceFileShippingInLineIndicator;
    }

    public Boolean isInvoiceFileShippingInLineIndicator() {
        return invoiceFileShippingInLineIndicator;
    }

    public void setInvoiceFileShippingInLineIndicator(Boolean fileShippingInLineIndicator) {
        this.invoiceFileShippingInLineIndicator = fileShippingInLineIndicator;
    }

    public Boolean getInvoiceFileSpecialHandlingInLineIndicator() {
        return invoiceFileSpecialHandlingInLineIndicator;
    }

    public Boolean isInvoiceFileSpecialHandlingInLineIndicator() {
        return invoiceFileSpecialHandlingInLineIndicator;
    }

    public void setInvoiceFileSpecialHandlingInLineIndicator(Boolean fileSpecHandlingInLineIndicator) {
        this.invoiceFileSpecialHandlingInLineIndicator = fileSpecHandlingInLineIndicator;
    }

    public Boolean getInvoiceFileTaxInLineIndicator() {
        return invoiceFileTaxInLineIndicator;
    }

    public Boolean isInvoiceFileTaxInLineIndicator() {
        return invoiceFileTaxInLineIndicator;
    }

    public void setInvoiceFileTaxInLineIndicator(Boolean fileTaxInLineIndicator) {
        this.invoiceFileTaxInLineIndicator = fileTaxInLineIndicator;
    }

    public Integer getPurapDocumentIdentifier() {
        return purapDocumentIdentifier;
    }

    public void setPurapDocumentIdentifier(Integer id) {
        this.purapDocumentIdentifier = id;
    }

    public Integer getAccountsPayablePurchasingDocumentLinkIdentifier() {
        return accountsPayablePurchasingDocumentLinkIdentifier;
    }

    public void setAccountsPayablePurchasingDocumentLinkIdentifier(
            Integer accountsPayablePurchasingDocumentLinkIdentifier) {
        this.accountsPayablePurchasingDocumentLinkIdentifier = accountsPayablePurchasingDocumentLinkIdentifier;
    }

    public String getInvoiceCustomerNumber() {
        return invoiceCustomerNumber;
    }

    public void setInvoiceCustomerNumber(String invoiceCustomerNumber) {
        this.invoiceCustomerNumber = invoiceCustomerNumber;
    }

    public String getInvoiceFileDate() {
        return invoiceFileDate;
    }

    public void setInvoiceFileDate(String invoiceDate) {
        this.invoiceFileDate = invoiceDate;
    }

    public BigDecimal getInvoiceItemDiscountAmount() {
        if (this.isInvoiceFileDiscountInLineIndicator()) {
            BigDecimal returnValue = zero;
            try {
                for (ElectronicInvoiceRejectItem eiri : this.invoiceRejectItems) {
                    BigDecimal toAddAmount = eiri.getInvoiceItemDiscountAmount();
                    LOG.debug(
                            "getTotalAmount() setting returnValue with arithmetic => '{}' + '{}'",
                            returnValue::doubleValue,
                            toAddAmount::doubleValue
                    );
                    returnValue = returnValue.add(toAddAmount);
                }
                LOG.debug("getTotalAmount() returning amount {}", returnValue::doubleValue);
                return returnValue;
            } catch (NumberFormatException n) {
                // do nothing this is already rejected
                LOG.error(
                        "getTotalAmount() Error attempting to calculate total amount for invoice with filename {}",
                        this.invoiceFileName
                );
                return zero;
            }
        } else {
            return invoiceItemDiscountAmount;
        }
    }

    public void setInvoiceItemDiscountAmount(BigDecimal invoiceDiscountAmount) {
        this.invoiceItemDiscountAmount = invoiceDiscountAmount;
    }

    public String getInvoiceItemDiscountCurrencyCode() {
        return invoiceItemDiscountCurrencyCode;
    }

    public void setInvoiceItemDiscountCurrencyCode(String invoiceDiscountAmountCurrency) {
        this.invoiceItemDiscountCurrencyCode = invoiceDiscountAmountCurrency;
    }

    public String getInvoiceFileName() {
        return invoiceFileName;
    }

    public void setInvoiceFileName(String invoiceFileName) {
        this.invoiceFileName = invoiceFileName;
    }

    public BigDecimal getInvoiceItemGrossAmount() {
        return invoiceItemGrossAmount;
    }

    public void setInvoiceItemGrossAmount(BigDecimal invoiceGrossAmount) {
        this.invoiceItemGrossAmount = invoiceGrossAmount;
    }

    public String getInvoiceItemGrossCurrencyCode() {
        return invoiceItemGrossCurrencyCode;
    }

    public void setInvoiceItemGrossCurrencyCode(String invoiceGrossAmountCurrency) {
        this.invoiceItemGrossCurrencyCode = invoiceGrossAmountCurrency;
    }

    public BigDecimal getInvoiceItemNetAmount() {
        return invoiceItemNetAmount;
    }

    public void setInvoiceItemNetAmount(BigDecimal invoiceNetAmount) {
        this.invoiceItemNetAmount = invoiceNetAmount;
    }

    public boolean isInvoiceNumberAcceptIndicator() {
        return invoiceNumberAcceptIndicator;
    }

    public void setInvoiceNumberAcceptIndicator(boolean invoiceNumberAcceptIndicator) {
        this.invoiceNumberAcceptIndicator = invoiceNumberAcceptIndicator;
    }

    public boolean isInvoiceResearchIndicator() {
        return invoiceResearchIndicator;
    }

    public boolean getInvoiceResearchIndicator() {
        return invoiceResearchIndicator;
    }

    public void setInvoiceResearchIndicator(boolean invoiceResearchIndicator) {
        this.invoiceResearchIndicator = invoiceResearchIndicator;
    }

    public Timestamp getAccountsPayableApprovalTimestamp() {
        return accountsPayableApprovalTimestamp;
    }

    public void setAccountsPayableApprovalTimestamp(Timestamp accountsPayableApprovalTimestamp) {
        this.accountsPayableApprovalTimestamp = accountsPayableApprovalTimestamp;
    }

    public String getInvoiceItemNetCurrencyCode() {
        return invoiceItemNetCurrencyCode;
    }

    public void setInvoiceItemNetCurrencyCode(String invoiceNetAmountCurrency) {
        this.invoiceItemNetCurrencyCode = invoiceNetAmountCurrency;
    }

    public String getInvoiceFileNumber() {
        return invoiceFileNumber;
    }

    public void setInvoiceFileNumber(String invoiceNumber) {
        this.invoiceFileNumber = invoiceNumber;
    }

    public String getInvoiceOrderPurchaseOrderDate() {
        return invoiceOrderPurchaseOrderDate;
    }

    public void setInvoiceOrderPurchaseOrderDate(String invoiceOrderDate) {
        this.invoiceOrderPurchaseOrderDate = invoiceOrderDate;
    }

    public String getInvoiceOrderPurchaseOrderIdentifier() {
        return invoiceOrderPurchaseOrderIdentifier;
    }

    public void setInvoiceOrderPurchaseOrderIdentifier(String invoiceOrderId) {
        this.invoiceOrderPurchaseOrderIdentifier = invoiceOrderId;
    }

    public Timestamp getInvoiceProcessTimestamp() {
        return invoiceProcessTimestamp;
    }

    public void setInvoiceProcessTimestamp(Timestamp invoiceProcessTimestamp) {
        this.invoiceProcessTimestamp = invoiceProcessTimestamp;
    }

    public String getInvoicePurchaseOrderNumber() {
        return invoicePurchaseOrderNumber;
    }

    public void setInvoicePurchaseOrderNumber(String invoicePurchaseOrderId) {
        this.invoicePurchaseOrderNumber = invoicePurchaseOrderId;
    }

    public String getInvoiceShipDate() {
        return invoiceShipDate;
    }

    public void setInvoiceShipDate(String invoiceShipDate) {
        this.invoiceShipDate = invoiceShipDate;
    }

    public BigDecimal getInvoiceItemShippingAmount() {
        if (this.isInvoiceFileShippingInLineIndicator()) {
            BigDecimal returnValue = zero;
            try {
                for (ElectronicInvoiceRejectItem eiri : this.invoiceRejectItems) {
                    BigDecimal toAddAmount = eiri.getInvoiceItemShippingAmount();
                    LOG.debug(
                            "getTotalAmount() setting returnValue with arithmetic => '{}' + '{}'",
                            returnValue::doubleValue,
                            toAddAmount::doubleValue
                    );
                    returnValue = returnValue.add(toAddAmount);
                }
                LOG.debug("getTotalAmount() returning amount {}", returnValue::doubleValue);
                return returnValue;
            } catch (NumberFormatException n) {
                // do nothing this is already rejected
                LOG.error(
                        "getTotalAmount() Error attempting to calculate total amount for invoice with filename {}",
                        this.invoiceFileName
                );
                return zero;
            }
        } else {
            return invoiceItemShippingAmount;
        }
    }

    public void setInvoiceItemShippingAmount(BigDecimal invoiceShippingAmount) {
        this.invoiceItemShippingAmount = invoiceShippingAmount;
    }

    public String getInvoiceItemShippingCurrencyCode() {
        return invoiceItemShippingCurrencyCode;
    }

    public void setInvoiceItemShippingCurrencyCode(String invoiceShippingAmountCurrency) {
        this.invoiceItemShippingCurrencyCode = invoiceShippingAmountCurrency;
    }

    public String getInvoiceItemShippingDescription() {
        return invoiceItemShippingDescription;
    }

    public void setInvoiceItemShippingDescription(String invoiceShippingDescription) {
        this.invoiceItemShippingDescription = invoiceShippingDescription;
    }

    public BigDecimal getInvoiceItemSpecialHandlingAmount() {
        if (this.isInvoiceFileSpecialHandlingInLineIndicator()) {
            BigDecimal returnValue = zero;
            try {
                for (ElectronicInvoiceRejectItem eiri : this.invoiceRejectItems) {
                    BigDecimal toAddAmount = eiri.getInvoiceItemSpecialHandlingAmount();
                    LOG.debug(
                            "getTotalAmount() setting returnValue with arithmetic => '{}' + '{}'",
                            returnValue::doubleValue,
                            toAddAmount::doubleValue
                    );
                    returnValue = returnValue.add(toAddAmount);
                }
                LOG.debug("getTotalAmount() returning amount {}", returnValue::doubleValue);
                return returnValue;
            } catch (NumberFormatException n) {
                // do nothing this is already rejected
                LOG.error(
                        "getTotalAmount() Error attempting to calculate total amount for invoice with filename {}",
                        this.invoiceFileName
                );
                return zero;
            }
        } else {
            return invoiceItemSpecialHandlingAmount;
        }
    }

    public void setInvoiceItemSpecialHandlingAmount(BigDecimal invoiceSpecialHandlingAmount) {
        this.invoiceItemSpecialHandlingAmount = invoiceSpecialHandlingAmount;
    }

    public String getInvoiceItemSpecialHandlingCurrencyCode() {
        return invoiceItemSpecialHandlingCurrencyCode;
    }

    public void setInvoiceItemSpecialHandlingCurrencyCode(String invoiceSpecialHandlingAmountCurrency) {
        this.invoiceItemSpecialHandlingCurrencyCode = invoiceSpecialHandlingAmountCurrency;
    }

    public String getInvoiceItemSpecialHandlingDescription() {
        return invoiceItemSpecialHandlingDescription;
    }

    public void setInvoiceItemSpecialHandlingDescription(String invoiceSpecialHandlingDescription) {
        this.invoiceItemSpecialHandlingDescription = invoiceSpecialHandlingDescription;
    }

    public BigDecimal getInvoiceItemSubTotalAmount() {
        return invoiceItemSubTotalAmount;
    }

    public void setInvoiceItemSubTotalAmount(BigDecimal invoiceSubTotalAmount) {
        this.invoiceItemSubTotalAmount = invoiceSubTotalAmount;
    }

    public String getInvoiceItemSubTotalCurrencyCode() {
        return invoiceItemSubTotalCurrencyCode;
    }

    public void setInvoiceItemSubTotalCurrencyCode(String invoiceSubTotalAmountCurrency) {
        this.invoiceItemSubTotalCurrencyCode = invoiceSubTotalAmountCurrency;
    }

    public BigDecimal getInvoiceItemTaxAmount() {
        BigDecimal returnValue = zero;
        boolean enableSalesTaxInd = SpringContext.getBean(ParameterService.class).getParameterValueAsBoolean(
                KfsParameterConstants.PURCHASING_DOCUMENT.class, PurapParameterConstants.ENABLE_SALES_TAX_IND);

        try {
            //if sales tax enabled, calculate total by totaling items
            if (enableSalesTaxInd) {
                for (ElectronicInvoiceRejectItem eiri : this.invoiceRejectItems) {
                    BigDecimal toAddAmount = eiri.getInvoiceItemTaxAmount();
                    LOG.debug(
                            "getTotalAmount() setting returnValue with arithmetic => '{}' + '{}'",
                            returnValue::doubleValue,
                            toAddAmount::doubleValue
                    );
                    returnValue = returnValue.add(toAddAmount);
                }
            } else {
                // else take the total, which should be the summary tax total
                returnValue = returnValue.add(this.invoiceItemTaxAmount);
            }

            LOG.debug("getTotalAmount() returning amount {}", returnValue::doubleValue);
            return returnValue;
        } catch (NumberFormatException n) {
            // do nothing this is already rejected
            LOG.error(
                    "getTotalAmount() Error attempting to calculate total amount for invoice with filename {}",
                    this.invoiceFileName
            );
            return zero;
        }
    }

    public void setInvoiceItemTaxAmount(BigDecimal invoiceTaxAmount) {
        this.invoiceItemTaxAmount = invoiceTaxAmount;
    }

    public String getInvoiceItemTaxCurrencyCode() {
        return invoiceItemTaxCurrencyCode;
    }

    public void setInvoiceItemTaxCurrencyCode(String invoiceTaxAmountCurrency) {
        this.invoiceItemTaxCurrencyCode = invoiceTaxAmountCurrency;
    }

    public String getInvoiceItemTaxDescription() {
        return invoiceItemTaxDescription;
    }

    public void setInvoiceItemTaxDescription(String invoiceTaxDescription) {
        this.invoiceItemTaxDescription = invoiceTaxDescription;
    }

    public String getInvoiceOrderMasterAgreementInformationDate() {
        return invoiceOrderMasterAgreementInformationDate;
    }

    public void setInvoiceOrderMasterAgreementInformationDate(String masterAgreementInfoDate) {
        this.invoiceOrderMasterAgreementInformationDate = masterAgreementInfoDate;
    }

    public String getInvoiceOrderMasterAgreementInformationIdentifier() {
        return invoiceOrderMasterAgreementInformationIdentifier;
    }

    public void setInvoiceOrderMasterAgreementInformationIdentifier(String masterAgreementInfoId) {
        this.invoiceOrderMasterAgreementInformationIdentifier = masterAgreementInfoId;
    }

    public String getInvoiceOrderMasterAgreementReferenceDate() {
        return invoiceOrderMasterAgreementReferenceDate;
    }

    public void setInvoiceOrderMasterAgreementReferenceDate(String masterAgreementReferenceDate) {
        this.invoiceOrderMasterAgreementReferenceDate = masterAgreementReferenceDate;
    }

    public String getInvoiceOrderMasterAgreementReferenceIdentifier() {
        return invoiceOrderMasterAgreementReferenceIdentifier;
    }

    public void setInvoiceOrderMasterAgreementReferenceIdentifier(String masterAgreementReferenceId) {
        this.invoiceOrderMasterAgreementReferenceIdentifier = masterAgreementReferenceId;
    }

    public String getInvoiceOrderReferenceDocumentReferencePayloadIdentifier() {
        return invoiceOrderReferenceDocumentReferencePayloadIdentifier;
    }

    public void setInvoiceOrderReferenceDocumentReferencePayloadIdentifier(String referenceDocumentRefPayloadId) {
        this.invoiceOrderReferenceDocumentReferencePayloadIdentifier = referenceDocumentRefPayloadId;
    }

    public String getInvoiceOrderReferenceDocumentReferenceText() {
        return invoiceOrderReferenceDocumentReferenceText;
    }

    public void setInvoiceOrderReferenceDocumentReferenceText(String referenceDocumentRefText) {
        this.invoiceOrderReferenceDocumentReferenceText = referenceDocumentRefText;
    }

    public String getInvoiceOrderReferenceOrderIdentifier() {
        return invoiceOrderReferenceOrderIdentifier;
    }

    public void setInvoiceOrderReferenceOrderIdentifier(String referenceOrderId) {
        this.invoiceOrderReferenceOrderIdentifier = referenceOrderId;
    }

    public String getInvoiceRemitToAddressCityName() {
        return invoiceRemitToAddressCityName;
    }

    public void setInvoiceRemitToAddressCityName(String remitToAddressCityName) {
        this.invoiceRemitToAddressCityName = remitToAddressCityName;
    }

    public String getInvoiceRemitToAddressCountryCode() {
        return invoiceRemitToAddressCountryCode;
    }

    public void setInvoiceRemitToAddressCountryCode(String remitToAddressCountryCode) {
        this.invoiceRemitToAddressCountryCode = remitToAddressCountryCode;
    }

    public String getInvoiceRemitToAddressCountryName() {
        return invoiceRemitToAddressCountryName;
    }

    public void setInvoiceRemitToAddressCountryName(String remitToAddressCountryName) {
        this.invoiceRemitToAddressCountryName = remitToAddressCountryName;
    }

    public String getInvoiceRemitToAddressLine1() {
        return invoiceRemitToAddressLine1;
    }

    public void setInvoiceRemitToAddressLine1(String remitToAddressLine1) {
        this.invoiceRemitToAddressLine1 = remitToAddressLine1;
    }

    public String getInvoiceRemitToAddressLine2() {
        return invoiceRemitToAddressLine2;
    }

    public void setInvoiceRemitToAddressLine2(String remitToAddressLine2) {
        this.invoiceRemitToAddressLine2 = remitToAddressLine2;
    }

    public String getInvoiceRemitToAddressLine3() {
        return invoiceRemitToAddressLine3;
    }

    public void setInvoiceRemitToAddressLine3(String remitToAddressLine3) {
        this.invoiceRemitToAddressLine3 = remitToAddressLine3;
    }

    public String getInvoiceRemitToAddressName() {
        return invoiceRemitToAddressName;
    }

    public void setInvoiceRemitToAddressName(String remitToAddressName) {
        this.invoiceRemitToAddressName = remitToAddressName;
    }

    public String getInvoiceRemitToAddressPostalCode() {
        return invoiceRemitToAddressPostalCode;
    }

    public void setInvoiceRemitToAddressPostalCode(String remitToAddressPostalCode) {
        this.invoiceRemitToAddressPostalCode = remitToAddressPostalCode;
    }

    public String getInvoiceRemitToAddressStateCode() {
        return invoiceRemitToAddressStateCode;
    }

    public void setInvoiceRemitToAddressStateCode(String remitToAddressStateCode) {
        this.invoiceRemitToAddressStateCode = remitToAddressStateCode;
    }

    public String getInvoiceRemitToAddressType() {
        return invoiceRemitToAddressType;
    }

    public void setInvoiceRemitToAddressType(String remitToAddressType) {
        this.invoiceRemitToAddressType = remitToAddressType;
    }

    public String getInvoiceShipToAddressCityName() {
        return invoiceShipToAddressCityName;
    }

    public void setInvoiceShipToAddressCityName(String shipToAddressCityName) {
        this.invoiceShipToAddressCityName = shipToAddressCityName;
    }

    public String getInvoiceShipToAddressCountryCode() {
        return invoiceShipToAddressCountryCode;
    }

    public void setInvoiceShipToAddressCountryCode(String shipToAddressCountryCode) {
        this.invoiceShipToAddressCountryCode = shipToAddressCountryCode;
    }

    public String getInvoiceShipToAddressCountryName() {
        return invoiceShipToAddressCountryName;
    }

    public void setInvoiceShipToAddressCountryName(String shipToAddressCountryName) {
        this.invoiceShipToAddressCountryName = shipToAddressCountryName;
    }

    public String getInvoiceShipToAddressLine1() {
        return invoiceShipToAddressLine1;
    }

    public void setInvoiceShipToAddressLine1(String shipToAddressLine1) {
        this.invoiceShipToAddressLine1 = shipToAddressLine1;
    }

    public String getInvoiceShipToAddressLine2() {
        return invoiceShipToAddressLine2;
    }

    public void setInvoiceShipToAddressLine2(String shipToAddressLine2) {
        this.invoiceShipToAddressLine2 = shipToAddressLine2;
    }

    public String getInvoiceShipToAddressLine3() {
        return invoiceShipToAddressLine3;
    }

    public void setInvoiceShipToAddressLine3(String shipToAddressLine3) {
        this.invoiceShipToAddressLine3 = shipToAddressLine3;
    }

    public String getInvoiceShipToAddressName() {
        return invoiceShipToAddressName;
    }

    public void setInvoiceShipToAddressName(String shipToAddressName) {
        this.invoiceShipToAddressName = shipToAddressName;
    }

    public String getInvoiceShipToAddressPostalCode() {
        return invoiceShipToAddressPostalCode;
    }

    public void setInvoiceShipToAddressPostalCode(String shipToAddressPostalCode) {
        this.invoiceShipToAddressPostalCode = shipToAddressPostalCode;
    }

    public String getInvoiceShipToAddressStateCode() {
        return invoiceShipToAddressStateCode;
    }

    public void setInvoiceShipToAddressStateCode(String shipToAddressStateCode) {
        this.invoiceShipToAddressStateCode = shipToAddressStateCode;
    }

    public String getInvoiceShipToAddressType() {
        return invoiceShipToAddressType;
    }

    public void setInvoiceShipToAddressType(String shipToAddressType) {
        this.invoiceShipToAddressType = shipToAddressType;
    }

    public String getInvoiceOrderSupplierOrderInformationIdentifier() {
        return invoiceOrderSupplierOrderInformationIdentifier;
    }

    public void setInvoiceOrderSupplierOrderInformationIdentifier(String supplierOrderInfoId) {
        this.invoiceOrderSupplierOrderInformationIdentifier = supplierOrderInfoId;
    }

    public Integer getVendorDetailAssignedIdentifier() {
        return vendorDetailAssignedIdentifier;
    }

    public void setVendorDetailAssignedIdentifier(Integer vendorDetailID) {
        this.vendorDetailAssignedIdentifier = vendorDetailID;
    }

    public String getVendorDunsNumber() {
        return vendorDunsNumber;
    }

    public void setVendorDunsNumber(String vendorDunsNumber) {
        this.vendorDunsNumber = vendorDunsNumber;
    }

    public Integer getVendorHeaderGeneratedIdentifier() {
        return vendorHeaderGeneratedIdentifier;
    }

    public void setVendorHeaderGeneratedIdentifier(Integer vendorHeaderID) {
        this.vendorHeaderGeneratedIdentifier = vendorHeaderID;
    }

    public void addRejectItem(ElectronicInvoiceRejectItem item) {
        item.setPurapDocumentIdentifier(this.purapDocumentIdentifier);
        item.setElectronicInvoiceRejectDocument(this);

        invoiceRejectItems.add(item);
    }

    public void addRejectReason(ElectronicInvoiceRejectReason reason) {
        reason.setPurapDocumentIdentifier(this.purapDocumentIdentifier);
        reason.setElectronicInvoiceRejectDocument(this);

        invoiceRejectReasons.add(reason);
    }

    public PurApRelatedViews getRelatedViews() {
        if (relatedViews == null) {
            relatedViews = new PurApRelatedViews(this.documentNumber,
                    this.accountsPayablePurchasingDocumentLinkIdentifier);
        }
        return relatedViews;
    }

    public void setRelatedViews(PurApRelatedViews relatedViews) {
        this.relatedViews = relatedViews;
    }

    public boolean isBoNotesSupport() {
        return true;
    }

    public boolean isDocumentCreationInProgress() {
        return isDocumentCreationInProgress;
    }

    public void setDocumentCreationInProgress(boolean isDocumentCreationInProgress) {
        this.isDocumentCreationInProgress = isDocumentCreationInProgress;
    }

    public String getVendorNumber() {
        if (StringUtils.isNotEmpty(vendorNumber)) {
            return vendorNumber;
        } else if (ObjectUtils.isNotNull(vendorDetail)) {
            return vendorDetail.getVendorNumber();
        } else if (getVendorHeaderGeneratedIdentifier() != null && getVendorDetailAssignedIdentifier() != null) {
            VendorDetail vendorDetail = new VendorDetail();
            vendorDetail.setVendorHeaderGeneratedIdentifier(getVendorHeaderGeneratedIdentifier());
            vendorDetail.setVendorDetailAssignedIdentifier(getVendorDetailAssignedIdentifier());
            return vendorDetail.getVendorNumber();
        } else {
            return "";
        }
    }

    public void setVendorNumber(String vendorNumber) {
        this.vendorNumber = vendorNumber;
    }

    /**
     * Always returns false.
     * This method is needed here because it's called by some tag files shared with PurAp documents.
     */
    public boolean getIsATypeOfPurAPRecDoc() {
        return false;
    }

    /**
     * Always returns false.
     * This method is needed here because it's called by some tag files shared with PurAp documents.
     */
    public boolean getIsATypeOfPurDoc() {
        return false;
    }

    /**
     * Always returns false.
     * This method is needed here because it's called by some tag files shared with PurAp documents.
     */
    public boolean getIsATypeOfPODoc() {
        return false;
    }

    /**
     * Always returns false.
     * This method is needed here because it's called by some tag files shared with PurAp documents.
     */
    public boolean getIsPODoc() {
        return false;
    }

    /**
     * Always returns false.
     * This method is needed here because it's called by some tag files shared with PurAp documents.
     */
    public boolean getIsReqsDoc() {
        return false;
    }

    public boolean isInvoiceResearchIndicatorForSearching() {
        return invoiceResearchIndicator;
    }

    public String getInvoiceResearchIndicatorForResult() {
        if (isInvoiceResearchIndicator()) {
            return "Yes";
        } else {
            return "No";
        }
    }

    public String getPurchaseOrderDeliveryCampusCodeForSearch() {
        return getPurchaseOrderDeliveryCampusCode();
    }

    public CampusParameter getPurchaseOrderDeliveryCampus() {
        return purchaseOrderDeliveryCampus;
    }

    public Date getAccountsPayableApprovalDateForSearch() {
        if (getAccountsPayableApprovalTimestamp() != null) {
            DateTimeService dateTimeService = SpringContext.getBean(DateTimeService.class);
            return dateTimeService.convertToSqlDate(getAccountsPayableApprovalTimestamp());
        }
        return null;
    }

    public Timestamp getInvoiceFileTimeStampForSearch() {
        Date invoiceDate = ElectronicInvoiceUtils.getDate(getInvoiceFileDate());
        if (invoiceDate != null) {
            return new Timestamp(invoiceDate.getTime());
        } else {
            return null;
        }
    }

    public void setInvoiceFileTimeStampForSearch(Timestamp invoiceFileTimeStamp) {
        //Not needed
    }

    public String getWorkflowStatusForResult() {
        return PurapSearchUtils.getWorkFlowStatusString(getDocumentHeader());
    }

    /**
     * Checks whether the related purchase order views need a warning to be displayed, i.e. if at least one of the
     * purchase orders has never been opened.
     *
     * @return true if at least one related purchase order needs a warning; false otherwise
     */
    public boolean getNeedWarningRelatedPOs() {
        List<PurchaseOrderView> poViews = getRelatedViews().getRelatedPurchaseOrderViews();
        for (PurchaseOrderView poView : poViews) {
            if (poView.getNeedWarning()) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void doRouteStatusChange(DocumentRouteStatusChange statusChangeEvent) {
        LOG.debug("doRouteStatusChange() started");
        super.doRouteStatusChange(statusChangeEvent);
        if (this.getDocumentHeader().getWorkflowDocument().isApproved()) {
            //Set the current date as approval timestamp
            this.setAccountsPayableApprovalTimestamp(SpringContext.getBean(DateTimeService.class)
                    .getCurrentTimestamp());
        }
    }

    @Override
    public List<Note> getNotes() {
        List<Note> notes = super.getNotes();
        if (StringUtils.isNotBlank(this.getObjectId())) {
            for (Note note : notes) {
                if (StringUtils.isNotBlank(note.getObjectId())) {
                    note.refresh();
                }
            }
        }

        return super.getNotes();
    }

    @Override
    public NoteType getNoteType() {
        return NoteType.BUSINESS_OBJECT;
    }

}
