/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document.authorization;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.kim.bo.impl.KimAttributes;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.purap.PurapAuthorizationConstants;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.PurapPropertyConstants;
import org.kuali.kfs.module.purap.RequisitionStatuses;
import org.kuali.kfs.module.purap.businessobject.RequisitionItem;
import org.kuali.kfs.module.purap.document.RequisitionDocument;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class RequisitionDocumentAuthorizer extends PurchasingAccountsPayableTransactionalDocumentAuthorizerBase {

    private static final Logger LOG = LogManager.getLogger();

    /**
     * Overridden so that we can remove the allowCapitalAssetEdit edit mode if the user (commodity reviewer)
     * doesn't have the permission to edit an object code on an accounting line (with a commodity code) since these
     * two edit abilities should go together.
     */
    @Override
    public Set<String> getEditModes(Document document, Person user, Set<String> editModes) {
        LOG.debug("getEditModes() started");

        editModes = super.getEditModes(document, user, editModes);
        RequisitionDocument requisitionDocument = (RequisitionDocument) document;
        Map<String, String> permissionDetails = new HashMap<>();

        permissionDetails.put(KFSPropertyConstants.DOCUMENT_TYPE_NAME, PurapConstants.PurapDocTypeCodes.REQUISITION_DOCUMENT_TYPE);
        permissionDetails.put(KimAttributes.ROUTE_NODE_NAME,
                RequisitionStatuses.NODE_COMMODITY_CODE_REVIEW);
        permissionDetails.put(KFSPropertyConstants.PROPERTY_NAME, PurapPropertyConstants.ITEMS +
                KFSConstants.DELIMITER + KFSPropertyConstants.SOURCE_ACCOUNTING_LINES + KFSConstants.DELIMITER +
                PurapPropertyConstants.FINANCIAL_OBJECT_CODE);

        if (requisitionDocument.isDocumentStoppedInRouteNode(
            RequisitionStatuses.NODE_COMMODITY_CODE_REVIEW)) {

            boolean removeCapitalAssetEdit = true;

            for (RequisitionItem purApItem: (List<RequisitionItem>) requisitionDocument.getItems()) {
                if (ObjectUtils.isNotNull(purApItem)) {
                    Map<String, String> roleQualifiers = new HashMap<>();
                    final String purchasingCommodityCode = purApItem.getPurchasingCommodityCode();
                    if (StringUtils.isNotBlank(purchasingCommodityCode)) {
                        roleQualifiers.put(PurapPropertyConstants.ITEM_COMMODITY_CODE, purchasingCommodityCode);
                        if (isAuthorizedByTemplate(document,
                            KFSConstants.PermissionTemplate.MODIFY_ACCOUNTING_LINES.namespace,
                            KFSConstants.PermissionTemplate.MODIFY_ACCOUNTING_LINES.name, user.getPrincipalId(),
                            permissionDetails, roleQualifiers)) {
                            removeCapitalAssetEdit = false;
                            break;
                        }
                    }
                }
            }
            if (removeCapitalAssetEdit) {
                editModes.remove(PurapAuthorizationConstants.RequisitionEditMode.ALLOW_CAPITAL_ASSET_EDITS);
            }
        }

        return editModes;
    }

}
