/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.util.cxml;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlTransient;
import jakarta.xml.bind.annotation.XmlType;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.kuali.kfs.core.api.impex.xml.XmlConstants;
import org.kuali.kfs.module.purap.businessobject.B2BShoppingCartItem;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "", propOrder = {"cxmlHeader", "message"})
@XmlRootElement(name = "cXML", namespace = XmlConstants.B2B_PUNCH_OUT_ORDER_NAMESPACE)
public class B2BShoppingCart extends B2BShoppingCartBase {

    @XmlElement(name = "Message", namespace = XmlConstants.B2B_PUNCH_OUT_ORDER_NAMESPACE, required = true)
    private Message message = new Message();

    @XmlElement(name = "Header", namespace = XmlConstants.B2B_PUNCH_OUT_ORDER_NAMESPACE, required = true)
    private CxmlHeader cxmlHeader = new CxmlHeader();

    @XmlTransient
    private List<B2BShoppingCartItem> itemsList;

    public void addShoppingCartItem(final B2BShoppingCartItem item) {
        if (itemsList == null) {
            itemsList = new ArrayList<>();
        }
        itemsList.add(item);
    }

    public List<B2BShoppingCartItem> getItems() {
        if (CollectionUtils.isEmpty(itemsList)) {
            if (message.getPunchOutOrderMessage() != null
                && CollectionUtils.isNotEmpty(message.getPunchOutOrderMessage().getItemInList())) {
                return message.getPunchOutOrderMessage()
                        .getItemInList()
                        .stream()
                        .map(ItemIn::toB2BShoppingCartItem)
                        .collect(Collectors.toList());
            }
        }
        return itemsList;
    }

    @Override
    public boolean isSuccess() {
        return SUCCESS_STATUS_CODE.equals(message.getStatus().getCode());
    }

    public CxmlHeader getCxmlHeader() {
        return cxmlHeader;
    }

    public void setCxmlHeader(final CxmlHeader cxmlHeader) {
        this.cxmlHeader = cxmlHeader;
    }

    public String getBuyerCookieText() {
        return message.getPunchOutOrderMessage().getBuyerCookie();
    }

    public void setBuyerCookieText(final String buyerCookieText) {
        message.getPunchOutOrderMessage().setBuyerCookie(buyerCookieText);
    }

    public String getTotal() {
        return message.getPunchOutOrderMessage()
                .getPunchOutOrderMessageHeader()
                .getTotal()
                .getMoney()
                .getValue()
                .toString();
    }

    public void setTotal(final String totalAmount) {
        message.getPunchOutOrderMessage().getPunchOutOrderMessageHeader().getTotal().getMoney()
                .setValue(new BigDecimal(totalAmount));
    }

    public Message getMessage() {
        return message;
    }

    public void setMessage(final Message message) {
        this.message = message;
    }

    public String getMessageStatusCode() {
        return message.getStatus().getCode();
    }

    public void setMessageStatusCode(final String messageStatusCode) {
        message.getStatus().setCode(messageStatusCode);
    }

    public String getMessageStatusText() {
        return message.getStatus().getText();
    }

    public void setMessageStatusText(final String messageStatusText) {
        message.getStatus().setText(messageStatusText);
    }

    @Override
    public String toString() {
        final ToStringBuilder toString = new ToStringBuilder(this);

        toString.append("messageStatusCode", getMessageStatusCode());
        toString.append("messageStatusText", getMessageStatusText());
        toString.append("statusCode", getStatusCode());
        toString.append("statusText", getStatusText());
        toString.append("buyerCookieText", getBuyerCookieText());
        toString.append("totalAmount", getTotal());
        toString.append("CXMLHeader", cxmlHeader);
        toString.append("Items", itemsList);

        return toString.toString();
    }

}
