/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.rice.kim.api.identity;

import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.krad.bo.BusinessObject;

import java.util.List;
import java.util.Map;

/**
 * Person object for use by the KNS and KNS-based applications.  This provides an abstraction layer
 * between application code and the KIM objects to simplify use. 
 */
public interface Person extends BusinessObject {

    String getPrincipalId();

    String getPrincipalName();

    String getEntityId();

    String getEntityTypeCode();

    /**
     * The first name from the default name record for the entity.
     */
    String getFirstName();

    String getFirstNameUnmasked();

    String getMiddleName();

    String getMiddleNameUnmasked();

    String getLastName();

    String getLastNameUnmasked();

    /*
     * Method which composites the first, middle and last names.
     */
    String getName();

    String getNameUnmasked();

    String getEmailAddress();

    String getEmailAddressUnmasked();

    /**
     * Returns line1 of the default address for the Person.  Will lazy-load the information from the
     * IdentityManagementService if requested.
     */
    String getAddressLine1();

    String getAddressLine1Unmasked();

    /**
     * Returns line2 of the default address for the Person.  Will lazy-load the information from the
     * IdentityManagementService if requested.
     */
    String getAddressLine2();

    String getAddressLine2Unmasked();

    /**
     * Returns line3 of the default address for the Person.  Will lazy-load the information from the
     * IdentityManagementService if requested.
     */
    String getAddressLine3();

    String getAddressLine3Unmasked();

    /**
     * Returns the city name from the default address for the Person.  Will lazy-load the information from the
     * IdentityManagementService if requested.
     */
    String getAddressCity();

    String getAddressCityUnmasked();

    /**
     * Returns the state code from the default address for the Person.  Will lazy-load the information from the
     * IdentityManagementService if requested.
     */
    String getAddressStateProvinceCode();

    String getAddressStateProvinceCodeUnmasked();

    /**
     * Returns the postal code from the default address for the Person.  Will lazy-load the information from the
     * IdentityManagementService if requested.
     */
    String getAddressPostalCode();

    String getAddressPostalCodeUnmasked();

    /**
     * Returns the country code from the default address for the Person.  Will lazy-load the information from the
     * IdentityManagementService if requested.
     */
    String getAddressCountryCode();

    String getAddressCountryCodeUnmasked();

    /** Returns the default phone number for the entity.
     */
    String getPhoneNumber();

    String getPhoneNumberUnmasked();

    String getCampusCode();

    Map<String, String> getExternalIdentifiers();

    /** Checks whether the person has an affiliation of a particular type: staff/faculty/student/etc... */
    boolean hasAffiliationOfType(String affiliationTypeCode);

    List<String> getCampusCodesForAffiliationOfType(String affiliationTypeCode);

    String getEmployeeStatusCode();

    String getEmployeeTypeCode();

    KualiDecimal getBaseSalaryAmount();

    String getExternalId(String externalIdentifierTypeCode);

    String getPrimaryDepartmentCode();

    String getEmployeeId();

    boolean isActive();

    String getLookupRoleNamespaceCode();

    String getLookupRoleName();
}
