/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import axios from 'axios';
import { config } from 'dotenv';
import { exec } from 'child_process';
import { stringify } from 'qs';

// When running locally, it expects a .env file for environment variables
config({ path: './node-scripts/.env' });

const URLS = {
  SEARCH: 'rest/api/3/search',
  EDIT: 'rest/api/3/issue',
  BULK: 'rest/api/3/issue/bulk'
};

const JIRA_IDS = {
  ISSUE_TYPE: '4', // Improvement
  PARENT: '92820', // PD Maintenance
  PROJECT: '11602', // FINP
  FIX_VERSION: '19055', // SMALL
  PRIORITY: '10003', // Normal,
  IMPACTED_MODULES: '10614' // System
};

const CUSTOM_FIELD_MAP = {
  ACCEPTANCE_CRITERIA: 'customfield_11400',
  TESTING_STEPS: 'customfield_14800',
  IMPACTED_MODULES: 'customfield_11401'
};

const SUMMARY = '[npm Upgrade] [[packageName]]';

const BASE_SEARCH_JQL = `project = "FINP" AND status != "Closed" AND summary ~ "${SUMMARY}"`;

const FIELD_LIMITER =
  '&fields=issuetype,project,customfield_11400,priority,summary,parent,fixVersions,description';

const knownGroups = {
  '@babel': ['@babel', 'babel'],
  '@commitlint': ['@commitlint', 'commitizen', 'commitlint'],
  '@emotion': ['@emotion'],
  '@mui': ['@mui'],
  '@rollup': ['@rollup', 'rollup'],
  '@storybook': ['@storybook'],
  '@testing-library': ['@testing-library'],
  '@typescript-eslint': ['@typescript-eslint'],
  eslint: ['eslint'],
  jest: ['jest'],
  postcss: ['postcss'],
  stylelint: ['stylelint'],
  axios: ['axios']
};

// This will contain a list of packages that we are unable to upgrade at this time
const skipCheck = [
  // nanoid v4+ is only compatible with ESM projects, which we are not, so we need to remain on v3, which is still supported in parallel
  'nanoid',
  // The importFrom config option has been removed. Need to look at a new way of doing this,
  // though they didn't really give much help on how to do this.
  'postcss-custom-properties'
];

// This is a list of packages that need to be split when searching, as for whatever
// reason Jira doesn't return search results when sent intact.
const tokenize = ['regenerator-runtime'];

const generateMarkupBlock = (plainText, list, additionalPlainText) => {
  return {
    version: 1,
    type: 'doc',
    content: [
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: plainText
          }
        ]
      },
      {
        type: 'bulletList',
        content: list.map((item) => {
          return {
            type: 'listItem',
            content: [
              {
                type: 'paragraph',
                content: [
                  {
                    type: 'text',
                    text: item
                  }
                ]
              }
            ]
          };
        })
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: additionalPlainText
          }
        ]
      }
    ]
  };
};

const generateAcceptanceCriteria = (packages) => {
  return generateMarkupBlock(
    'The following packages are updated:',
    packages,
    'If any of these packages cannot be updated, they are added to the skipCheck list in the script.'
  );
};

const generateDescription = (groupName, packages) => {
  return generateMarkupBlock(
    `The following packages related to ${groupName} are out of date and need to be updated:`,
    packages,
    'If any of these packages cannot be updated, they are added to the skipCheck list in the script.'
  );
};

const generateTestingSteps = () => {
  return {
    version: 1,
    type: 'doc',
    content: [
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to the following lookups and ensure they work as expected:'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Organization Options'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Asset'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Object Code'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Labor Benefits Calculation '
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Dunning Letter Template'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Institution Configuration and do a few edits to verify the form works as expected.'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Customer Invoice and enter a description along with the following details:'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Customer: ABB2'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Chart: BL'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Account: 1031400'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Object: 5000'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Item Quantity: 1'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Invoice Item UOM Code: EA'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Invoice Item Unit Price: 300'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Add icon'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Submit'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Cash Control and enter a description along with the following details:'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Medium Code: Check'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Customer: ABB2'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Date: today’s date'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Amount: 150'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Add icon'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Submit'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click on the APPLICATION DOC # link (will appear blue on Cash Control doc) to open the Payment Application.'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Enter 150 in the Apply Amount field in the APPLY TO INVOICE DETAIL section and click the Apply button'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Blanket Approve'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Login as wwren and approve the Cash Control document in the Action List.'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to the Doc Search and search on today’s date. '
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Open the Payment Application document and click the Adjust button. '
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Update the Item Amount Applied to 100.  In the Unapplied section, add Customer Number ABB2 and enter 50 as the Unapplied Amount. Click Submit.'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Batch Schedule and run nightly jobs '
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Batch File and open the scrubberJob.txt file and verify the document opens successfully.'
          }
        ]
      }
    ]
  };
};

const getHeaders = () => {
  return {
    Authorization: `Basic ${Buffer.from(process.env.JIRA_TOKEN).toString(
      'base64'
    )}`,
    Accept: 'application/json'
  };
};

const search = async (packageName) => {
  const url = `${process.env.JIRA_URL}${URLS.SEARCH}${stringify(
    {
      jql: BASE_SEARCH_JQL.replace(
        '[[packageName]]',
        tokenize.includes(packageName)
          ? packageName.split('-').join('" AND summary ~ "')
          : packageName
      )
    },
    { addQueryPrefix: true }
  )}${FIELD_LIMITER}`;
  try {
    const { data } = await axios.get(url, { headers: getHeaders() });
    if (data.total > 0) {
      return data.issues[0];
    }
  } catch (error) {
    if (error.response.data) {
      console.dir(error.response.data, { depth: null });
    }
    process.exitCode = 1;
  }
};

const generateCreateBody = (groupName, group) => {
  return {
    fields: {
      summary: SUMMARY.replace('[[packageName]]', groupName),
      issuetype: {
        id: JIRA_IDS.ISSUE_TYPE
      },
      project: {
        id: JIRA_IDS.PROJECT
      },
      description: generateDescription(groupName, group.packages),
      fixVersions: [
        {
          id: JIRA_IDS.FIX_VERSION
        }
      ],
      priority: {
        id: JIRA_IDS.PRIORITY
      },
      parent: {
        id: JIRA_IDS.PARENT
      },
      [CUSTOM_FIELD_MAP.ACCEPTANCE_CRITERIA]: generateAcceptanceCriteria(
        group.packages
      ),
      [CUSTOM_FIELD_MAP.TESTING_STEPS]: generateTestingSteps(),
      [CUSTOM_FIELD_MAP.IMPACTED_MODULES]: [{ id: JIRA_IDS.IMPACTED_MODULES }]
    }
  };
};

const generateUpdateBody = (issue, groupName, packages) => {
  let updateDescription = false;
  let updateAcceptanceCriteria = false;

  let body = null;

  if (!issue.fields.description) {
    updateDescription = true;
  } else {
    issue.fields.description.content.forEach((content) => {
      if (updateDescription || content.type !== 'bulletList') {
        return;
      }
      content.content.forEach((listContent) => {
        if (!packages.includes(listContent.content[0].content[0].text)) {
          updateDescription = true;
        }
      });
    });

    if (!updateDescription) {
      issue.fields.description.content.forEach((content) => {
        if (updateDescription || content.type !== 'bulletList') {
          return;
        }
        content.content.forEach((listContent) => {
          if (!packages.includes(listContent.content[0].content[0].text)) {
            updateDescription = true;
          }
        });
      });
    }
  }

  if (!issue.fields[CUSTOM_FIELD_MAP.ACCEPTANCE_CRITERIA]) {
    updateAcceptanceCriteria = true;
  } else {
    issue.fields[CUSTOM_FIELD_MAP.ACCEPTANCE_CRITERIA].content.forEach(
      (content) => {
        if (updateAcceptanceCriteria || content.type !== 'bulletList') {
          return;
        }
        content.content.forEach((listContent) => {
          if (!packages.includes(listContent.content[0].content[0].text)) {
            updateDescription = true;
          }
        });
      }
    );

    if (!updateAcceptanceCriteria) {
      issue.fields[CUSTOM_FIELD_MAP.ACCEPTANCE_CRITERIA].content.forEach(
        (content) => {
          if (updateAcceptanceCriteria || content.type !== 'bulletList') {
            return;
          }
          content.content.forEach((listContent) => {
            if (!packages.includes(listContent.content[0].content[0].text)) {
              updateDescription = true;
            }
          });
        }
      );
    }
  }

  if (updateDescription) {
    body = {
      ...(body ?? {}),
      description: generateDescription(groupName, packages)
    };
  }

  if (updateDescription) {
    body = {
      ...(body ?? {}),
      [CUSTOM_FIELD_MAP.ACCEPTANCE_CRITERIA]:
        generateAcceptanceCriteria(packages)
    };
  }

  return body;
};

const bulkCreate = async (groupedValues) => {
  const url = `${process.env.JIRA_URL}${URLS.BULK}`;
  const headers = getHeaders();
  const bulk = [];
  for (const groupName in groupedValues) {
    if (groupedValues[groupName].issue) {
      continue;
    }
    bulk.push(generateCreateBody(groupName, groupedValues[groupName]));
  }
  let iteration = 0;
  const take = 50; // Maximum number of issues per bulk command
  while (iteration * take < bulk.length) {
    const body = {
      issueUpdates: bulk.slice(iteration * take, take)
    };
    iteration++;
    try {
      await axios.post(url, body, { headers });
    } catch (error) {
      console.dir(error.response.data, { depth: null });
      process.exitCode = 1;
    }
  }
};

const updateExisting = async (groupedValues) => {
  const headers = getHeaders();
  for (const groupName in groupedValues) {
    const group = groupedValues[groupName];
    if (!group.issue) {
      continue;
    }
    const body = generateUpdateBody(group.issue, groupName, group.packages);

    if (body === null) {
      continue;
    }

    const url = `${process.env.JIRA_URL}${URLS.EDIT}/${group.issue.id}`;
    try {
      await axios.put(url, { fields: body }, { headers });
    } catch (error) {
      console.dir(error.response.data.errors, { depth: null });
      process.exitCode = 1;
    }
  }
};

const processOutdated = async (outdatedList) => {
  const groupedValues = {};
  Object.keys(outdatedList).forEach((packageName) => {
    if (skipCheck.includes(packageName)) {
      return;
    }
    let namespace;
    let exit = false;
    let isType = false;
    if (packageName.startsWith('@types/')) {
      isType = true;
      namespace = packageName.split('/')[1];
    } else if (packageName.startsWith('@')) {
      namespace = packageName.split('/')[0];
    } else {
      namespace = packageName.split('-')[0];
    }

    Object.keys(knownGroups).forEach((groupName) => {
      if (!knownGroups[groupName].includes(namespace)) {
        return;
      }
      exit = true;
      if (!groupedValues[groupName]) {
        groupedValues[groupName] = {};
      }
      groupedValues[groupName].packages = [
        ...(groupedValues[groupName]?.packages ?? []),
        packageName
      ];
    });
    if (exit) {
      return;
    }
    const indexer = isType ? namespace : packageName;
    if (!groupedValues[indexer]) {
      groupedValues[indexer] = {};
    }
    groupedValues[indexer].packages = [
      ...(groupedValues[indexer]?.packages ?? []),
      packageName
    ];
  });

  for (const groupName in groupedValues) {
    groupedValues[groupName].issue = await search(groupName);
  }
  await bulkCreate(groupedValues);
  await updateExisting(groupedValues);
};

const getOutdated = async () => {
  exec('npm outdated --json --depth 9999', async (_error, stdout, _stderr) => {
    let output;
    try {
      output = JSON.parse(stdout);
    } catch (ex) {
      console.dir(ex, { depth: null });
      process.exitCode = 1;
    }
    await processOutdated(output);
  });
};

await getOutdated();
