/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package co.kuali.rice.kew.notes.service.impl;


import co.kuali.rice.coreservice.api.attachment.RiceAttachmentDataS3Constants;
import co.kuali.rice.coreservice.api.attachment.S3FileService;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.IOUtils;
import org.kuali.rice.coreservice.framework.parameter.ParameterService;
import org.kuali.rice.kew.notes.Attachment;
import org.kuali.rice.kew.notes.service.impl.AttachmentServiceImpl;
import org.kuali.rice.krad.util.KRADConstants;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Objects;

public class S3AttachmentServiceImpl extends AttachmentServiceImpl {

    private S3FileService riceS3FileService;
    private ParameterService parameterService;

    @Override
    public void persistAttachedFileAndSetAttachmentBusinessObjectValue(Attachment attachment) throws Exception {
        if (!isS3IntegrationEnabled()) {
            super.persistAttachedFileAndSetAttachmentBusinessObjectValue(attachment);
        }

        try {
            final Class<?> s3FileClass = Class.forName(RiceAttachmentDataS3Constants.S3_FILE_CLASS);
            final Object s3File = s3FileClass.newInstance();

            final Method setFileContents = s3FileClass.getMethod(RiceAttachmentDataS3Constants.SET_BYTE_CONTENTS_METHOD, byte[].class);
            setFileContents.invoke(s3File, (Object) IOUtils.toByteArray(attachment.getAttachedObject()));
            attachment.setFileDataId(riceS3FileService.createFile(s3File));
        } catch (NoSuchMethodException|ClassNotFoundException|IllegalAccessException|InstantiationException|InvocationTargetException e) {
            throw new RuntimeException(e);
        }

        if (isS3DualSaveEnabled()) {
            super.persistAttachedFileAndSetAttachmentBusinessObjectValue(attachment);
        }
    }

    @Override
    public byte[] findAttachedFile(Attachment attachment) throws Exception {
        if (!isS3IntegrationEnabled()) {
            return super.findAttachedFile(attachment);
        }

        try {
            final Object s3File = riceS3FileService.retrieveFile(attachment.getFileDataId());
            byte[] s3Bytes = null;
            byte[] fsBytes = null;
            if (s3File != null) {
                if (LOG.isDebugEnabled()) {
                    final Method getFileMetaData = s3File.getClass().getMethod(RiceAttachmentDataS3Constants.GET_FILE_META_DATA_METHOD);
                    LOG.debug("data found in S3, existing id: " + attachment.getFileDataId() + " metadata: " + getFileMetaData.invoke(s3File));
                }

                final Method getFileContents = s3File.getClass().getMethod(RiceAttachmentDataS3Constants.GET_BYTE_CONTENTS_METHOD);
                s3Bytes = (byte[]) getFileContents.invoke(s3File);
            }

            if (s3Bytes == null || isS3DualRetrieveEnabled()) {
                try {
                    fsBytes = super.findAttachedFile(attachment);
                } catch (FileNotFoundException e) {
                    LOG.info("file not found at path " + attachment.getFileLoc());
                }
            }

            if (s3Bytes != null && fsBytes != null) {
                final String s3MD5 = DigestUtils.md5Hex(s3Bytes);
                final String dbMD5 = DigestUtils.md5Hex(fsBytes);
                if (!Objects.equals(s3MD5, dbMD5)) {
                    LOG.error("S3 data MD5: " + s3MD5 + " does not equal FS data MD5: " + dbMD5 + " for id: " + attachment.getFileDataId());
                }
            }
            return s3Bytes != null ? s3Bytes : fsBytes;
        } catch (NoSuchMethodException|InvocationTargetException |IllegalAccessException|IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void deleteAttachedFile(Attachment attachment) throws Exception {
        if (!isS3IntegrationEnabled()) {
            super.deleteAttachedFile(attachment);
        }

        riceS3FileService.deleteFile(attachment.getFileDataId());

        if (isS3DualSaveEnabled()) {
            try {
                super.deleteAttachedFile(attachment);
            } catch (FileNotFoundException e) {
                LOG.info("file not found at path " + attachment.getFileLoc());

            }
        }
    }

    protected boolean isS3IntegrationEnabled() {
        if (parameterService.parameterExists(KRADConstants.KUALI_RICE_SYSTEM_NAMESPACE, KRADConstants.DetailTypes.ALL_DETAIL_TYPE, RiceAttachmentDataS3Constants.S3_INTEGRATION_ENABLED)) {
            return parameterService.getParameterValueAsBoolean(KRADConstants.KUALI_RICE_SYSTEM_NAMESPACE, KRADConstants.DetailTypes.ALL_DETAIL_TYPE, RiceAttachmentDataS3Constants.S3_INTEGRATION_ENABLED);
        } else {
            return false;
        }
    }

    protected boolean isS3DualSaveEnabled() {
        return parameterService.getParameterValueAsBoolean(KRADConstants.KUALI_RICE_SYSTEM_NAMESPACE, KRADConstants.DetailTypes.ALL_DETAIL_TYPE, RiceAttachmentDataS3Constants.S3_DUAL_SAVE_ENABLED);
    }

    protected boolean isS3DualRetrieveEnabled() {
        return parameterService.getParameterValueAsBoolean(KRADConstants.KUALI_RICE_SYSTEM_NAMESPACE, KRADConstants.DetailTypes.ALL_DETAIL_TYPE, RiceAttachmentDataS3Constants.S3_DUAL_RETRIEVE_ENABLED);
    }

    public S3FileService getRiceS3FileService() {
        return riceS3FileService;
    }

    public void setRiceS3FileService(S3FileService riceS3FileService) {
        this.riceS3FileService = riceS3FileService;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }
}
