/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.web.spring;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.ken.bo.NotificationBo;
import org.kuali.rice.ken.bo.NotificationContentTypeBo;
import org.kuali.rice.ken.exception.ErrorList;
import org.kuali.rice.ken.util.NotificationConstants;
import org.kuali.rice.ken.util.Util;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.Map;

/**
 * This class is the controller for sending Event notification messages via an end user interface.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@RequestMapping(value="/ken")
@Controller("sendEventNotificationMessageController")
public class SendEventNotificationMessageController extends BaseSendNotificationController {

    private static final Logger LOG = LogManager.getLogger(SendEventNotificationMessageController.class);

    /**
     * Handles the display of the form for sending a event notification message.
     */
    @RequestMapping(value="/SendEventNotificationMessage.form")
    public ModelAndView sendEventNotificationMessage(HttpServletRequest request) {
        String view = "SendEventNotificationMessage";

        LOG.debug("remoteUser: " + request.getRemoteUser());

        Map<String, Object> model = setupModelForSendNotification(request);
        model.put("errors", new ErrorList()); // need an empty one so we don't have an NPE

        return new ModelAndView(view, model);
    }

    /**
     * Handles submitting the actual event notification message.
     */
    @RequestMapping(value="/SubmitEventNotificationMessage.form")
    public ModelAndView submitEventNotificationMessage(HttpServletRequest request) {
        String routeMessage = "This message was submitted via the event notification message submission form by user ";
        String viewName = "SendEventNotificationMessage";

        return submitNotificationMessage(request, routeMessage, viewName);
    }

    /**
     * {@inheritDoc}
     *
     * Populates values pertaining to an event notification message.
     */
    @Override
    protected Map<String, Object> setupModelForSendNotification(HttpServletRequest request) {
        Map<String, Object> model = super.setupModelForSendNotification(request);

        model.put("summary", request.getParameter("summary"));
        model.put("description", request.getParameter("description"));
        model.put("location", request.getParameter("location"));
        model.put("startDateTime", request.getParameter("startDateTime"));
        model.put("stopDateTime", request.getParameter("stopDateTime"));

        return model;
    }

    /**
     * {@inheritDoc}
     *
     * Overrides to set the content type to "event" and add extra attributes.
     */
    @Override
    protected NotificationBo createNotification(HttpServletRequest request, Map<String, Object> model,
            ErrorList errors) throws ErrorList {
        NotificationBo notification = super.createNotification(request, model, errors);

        String message = getParameter(request, "message", model, errors, "You must fill in a message.");

        String summary = getParameter(request, "summary", model, errors, "You must fill in a summary.");
        String description = getParameter(request, "description", model, errors, "You must fill in a description.");
        String location = getParameter(request, "location", model, errors, "You must fill in a location.");

        String startDateTime = request.getParameter("startDateTime");
        Date startDate = getDate(startDateTime, errors,
                "You specified an invalid start date and time.  Please use the calendar picker.");
        if (startDate != null) {
            model.put("startDateTime", startDateTime);
        }

        String stopDateTime = request.getParameter("stopDateTime");
        Date stopDate = getDate(stopDateTime, errors,
                "You specified an invalid start date and time.  Please use the calendar picker.");
        if (stopDate != null) {
            model.put("stopDateTime", stopDateTime);
        } else {

        }

        if (stopDate != null && startDate != null) {
            if (stopDate.before(startDate)) {
                errors.addError("Event Stop Date/Time cannot be before Event Start Date/Time.");
            }
        }

        // stop processing if there are errors
        if (!errors.getErrors().isEmpty()) {
            throw errors;
        }

        NotificationContentTypeBo contentType = Util.retrieveFieldReference("contentType", "name",
                NotificationConstants.CONTENT_TYPES.EVENT_CONTENT_TYPE, NotificationContentTypeBo.class,
                dataObjectService, Boolean.TRUE);
        notification.setContentType(contentType);

        notification.setContent(NotificationConstants.XML_MESSAGE_CONSTANTS.CONTENT_EVENT_OPEN
                + NotificationConstants.XML_MESSAGE_CONSTANTS.MESSAGE_OPEN
                + message
                + NotificationConstants.XML_MESSAGE_CONSTANTS.MESSAGE_CLOSE
                + "<event>\n"
                + "  <summary>"
                + summary
                + "</summary>\n"
                + "  <description>"
                + description
                + "</description>\n"
                + "  <location>"
                + location
                + "</location>\n"
                + "  <startDateTime>"
                + Util.toUIDateTimeString(startDate)
                + "</startDateTime>\n"
                + "  <stopDateTime>"
                + Util.toUIDateTimeString(stopDate)
                + "</stopDateTime>\n"
                + "</event>"
                + NotificationConstants.XML_MESSAGE_CONSTANTS.CONTENT_CLOSE);

        return notification;
    }
}
