/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.web.spring;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.ken.bo.NotificationBo;
import org.kuali.rice.ken.bo.NotificationContentTypeBo;
import org.kuali.rice.ken.exception.ErrorList;
import org.kuali.rice.ken.util.NotificationConstants;
import org.kuali.rice.ken.util.Util;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;

/**
 * This class is the controller for sending Simple notification messages via an end user interface.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@RequestMapping(value="/ken")
@Controller("sendNotificationMessageController")
public class SendNotificationMessageController extends BaseSendNotificationController {

    private static final Logger LOG = LogManager.getLogger(SendNotificationMessageController.class);

    /**
     * Handles the display of the form for sending a simple notification message.
     */
    @RequestMapping(value="/SendSimpleNotificationMessage.form")
    public ModelAndView sendSimpleNotificationMessage(HttpServletRequest request) {
        String view = "SendSimpleNotificationMessage";

        LOG.debug("remoteUser: " + request.getRemoteUser());

        Map<String, Object> model = setupModelForSendNotification(request);
        model.put("errors", new ErrorList()); // need an empty one so we don't have an NPE

        return new ModelAndView(view, model);
    }

    /**
     * Handles submitting the actual simple notification message.
     */
    @RequestMapping(value="/SubmitSimpleNotificationMessage.form")
    public ModelAndView submitSimpleNotificationMessage(HttpServletRequest request) {
        String routeMessage = "This message was submitted via the simple notification message submission form by user ";
        String viewName = "SendSimpleNotificationMessage";

        return submitNotificationMessage(request, routeMessage, viewName);
    }

    /**
     * {@inheritDoc}
     *
     * Overrides to set the content type to "simple".
     */
    @Override
    protected NotificationBo createNotification(HttpServletRequest request, Map<String, Object> model,
            ErrorList errors) throws ErrorList {
        NotificationBo notification = super.createNotification(request, model, errors);

        String message = getParameter(request, "message", model, errors, "You must fill in a message.");

        // stop processing if there are errors
        if (!errors.getErrors().isEmpty()) {
            throw errors;
        }

        NotificationContentTypeBo contentType = Util.retrieveFieldReference("contentType", "name",
                NotificationConstants.CONTENT_TYPES.SIMPLE_CONTENT_TYPE, NotificationContentTypeBo.class,
                dataObjectService, Boolean.TRUE);
        notification.setContentType(contentType);

        notification.setContent(NotificationConstants.XML_MESSAGE_CONSTANTS.CONTENT_SIMPLE_OPEN
                + NotificationConstants.XML_MESSAGE_CONSTANTS.MESSAGE_OPEN
                + message
                + NotificationConstants.XML_MESSAGE_CONSTANTS.MESSAGE_CLOSE
                + NotificationConstants.XML_MESSAGE_CONSTANTS.CONTENT_CLOSE);

        return notification;
    }
}
