/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.doctype.service.impl;

import org.kuali.rice.kew.api.doctype.DocumentType;
import org.kuali.rice.kew.api.document.Document;
import org.kuali.rice.kew.framework.document.security.AuthorizableAction;
import org.kuali.rice.kew.framework.document.security.Authorization;
import org.kuali.rice.kew.framework.document.security.DocumentTypeAuthorizer;
import org.kuali.rice.kew.routeheader.DocumentRouteHeaderValue;

import java.util.Collection;
import java.util.Map;

/**
 * Default DocumentTypeAuthorizer implementation which performs KIM checks for authorizable actions.
 * @since 2.1.3
 */
public class KimDocumentTypeAuthorizer extends DocumentActionsPermissionBase implements DocumentTypeAuthorizer {

    @Override
    public Authorization isActionAuthorized(AuthorizableAction action, String principalId, DocumentType documentType, Document document, Map<ActionArgument, Object> actionParameters) {
        org.kuali.rice.kew.doctype.bo.DocumentType documentTypeBo = org.kuali.rice.kew.doctype.bo.DocumentType.from(documentType);
        boolean success = false;
        switch (action.type) {
            case INITIATION:
                success = canInitiate(principalId, documentTypeBo);
                break;
            case SU_APPROVE_ACTION_REQUEST:
                success = canSuperUserApproveSingleActionRequest(principalId, documentTypeBo, (Collection<String>) actionParameters.get(ActionArgument.ROUTENODE_NAMES), (String) actionParameters.get(ActionArgument.DOCSTATUS));
                break;
            case ACTION:
                switch (action.actionType) {
                    case BLANKET_APPROVE:
                        success = canBlanketApprove(principalId, DocumentRouteHeaderValue.from(document));
                        break;
                    case SU_APPROVE:
                        success = canSuperUserApproveDocument(principalId, documentTypeBo, (Collection<String>) actionParameters.get(ActionArgument.ROUTENODE_NAMES), (String) actionParameters.get(ActionArgument.DOCSTATUS));
                        break;
                    case SU_DISAPPROVE:
                        success = canSuperUserDisapproveDocument(principalId, documentTypeBo, (Collection<String>) actionParameters.get(ActionArgument.ROUTENODE_NAMES), (String) actionParameters.get(ActionArgument.DOCSTATUS));
                        break;
                    case CANCEL:
                        success = canCancel(principalId, DocumentRouteHeaderValue.from(document));
                        break;
                    case RECALL:
                        success = canRecall(principalId, DocumentRouteHeaderValue.from(document));
                        break;
                    case ROUTE :
                        success = canRoute(principalId, DocumentRouteHeaderValue.from(document));
                        break;
                    case SAVE:
                        success = canSave(principalId, DocumentRouteHeaderValue.from(document));
                        break;
                    case RETURN_TO_PREVIOUS:
                        success = canReturnToPreviousRouteNode(principalId, DocumentRouteHeaderValue.from(document));
                        break;
                    default:
                        throw new RuntimeException("Unknown document action check");
                }
                break;
            default:
                throw new RuntimeException("Unknown authorization check");
        }
        return new Authorization(success);
    }

}
