/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.impl.action;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.exception.RiceIllegalArgumentException;
import org.kuali.rice.kew.actions.BlanketApproveAction;
import org.kuali.rice.kew.actiontaken.ActionTakenValue;
import org.kuali.rice.kew.api.KewApiServiceLocator;
import org.kuali.rice.kew.api.WorkflowRuntimeException;
import org.kuali.rice.kew.api.document.DocumentOrchestrationQueue;
import org.kuali.rice.kew.api.document.DocumentProcessingOptions;
import org.kuali.rice.kew.api.document.OrchestrationConfig;
import org.kuali.rice.kew.api.document.attribute.DocumentAttributeIndexingQueue;
import org.kuali.rice.kew.routeheader.DocumentRouteHeaderValue;
import org.kuali.rice.kew.service.KEWServiceLocator;
import org.kuali.rice.kim.api.identity.principal.Principal;
import org.kuali.rice.ksb.api.messaging.AsyncCapableService;
import org.kuali.rice.ksb.api.messaging.AsyncWrappable;

import javax.xml.namespace.QName;

/**
 * References implementation of the {@code DocumentOrchestrationQueue}.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class DocumentOrchestrationQueueImpl implements DocumentOrchestrationQueue, AsyncWrappable<DocumentOrchestrationQueue> {
	
	private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(DocumentOrchestrationQueueImpl.class);

    private AsyncCapableService asyncCapableService;

    @Override
    public void orchestrateDocument(String documentId, String principalId, OrchestrationConfig orchestrationConfig,
            DocumentProcessingOptions documentProcessingOptions) {
        if (StringUtils.isBlank(principalId)) {
            throw new RiceIllegalArgumentException("principalId is null or blank");
        }

        if (StringUtils.isBlank(documentId)) {
            throw new RiceIllegalArgumentException("documentId is null");
        }

        if (orchestrationConfig == null) {
            throw new RiceIllegalArgumentException("orchestrationConfig is null");
        }
        if (documentProcessingOptions == null) {
            documentProcessingOptions = DocumentProcessingOptions.createDefault();
        }

        LOG.info("Performing document orchestration on documentId=" + documentId);
        KEWServiceLocator.getRouteHeaderService().lockRouteHeader(documentId);
        DocumentRouteHeaderValue document = KEWServiceLocator.getRouteHeaderService().getRouteHeader(documentId);
		ActionTakenValue actionTaken = KEWServiceLocator.getActionTakenService().findByActionTakenId(orchestrationConfig.getActionTakenId());
		Principal principal = KEWServiceLocator.getIdentityHelperService().getPrincipal(principalId);
		BlanketApproveAction blanketApprove = new BlanketApproveAction(document, principal, "", orchestrationConfig.getNodeNames());
		try {
			blanketApprove.performDeferredBlanketApproveWork(actionTaken, documentProcessingOptions);
		} catch (Exception e) {
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
			throw new WorkflowRuntimeException(e);
		}
		if (documentProcessingOptions.isIndexSearchAttributes()) {
            DocumentAttributeIndexingQueue queue = KewApiServiceLocator.getDocumentAttributeIndexingQueue(document.getDocumentType().getApplicationId());
            queue.indexDocument(documentId);
		}
        LOG.info("Document orchestration complete against documentId=" + documentId);
    }

    @Override
    public DocumentOrchestrationQueue wrap(QName qname, String applicationId) {
        return new DocumentOrchestrationQueueAsyncCapableImpl(asyncCapableService,this, qname, applicationId);
    }

    public AsyncCapableService getAsyncCapableService() {
        return asyncCapableService;
    }

    public void setAsyncCapableService(AsyncCapableService asyncCapableService) {
        this.asyncCapableService = asyncCapableService;
    }
}
