/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.web.spring;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.ken.service.NotificationAuthorizationService;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

/**
 * Spring HandlerInterceptor implementation that implements security.  For now this just
 * adds a flag to the request indicating whether the authenticated user is a Notification
 * System administrator.
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class SecurityInterceptor extends HandlerInterceptorAdapter {
    private static final Logger LOG = LogManager.getLogger(SecurityInterceptor.class);

    /**
     * Request attribute key under which to register the userIsAdmin flag
     */
    private static final String USER_IS_ADMIN_KEY = "userIsAdmin";

    protected NotificationAuthorizationService notificationAuthzService;
    
    /**
     * Sets the NotificationAuthorizationService member
     * @param notificationAuthzService NotificationAuthorizationService used to determine whether user is administrator
     */
    public void setNotificationAuthorizationService(NotificationAuthorizationService notificationAuthzService) {
        this.notificationAuthzService = notificationAuthzService;
    }

    /**
     * Decorate the incoming request with an attribute that indicates whether the user is a Notification System administrator
     * @see org.springframework.web.servlet.handler.HandlerInterceptorAdapter#preHandle(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, java.lang.Object)
     */
    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        String user = request.getRemoteUser();
        boolean isAdmin = false;
        if (user != null) {
            isAdmin = notificationAuthzService.isUserAdministrator(user);
        }
        LOG.debug("Setting request attribute '" + USER_IS_ADMIN_KEY + "' to " + isAdmin);
        request.setAttribute(USER_IS_ADMIN_KEY, Boolean.valueOf(isAdmin));
        return true;
    }
}
