/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.engine.node.service.impl;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.kew.engine.node.Branch;
import org.kuali.rice.kew.engine.node.BranchState;
import org.kuali.rice.kew.engine.node.service.BranchService;
import org.kuali.rice.krad.data.DataObjectService;
import org.springframework.beans.factory.annotation.Required;

import java.util.List;

public class BranchServiceImpl implements BranchService {
    private static final Logger LOG = LogManager.getLogger(BranchServiceImpl.class);

    private DataObjectService dataObjectService;
    
    @Override
    public void save(Branch branch){
        dataObjectService.save(branch);
    }
    
    @Override
    public void deleteBranchStates(List statesToBeDeleted){
        for(BranchState bs : (List<BranchState>)statesToBeDeleted){
            dataObjectService.delete(bs);
        }
    }

    /**
     * Walks up the Branch/scope hierarchy trying to find a variable with the specified name
     * @param branch the lowermost branch at which to start the search
     * @param name name of the variable to search for
     * @return a BranchState object in the first Branch/scope in which the variable was found
     */
    private BranchState resolveScopedVariable(Branch branch, String name) {
        Branch b = branch;
        while (b != null) {
            for (BranchState bs: b.getBranchState()) {
                LOG.debug(bs);
            }
            LOG.debug("Resolving variable: '" + name + "' in scope (branch): '" + branch.getName() + "' (" + branch.getBranchId() + ")");
            BranchState bs = b.getBranchState(name);
            if (bs != null) {
                return bs;    
            }
            b = b.getParentBranch();
        }
        return null;
    }

    @Override
    public String getScopedVariableValue(Branch branch, String name) {
        BranchState bs = resolveScopedVariable(branch, name);
        if (bs != null) return bs.getValue();
        return null;
    }

    @Override
    public String setScopedVariableValue(Branch branch, String name, String value) {
        LOG.debug("Setting scoped variable value: " + name + " " + value);
        BranchState bs = resolveScopedVariable(branch, name);
        String oldValue = null;
        if (bs == null) {
            LOG.debug("Defining new variable named '" + name + "' at scope '" + branch + "'");
            // create new variable at initial search scope
            bs = new BranchState();
            bs.setKey(name);
            bs.setValue(value);
            bs.setBranch(branch);
            branch.addBranchState(bs);
        } else {
            oldValue = bs.getValue();
            LOG.debug("Replacing old value of variable '" + name + "' (" + oldValue + ") at scope '" + branch + "' with new value: " + value);
            bs.setValue(value);
        }
        // now save the Branch whose state we just modified
        save(bs.getBranch());
        return oldValue;
    }


    public DataObjectService getDataObjectService() {
        return dataObjectService;
    }

    @Required
    public void setDataObjectService(DataObjectService dataObjectService) {
        this.dataObjectService = dataObjectService;
    }
    
}
