/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.engine.transition;

import org.kuali.rice.kew.engine.RouteContext;
import org.kuali.rice.kew.engine.RouteHelper;
import org.kuali.rice.kew.engine.node.Node;
import org.kuali.rice.kew.engine.node.ProcessResult;
import org.kuali.rice.kew.engine.node.RouteNode;
import org.kuali.rice.kew.engine.node.RouteNodeInstance;

import java.util.ArrayList;
import java.util.List;


/**
 * Common superclass for all Transition Engines.  A TransitionEngine handles transitioning into and out of
 * a {@link RouteNodeInstance}.  The TransitionEngine is also responsible for determining if a Node has completed.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public abstract class TransitionEngine {
    
	private RouteHelper helper;
	
	public RouteNodeInstance transitionTo(RouteNodeInstance nextNodeInstance, RouteContext context) throws Exception {
		return nextNodeInstance;
	}
    
    /**
     * Tell the WorkflowEngine processing the activeNodeInstance if the node is complete and transitionFrom can 
     * be called.
     *
     * @return boolean
     * @param context for routing
     * @throws Exception
     */
    public abstract ProcessResult isComplete(RouteContext context) throws Exception;
	
    public Transition transitionFrom(RouteContext context, ProcessResult processResult) throws Exception {
        return new Transition(resolveNextNodeInstances(context.getNodeInstance()));
    }
    
    protected void setRouteHelper(RouteHelper helper) {
    	this.helper = helper;
    }
    
    protected RouteHelper getRouteHelper() {
    	return helper;
    }
    
    protected Node getNode(RouteNode routeNode, Class nodeClass) throws Exception {
		return helper.getNode(routeNode);
    }
    
    /**
     * Determines the next nodes instances for the transition.  If the node instance already
     * has next nodes instances (i.e. a dynamic node), then those will be returned.  Otherwise
     * it will resolve the next nodes from the RouteNode prototype.
     * @param nodeInstance for the transition
     * @param nextRouteNodes list of route notes
     * @return list of route note instances
     */
    protected List<RouteNodeInstance> resolveNextNodeInstances(RouteNodeInstance nodeInstance, List<RouteNode> nextRouteNodes) {
        List<RouteNodeInstance> nextNodeInstances = new ArrayList<RouteNodeInstance>();
        for (RouteNode nextRouteNode : nextRouteNodes) {
            RouteNodeInstance nextNodeInstance = getRouteHelper().getNodeFactory().createRouteNodeInstance(nodeInstance.getDocumentId(), nextRouteNode);
            nextNodeInstance.setBranch(nodeInstance.getBranch());
            nextNodeInstance.setProcess(nodeInstance.getProcess());
            nextNodeInstances.add(nextNodeInstance);
        }
        return nextNodeInstances;
    }
    
    protected List<RouteNodeInstance> resolveNextNodeInstances(RouteNodeInstance nodeInstance) {
        return resolveNextNodeInstances(nodeInstance, nodeInstance.getRouteNode().getNextNodes());
    }
    
}
