/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.routeheader.service.impl;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.kuali.rice.kew.api.KewApiConstants;
import org.kuali.rice.kew.api.action.ActionItem;
import org.kuali.rice.kew.docsearch.SearchableAttributeValue;
import org.kuali.rice.kew.docsearch.dao.SearchableAttributeDAO;
import org.kuali.rice.kew.doctype.bo.DocumentType;
import org.kuali.rice.kew.exception.WorkflowServiceErrorException;
import org.kuali.rice.kew.exception.WorkflowServiceErrorImpl;
import org.kuali.rice.kew.routeheader.DocumentRouteHeaderValue;
import org.kuali.rice.kew.routeheader.DocumentRouteHeaderValueContent;
import org.kuali.rice.kew.routeheader.dao.DocumentRouteHeaderDAO;
import org.kuali.rice.kew.routeheader.service.RouteHeaderService;
import org.kuali.rice.kew.service.KEWServiceLocator;
import org.kuali.rice.kim.api.identity.principal.Principal;
import org.kuali.rice.kim.api.services.KimApiServiceLocator;
import org.kuali.rice.krad.data.DataObjectService;
import org.kuali.rice.krad.data.PersistenceOption;
import org.springframework.beans.factory.annotation.Required;

import javax.persistence.OptimisticLockException;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.*;

public class RouteHeaderServiceImpl implements RouteHeaderService {

    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(RouteHeaderServiceImpl.class);

    private DocumentRouteHeaderDAO routeHeaderDAO;
    private SearchableAttributeDAO searchableAttributeDAO;

    private DataObjectService dataObjectService;

    @Override
    public DocumentRouteHeaderValue getRouteHeader(String documentId) {
        return getDataObjectService().find(DocumentRouteHeaderValue.class, documentId);
    }

    @Override
    public DocumentRouteHeaderValue getRouteHeader(String documentId, boolean clearCache) {
    	return getRouteHeaderDAO().findRouteHeader(documentId, clearCache);
    }

    @Override
    public Collection<DocumentRouteHeaderValue> getRouteHeaders(Collection<String> documentIds) {
         return getRouteHeaderDAO().findRouteHeaders(documentIds);
    }
    
    @Override
    public Collection<DocumentRouteHeaderValue> getRouteHeaders(Collection<String> documentIds, boolean clearCache) {
    	return getRouteHeaderDAO().findRouteHeaders(documentIds, clearCache);
    }
    
    @Override
    public Map<String,DocumentRouteHeaderValue> getRouteHeadersForActionItems(Collection<ActionItem> actionItems) {
    	Map<String,DocumentRouteHeaderValue> routeHeaders = new HashMap<>();
    	List<String> documentIds = new ArrayList<>(actionItems.size());
    	for (ActionItem actionItem : actionItems) {
    		documentIds.add(actionItem.getDocumentId());
    	}
    	Collection<DocumentRouteHeaderValue> actionItemRouteHeaders = getRouteHeaders(documentIds);
    	if (actionItemRouteHeaders != null) {
    		for (DocumentRouteHeaderValue routeHeader : actionItemRouteHeaders) {
    			routeHeaders.put(routeHeader.getDocumentId(), routeHeader);
    		}
    	}
    	return routeHeaders;
    }
    
    @Override
    public void lockRouteHeader(String documentId) {
        getRouteHeaderDAO().lockRouteHeader(documentId);
        LOG.debug("Successfully locked document [docId=" + documentId + "]");
    }

    @Override
    public DocumentRouteHeaderValue saveRouteHeader(DocumentRouteHeaderValue routeHeader) {
        if ( LOG.isDebugEnabled() ) {
            LOG.debug( "About to Save the route Header: " + routeHeader.getDocumentId() + " / version="
                    + routeHeader.getVersionNumber() );
            if (StringUtils.isNotBlank(routeHeader.getDocumentId())) {
                DocumentRouteHeaderValue currHeader = getDataObjectService().find(DocumentRouteHeaderValue.class,
                        routeHeader.getDocumentId());
                if ( currHeader != null ) {
                    LOG.debug( "Current Header Version: " + currHeader.getVersionNumber() );
                }
            }
            else {
                LOG.debug( "Current Header: null" );
            }
            LOG.debug( ExceptionUtils.getStackTrace(new Throwable()) );
        }
        try {
            // before saving, copy off the document content, since it's transient it will get erased during a JPA merge
            DocumentRouteHeaderValueContent content = routeHeader.getDocumentContent();
            DocumentRouteHeaderValue drvPersisted = dataObjectService.save(routeHeader, PersistenceOption.FLUSH);
            // now let's save the content and reattach it to our document
            content.setDocumentId(drvPersisted.getDocumentId());
            content = dataObjectService.save(content);
            drvPersisted.setDocumentContent(content);
            return drvPersisted;
        } catch ( RuntimeException ex ) {
            if ( ex.getCause() instanceof OptimisticLockException) {
                LOG.error( "Optimistic Locking Exception saving document header or content. Offending object: "
                        + ex.getCause()
                        + "; DocumentId = " + routeHeader.getDocumentId() + " ;  Version Number = "
                        + routeHeader.getVersionNumber());
            }
            LOG.error( "Unable to save document header or content. Route Header: " + routeHeader, ex );
            throw ex;
        }
    }

    @Override
    public void deleteRouteHeader(DocumentRouteHeaderValue routeHeader) {
        dataObjectService.delete(routeHeader);
    }

    @Override
    public String getNextDocumentId() {
        return getRouteHeaderDAO().getNextDocumentId();
    }

    @Override
    public Collection findPendingByResponsibilityIds(Set responsibilityIds) {
        return getRouteHeaderDAO().findPendingByResponsibilityIds(responsibilityIds);
    }

    @Override
    public void clearRouteHeaderSearchValues(String documentId) {
        getRouteHeaderDAO().clearRouteHeaderSearchValues(documentId);
    }
    
    @Override
    public void updateRouteHeaderSearchValues(String documentId, List<SearchableAttributeValue> searchAttributes) {
        getRouteHeaderDAO().clearRouteHeaderSearchValues(documentId);
        HashSet<String> dupedSet = new HashSet<>();
        //"de-dupe" for value,key,and doc header id
        for (SearchableAttributeValue searchAttribute : searchAttributes) {
            if(searchAttribute != null){
                String fakeKey = searchAttribute.getSearchableAttributeKey() + "-" + searchAttribute.getSearchableAttributeValue();
                if(!dupedSet.contains(fakeKey)){
                    getRouteHeaderDAO().save(searchAttribute);
                    dupedSet.add(fakeKey);
                }
            }
        }
        LOG.warn("Deduplication adjusted incoming SearchableAttributeValue list from original: " + searchAttributes.size() + " entries into : "  + (searchAttributes.size() - dupedSet.size()) + " entries.");
    }

    @Override
    public void validateRouteHeader(DocumentRouteHeaderValue routeHeader){
        LOG.debug("Enter validateRouteHeader(..)");
        List errors = new ArrayList();

        if (routeHeader.getDocRouteStatus() == null || routeHeader.getDocRouteStatus().trim().equals("")) {
            errors.add(new WorkflowServiceErrorImpl("RouteHeader route status null.", "routeheader.routestatus.empty"));
        } else if (!KewApiConstants.DOCUMENT_STATUSES.containsKey(routeHeader.getDocRouteStatus())){
            errors.add(new WorkflowServiceErrorImpl("RouteHeader route status invalid.", "routeheader.routestatus.invalid"));
        }

        if(routeHeader.getDocRouteLevel() == null || routeHeader.getDocRouteLevel().intValue() < 0){
            errors.add(new WorkflowServiceErrorImpl("RouteHeader route level invalid.", "routeheader.routelevel.invalid"));
        }

        if(routeHeader.getDateLastModified() == null){
            errors.add(new WorkflowServiceErrorImpl("RouteHeader status modification date empty.", "routeheader.statusmoddate.empty"));
        }

        if(routeHeader.getCreateDate() == null){
            errors.add(new WorkflowServiceErrorImpl("RouteHeader status create date empty.", "routeheader.createdate.empty"));
        }
        if(routeHeader.getDocVersion() == null || routeHeader.getDocVersion().intValue() < 0){
            errors.add(new WorkflowServiceErrorImpl("RouteHeader doc version invalid.", "routeheader.docversion.invalid"));
        }

        if (routeHeader.getInitiatorWorkflowId () == null || routeHeader.getInitiatorWorkflowId().trim().equals("")) {
            errors.add(new WorkflowServiceErrorImpl("RouteHeader initiator null.", "routeheader.initiator.empty"));
        }
        else
        {
           	Principal principal = KimApiServiceLocator.getIdentityService().getPrincipal(routeHeader.getInitiatorWorkflowId());
            if(principal == null)
            {
               	errors.add(new WorkflowServiceErrorImpl("RouteHeader initiator id invalid.", "routeheader.initiator.invalid"));
            }
        }

        if(!StringUtils.isBlank(routeHeader.getDocumentTypeId())){
            DocumentType docType = KEWServiceLocator.getDocumentTypeService().findById(routeHeader.getDocumentTypeId());
            if(docType == null){
                errors.add(new WorkflowServiceErrorImpl("RouteHeader document type id invalid.", "routeheader.doctypeid.invalid"));
            }
        }

        LOG.debug("Exit validateRouteHeader(..) ");
        if (!errors.isEmpty()) {
            throw new WorkflowServiceErrorException("RouteHeader Validation Error", errors);
        }
    }

    @Override
    public String getApplicationIdByDocumentId(String documentId) {
    	return getRouteHeaderDAO().getApplicationIdByDocumentId(documentId);
    }

    @Override
    public DocumentRouteHeaderValueContent getContent(String documentId) {
    	if (documentId == null) {
    		return new DocumentRouteHeaderValueContent();
    	}
        DocumentRouteHeaderValueContent content = getRouteHeaderDAO().getContent(documentId);
    	if (content == null) {
    		content = new DocumentRouteHeaderValueContent(documentId);
    	}
    	return content;
    }

    @Override
    public boolean hasSearchableAttributeValue(String documentId, String searchableAttributeKey, String searchableAttributeValue) {
	return getRouteHeaderDAO().hasSearchableAttributeValue(documentId, searchableAttributeKey, searchableAttributeValue);
    }

    @Override
    public String getDocumentStatus(String documentId) {
    	return getRouteHeaderDAO().getDocumentStatus(documentId);
    }
    
    @Override
    public String getAppDocId(String documentId) {
 	 	if (documentId == null) {
 	 		return null;
 	 	}
 	 	return getRouteHeaderDAO().getAppDocId(documentId);
    }

    @Override
    public String getAppDocStatus(String documentId) {
        if (documentId == null) {
            return null;
        }
        return getRouteHeaderDAO().getAppDocStatus(documentId);
    }

    public DocumentRouteHeaderDAO getRouteHeaderDAO() {
        return routeHeaderDAO;
    }

    public void setRouteHeaderDAO(DocumentRouteHeaderDAO routeHeaderDAO) {
        this.routeHeaderDAO = routeHeaderDAO;
    }

	@Override
    public List<Timestamp> getSearchableAttributeDateTimeValuesByKey(
			String documentId, String key) {
		return getSearchableAttributeDAO().getSearchableAttributeDateTimeValuesByKey(documentId, key);
	}

	@Override
    public List<BigDecimal> getSearchableAttributeFloatValuesByKey(
			String documentId, String key) {
		return getSearchableAttributeDAO().getSearchableAttributeFloatValuesByKey(documentId, key);
	}

	@Override
    public List<Long> getSearchableAttributeLongValuesByKey(String documentId,
                                                            String key) {
		return getSearchableAttributeDAO().getSearchableAttributeLongValuesByKey(documentId, key);
	}

	@Override
    public List<String> getSearchableAttributeStringValuesByKey(
			String documentId, String key) {

		return getSearchableAttributeDAO().getSearchableAttributeStringValuesByKey(documentId, key);
	}

	public void setSearchableAttributeDAO(SearchableAttributeDAO searchableAttributeDAO) {
		this.searchableAttributeDAO = searchableAttributeDAO;
	}

	public SearchableAttributeDAO getSearchableAttributeDAO() {
		return searchableAttributeDAO;
	}

	@Override
    public Collection findByDocTypeAndAppId(String documentTypeName,
                                            String appId) {
		return getRouteHeaderDAO().findByDocTypeAndAppId(documentTypeName, appId);
	}

    public DataObjectService getDataObjectService(){
        return dataObjectService;
    }

    @Required
    public void setDataObjectService(DataObjectService dataObjectService) {
        this.dataObjectService = dataObjectService;
    }
}
