/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.rule.xmlrouting;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathFactory;
import javax.xml.xpath.XPathFunctionResolver;

import org.kuali.rice.kew.api.WorkflowRuntimeException;
import org.w3c.dom.Node;


/**
 * Provides utilities for obtaining XPath instances which are "good-to-go" with access to the Workflow
 * namespace and custom XPath functions.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class XPathHelper {

	/**
	 * Creates a new XPath instance and initializes it with the WorkflowNamespaceContext and the
	 * WorkflowFunctionResolver.
	 */
	public static XPath newXPath() {
		XPath xPath = XPathFactory.newInstance().newXPath();
		xPath.setNamespaceContext(new WorkflowNamespaceContext());
		WorkflowFunctionResolver resolver = new WorkflowFunctionResolver();
		xPath.setXPathFunctionResolver(resolver); 
		resolver.setXpath(xPath);
		return xPath;
	}

	/**
	 * Creates a new XPath instances and initializes it with the WorkflowNamespaceContext and the
	 * WorkflowFunctionResolver.  Also sets the root node on the WorkflowFunctionResolver to 
	 * the given Node.  This is required for some of the functions in the function resolver
	 * to perform properly.
	 */
	public static XPath newXPath(Node rootNode) {
		XPath xPath = newXPath();
		WorkflowFunctionResolver resolver = extractFunctionResolver(xPath);
		resolver.setRootNode(rootNode);
		return xPath;
	}
	
	/**
	 * A utility to extract the WorkflowFunctionResolver from the given XPath instances.  If the XPath instance
	 * does not contain a WorkflowFunctionResolver, then this method will throw a WorkflowRuntimeException.
	 * 
	 * @throws WorkflowRuntimeException if the given XPath instance does not contain a WorklflowFunctionResolver
	 */
	public static WorkflowFunctionResolver extractFunctionResolver(XPath xPath) {
		XPathFunctionResolver resolver = xPath.getXPathFunctionResolver();
		if (!hasWorkflowFunctionResolver(xPath)) {
			throw new WorkflowRuntimeException("The XPathFunctionResolver on the given XPath instance is not an instance of WorkflowFunctionResolver, was: " + resolver);
		}
		return (WorkflowFunctionResolver)resolver;
	}
	
	/**
	 * Returns true if the given XPath instance has a WorkflowFunctionResolver, false otherwise.
	 */
	public static boolean hasWorkflowFunctionResolver(XPath xPath) {
		return xPath.getXPathFunctionResolver() instanceof WorkflowFunctionResolver;
	}
	
}
