/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.xml.export;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.jdom2.Element;
import org.jdom2.Namespace;
import org.kuali.rice.core.api.impex.ExportDataSet;
import org.kuali.rice.core.api.util.xml.XmlRenderer;
import org.kuali.rice.core.framework.impex.xml.XmlExporter;
import org.kuali.rice.kew.export.KewExportDataSet;
import org.kuali.rice.kew.rule.RuleBaseValues;
import org.kuali.rice.kew.rule.RuleExtensionBo;
import org.kuali.rice.kew.rule.RuleExtensionValue;
import org.kuali.rice.kew.rule.RuleResponsibilityBo;
import org.kuali.rice.kew.rule.bo.RuleTemplateAttributeBo;
import org.kuali.rice.kew.rule.web.WebRuleUtils;
import org.kuali.rice.kim.api.group.Group;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import static org.kuali.rice.core.api.impex.xml.XmlConstants.*;

/**
 * Exports rules to XML.
 *
 * @see RuleBaseValues
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class RuleXmlExporter implements XmlExporter {

    protected final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(getClass());

    private XmlRenderer renderer;
    
    public RuleXmlExporter(Namespace namespace) {
    	this.renderer = new XmlRenderer(namespace);
    }
    
	@Override
	public boolean supportPrettyPrint() {
		return true;
	}

    @Override
    public Element export(ExportDataSet exportDataSet) {
    	KewExportDataSet dataSet = KewExportDataSet.fromExportDataSet(exportDataSet);
        if (!dataSet.getRules().isEmpty()) {
            Element rootElement = renderer.renderElement(null, RULES);
            rootElement.setAttribute(SCHEMA_LOCATION_ATTR, RULE_SCHEMA_LOCATION, SCHEMA_NAMESPACE);
            for (Iterator iterator = dataSet.getRules().iterator(); iterator.hasNext();) {
            	RuleBaseValues rule = (RuleBaseValues) iterator.next();
            	exportRule(rootElement, rule);
            }
            return rootElement;
        }
        return null;
    }

    public void exportRule(Element parent, RuleBaseValues rule) {
    	Element ruleElement = renderer.renderElement(parent, RULE);
        if (rule.getName() != null) {
            renderer.renderTextElement(ruleElement, NAME, rule.getName());
        }
        renderer.renderTextElement(ruleElement, DOCUMENT_TYPE, rule.getDocTypeName());
        if (rule.getRuleTemplateName() != null) {
            renderer.renderTextElement(ruleElement, RULE_TEMPLATE, rule.getRuleTemplateName());
        }
        renderer.renderTextElement(ruleElement, DESCRIPTION, rule.getDescription());
        if(rule.getFromDateString() != null){
            renderer.renderTextElement(ruleElement, FROM_DATE, rule.getFromDateString());
        }
        if(rule.getToDateString() != null){
            renderer.renderTextElement(ruleElement, TO_DATE, rule.getToDateString());
        }
        if (rule.getRuleExpressionDef() != null) {
            Element expressionElement = renderer.renderTextElement(ruleElement, EXPRESSION, rule.getRuleExpressionDef().getExpression());
            if (rule.getRuleExpressionDef().getType() != null) {
                expressionElement.setAttribute("type", rule.getRuleExpressionDef().getType());
            }
        }
        renderer.renderBooleanElement(ruleElement, FORCE_ACTION, rule.isForceAction(), false);
        
        if (CollectionUtils.isEmpty(rule.getRuleExtensions()) &&
        		/* field values is not empty */
        		!(rule.getFieldValues() == null || rule.getFieldValues().size() == 0)) {
        	// the rule is in the wrong state (as far as we are concerned).
        	// translate it
        	WebRuleUtils.translateResponsibilitiesForSave(rule);
        	WebRuleUtils.translateFieldValuesForSave(rule);
        	
        	// do our exports
    		exportRuleExtensions(ruleElement, rule.getRuleExtensions());
        	
        	// translate it back
        	WebRuleUtils.populateRuleMaintenanceFields(rule);
        } else { 
        	exportRuleExtensions(ruleElement, rule.getRuleExtensions());
        }
        
        // put responsibilities in a single collection 
        Set<RuleResponsibilityBo> responsibilities = new HashSet<RuleResponsibilityBo>();
        responsibilities.addAll(rule.getRuleResponsibilities());
        responsibilities.addAll(rule.getPersonResponsibilities());
        responsibilities.addAll(rule.getGroupResponsibilities());
        responsibilities.addAll(rule.getRoleResponsibilities());
        
        exportResponsibilities(ruleElement, responsibilities);
    }

    private void exportRuleExtensions(Element parent, List ruleExtensions) {
        if (!ruleExtensions.isEmpty()) {
            Element extsElement = renderer.renderElement(parent, RULE_EXTENSIONS);
            for (Iterator iterator = ruleExtensions.iterator(); iterator.hasNext();) {
                RuleExtensionBo extension = (RuleExtensionBo) iterator.next();
                Element extElement = renderer.renderElement(extsElement, RULE_EXTENSION);
                RuleTemplateAttributeBo attribute = extension.getRuleTemplateAttribute();
                renderer.renderTextElement(extElement, ATTRIBUTE, attribute.getRuleAttribute().getName());
                renderer.renderTextElement(extElement, RULE_TEMPLATE, attribute.getRuleTemplate().getName());
                exportRuleExtensionValues(extElement, extension.getExtensionValues());
            }
        }
    }

    // sorts by rule extension value key in order to establish a deterministic order
    private void exportRuleExtensionValues(Element parent, List<RuleExtensionValue> extensionValues) {
        if (!extensionValues.isEmpty()) {
            List<RuleExtensionValue> sorted = new ArrayList<RuleExtensionValue>(extensionValues);
            // establish deterministic ordering of keys
            Collections.sort(sorted, new Comparator<RuleExtensionValue>() {
                @Override
                public int compare(RuleExtensionValue o1, RuleExtensionValue o2) {
                    if (o1 == null) return -1;
                    if (o2 == null) return 1;
                    return ObjectUtils.compare(o1.getKey(), o2.getKey());
                }
            });
            Element extValuesElement = renderer.renderElement(parent, RULE_EXTENSION_VALUES);
            for (Iterator iterator = sorted.iterator(); iterator.hasNext();) {
                RuleExtensionValue extensionValue = (RuleExtensionValue) iterator.next();
                Element extValueElement = renderer.renderElement(extValuesElement, RULE_EXTENSION_VALUE);
                renderer.renderTextElement(extValueElement, KEY, extensionValue.getKey());
                renderer.renderTextElement(extValueElement, VALUE, extensionValue.getValue());
            }
        }
    }

    private void exportResponsibilities(Element parent, Collection<? extends RuleResponsibilityBo> responsibilities) {
        if (responsibilities != null && !responsibilities.isEmpty()) {
            Element responsibilitiesElement = renderer.renderElement(parent, RESPONSIBILITIES);
            for (RuleResponsibilityBo ruleResponsibility : responsibilities) {
                Element respElement = renderer.renderElement(responsibilitiesElement, RESPONSIBILITY);
              //KULRICE-12282: commenting out the responsibility Id from the exported xml.
               // renderer.renderTextElement(respElement, RESPONSIBILITY_ID, "" + ruleResponsibility.getResponsibilityId());
                if (ruleResponsibility.isUsingPrincipal()) {
				    renderer.renderTextElement(respElement, PRINCIPAL_NAME, ruleResponsibility.getPrincipal().getPrincipalName());
				} else if (ruleResponsibility.isUsingGroup()) {
					Group group = ruleResponsibility.getGroup();
				    Element groupElement = renderer.renderTextElement(respElement, GROUP_NAME, group.getName());
				    groupElement.setAttribute(NAMESPACE, group.getNamespaceCode());
				} else if (ruleResponsibility.isUsingRole()) {
				    renderer.renderTextElement(respElement, ROLE, ruleResponsibility.getRuleResponsibilityName());
				    renderer.renderTextElement(respElement, APPROVE_POLICY, ruleResponsibility.getApprovePolicy());
				}
                if (!StringUtils.isBlank(ruleResponsibility.getActionRequestedCd())) {
                	renderer.renderTextElement(respElement, ACTION_REQUESTED, ruleResponsibility.getActionRequestedCd());
                }
                if (ruleResponsibility.getPriority() != null) {
                	renderer.renderTextElement(respElement, PRIORITY, ruleResponsibility.getPriority().toString());
                }
            }
        }
    }
}
