/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.api.exception;

import org.kuali.rice.core.api.util.xml.XmlException;

/**
 * This error is thrown whenever a child document type is trying to be processed before its 
 * parent document type has been parsed; this provides a means for delaying the processing
 * of child doc types until their parents are parsed.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public class InvalidParentDocTypeException extends XmlException {
	
	/** The name of the parent document that still needs to be parsed. */
	private final String parentName;
	/** The name of the child document that was expecting the parentName document to exist. */
	private final String childName;
	
	/**
	 * Constructs an InvalidParentDocTypeException, given a document type parent name and a child name.
	 * 
	 * @param docParent The name of the unprocessed document type parent.
	 * @param docChild The name of the unprocessed document type child.
	 */
	public InvalidParentDocTypeException(String docParent, String docChild) {
		super("parent: " + docParent + " child: " + docChild);
		parentName = docParent;
		childName = docChild;
	}
	
	/**
	 * Constructs an InvalidParentDocTypeException, given a document type parent name, a child name, and an error message.
	 * 
	 * @param docParent The name of the unprocessed document type parent.
	 * @param docChild The name of the unprocessed document type child.
	 * @param message The error message.
	 */
	public InvalidParentDocTypeException(String docParent, String docChild, String message) {
		super(message);
		parentName = docParent;
		childName = docChild;
	}

	/**
	 * Constructs an InvalidParentDocTypeException, given a document type parent name, a child name, an error message, and a cause.
	 * 
	 * @param docParent The name of the unprocessed document type parent.
	 * @param docChild The name of the unprocessed document type child.
	 * @param message The error message.
	 * @param throwable The cause.
	 */
	public InvalidParentDocTypeException(String docParent, String docChild, String message, Throwable throwable) {
		super(message, throwable);
		parentName = docParent;
		childName = docChild;
	}

	/**
	 * Constructs an InvalidParentDocTypeException, given a document type parent name, a child name, and a cause.
	 * 
	 * @param docParent The name of the unprocessed document type parent.
	 * @param docChild The name of the unprocessed document type child.
	 * @param throwable The cause.
	 */
	public InvalidParentDocTypeException(String docParent, String docChild, Throwable throwable) {
		super(throwable);
		parentName = docParent;
		childName = docChild;
	}
	
	/**
	 * Retrieves the name of the parent document type that has not been processed yet.
	 * 
	 * @return The name of the unprocessed document type parent, which may or may not be null.
	 */
	public String getParentName() {
		return parentName;
	}
	
	/**
	 * Retrieves the name of the child document type that depends on the given parent.
	 * 
	 * @return The name of the unprocessed document type child, which may or may not be null.
	 */
	public String getChildName() {
		return childName;
	}
}
