/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kim.impl.group;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.collections.ListUtils;
import org.apache.commons.lang.StringUtils;
import org.kuali.rice.kew.api.KewApiConstants;
import org.kuali.rice.kew.api.KewApiServiceLocator;
import org.kuali.rice.kim.api.group.GroupService;
import org.kuali.rice.kim.api.services.KimApiServiceLocator;
import org.kuali.rice.krad.data.DataObjectService;
import org.kuali.rice.krad.data.PersistenceOption;
import org.kuali.rice.krad.service.KRADServiceLocator;

/**
 * Concrete Implementation of {@link GroupInternalService}
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public class GroupInternalServiceImpl implements GroupInternalService {

    protected DataObjectService getDataObjectService() {
        return KRADServiceLocator.getDataObjectService();
    }

    protected GroupService getGroupService(){
    	return KimApiServiceLocator.getGroupService();
    }

    @Override
    public GroupBo saveWorkgroup(GroupBo group) {
    	GroupService ims = getGroupService();
        List<String> oldIds = Collections.emptyList();
    	if (StringUtils.isNotEmpty(group.getId())) {
            oldIds = ims.getMemberPrincipalIds(group.getId());
        }
        group = getDataObjectService().save(group,PersistenceOption.FLUSH);
        List<String> newIds = ims.getMemberPrincipalIds(group.getId());
        updateForWorkgroupChange(group.getId(), oldIds, newIds);
        return group;
    }

    @Override
    public void updateForWorkgroupChange(String groupId,
    		List<String> oldPrincipalIds, List<String> newPrincipalIds) {
        MembersDiff membersDiff = getMembersDiff(oldPrincipalIds, newPrincipalIds);
        for (String removedPrincipalId : membersDiff.getRemovedPrincipalIds()) {
        	updateForUserRemovedFromGroup(removedPrincipalId, groupId);
        }
        for (String addedPrincipalId : membersDiff.getAddedPrincipalIds()) {
        	updateForUserAddedToGroup(addedPrincipalId, groupId);
        }
    }

    @Override
    public void updateForUserAddedToGroup(String principalId, String groupId) {
        // first verify that the user is still a member of the workgroup
    	if(getGroupService().isMemberOfGroup(principalId, groupId))
    	{
    	    KewApiServiceLocator.getGroupMembershipChangeQueue()
    	        .notifyMembershipChange(KewApiConstants.GroupMembershipChangeOperations.ADDED, groupId, principalId);
    	}
    }

    @Override
    public void updateForUserRemovedFromGroup(String principalId, String groupId) {
        // first verify that the user is no longer a member of the workgroup
    	if(!getGroupService().isMemberOfGroup(principalId, groupId))
    	{
            KewApiServiceLocator.getGroupMembershipChangeQueue()
                .notifyMembershipChange(KewApiConstants.GroupMembershipChangeOperations.REMOVED, groupId, principalId);
    	}

    }

    private MembersDiff getMembersDiff(List<String> oldMemberPrincipalIds, List<String> newMemberPrincipalIds) {

    	// ListUtils does not check the null case.  Which can happen when adding a new group
    	// so, if they're null make them empty lists.
    	if(oldMemberPrincipalIds == null) {
            oldMemberPrincipalIds = new ArrayList<String>();
        }
    	if(newMemberPrincipalIds == null) {
            newMemberPrincipalIds = new ArrayList<String>();
        }

        Set<String> addedPrincipalIds = new HashSet<String>(ListUtils.subtract(newMemberPrincipalIds, oldMemberPrincipalIds));
        Set<String> removedPrincipalIds = new HashSet<String>(ListUtils.subtract(oldMemberPrincipalIds, newMemberPrincipalIds));
        return new MembersDiff(addedPrincipalIds, removedPrincipalIds);
    }

    private class MembersDiff {
        private final Set<String> addedPrincipalIds;

        private final Set<String> removedPrincipalIds;

        public MembersDiff(Set<String> addedPrincipalIds, Set<String>removedPrincipalIds) {
            this.addedPrincipalIds = addedPrincipalIds;
            this.removedPrincipalIds = removedPrincipalIds;
        }

        public Set<String> getAddedPrincipalIds() {
            return addedPrincipalIds;
        }

        public Set<String> getRemovedPrincipalIds() {
            return removedPrincipalIds;
        }
    }

}
