/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kim.impl.group;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.criteria.Predicate;
import org.kuali.rice.core.api.criteria.PredicateUtils;
import org.kuali.rice.core.api.criteria.QueryByCriteria;
import org.kuali.rice.kim.api.KimConstants;
import org.kuali.rice.kim.api.group.Group;
import org.kuali.rice.kim.api.group.GroupQueryResults;
import org.kuali.rice.kim.api.group.GroupService;
import org.kuali.rice.kim.api.services.KimApiServiceLocator;
import org.kuali.rice.krad.lookup.LookupableImpl;
import org.kuali.rice.krad.lookup.LookupForm;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.kuali.rice.core.api.criteria.PredicateFactory.*;
import static org.kuali.rice.core.api.criteria.PredicateFactory.and;
import static org.kuali.rice.core.api.criteria.PredicateFactory.lessThan;

/**
 * Custom lookupable for the {@link GroupBo} lookup to call the group service for searching
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class GroupLookupableImpl extends LookupableImpl {
    private static final long serialVersionUID = -3149952849854425077L;

    /**
     * Translates any search criteria on principal name to member id and active data, then calls group service to
     * retrieve matching groups
     *
     * @return List&lt;GroupBo&gt;
     */
    @Override
    protected Collection<?> executeSearch(Map<String, String> adjustedSearchCriteria,
                List<String> wildcardAsLiteralSearchCriteria, boolean bounded, Integer searchResultsLimit) {
        Map<String, String> criteriaMap = new HashMap<String, String>(adjustedSearchCriteria);
        QueryByCriteria.Builder criteria = QueryByCriteria.Builder.create();

        if (!criteriaMap.isEmpty()) {
            List<Predicate> predicates = new ArrayList<Predicate>();
            //principalId doesn't exist on 'Group'.  Lets do this predicate conversion separately
            if (StringUtils.isNotBlank(criteriaMap.get(KimConstants.UniqueKeyConstants.PRINCIPAL_NAME))) {
                String principalId = KimApiServiceLocator.getIdentityService().getPrincipalByPrincipalName(
                        criteriaMap.get(KimConstants.UniqueKeyConstants.PRINCIPAL_NAME)).getPrincipalId();
                Timestamp currentTime = new Timestamp(Calendar.getInstance().getTimeInMillis());
                predicates.add(and(equal("members.memberId", principalId), equal("members.typeCode",
                        KimConstants.KimGroupMemberTypes.PRINCIPAL_MEMBER_TYPE.getCode()), and(or(isNull(
                        "members.activeFromDateValue"), greaterThanOrEqual("members.activeFromDateValue", currentTime)),
                        or(isNull("members.activeToDateValue"), lessThan("members.activeToDateValue", currentTime)))));

            }
            criteriaMap.remove(KimConstants.UniqueKeyConstants.PRINCIPAL_NAME);

            predicates.add(PredicateUtils.convertMapToPredicate(criteriaMap));
            criteria.setPredicates(and(predicates.toArray(new Predicate[predicates.size()])));
        }

        GroupQueryResults groupResults = getGroupService().findGroups(criteria.build());
        List<Group> groups = groupResults.getResults();

        //have to convert back to Bos :(
        List<GroupBo> groupBos = new ArrayList<GroupBo>(groups.size());
        for (Group group : groups) {
            groupBos.add(GroupBo.from(group));
        }

        return groupBos;
    }

    public GroupService getGroupService() {
        return KimApiServiceLocator.getGroupService();
    }
}
