/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kns.web.struts.action;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.ojb.broker.metadata.ClassNotPersistenceCapableException;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.upload.FormFile;
import org.kuali.rice.core.api.CoreApiServiceLocator;
import org.kuali.rice.core.api.encryption.EncryptionService;
import org.kuali.rice.core.api.util.ClassLoaderUtils;
import org.kuali.rice.core.api.util.RiceConstants;
import org.kuali.rice.core.api.util.RiceKeyConstants;
import org.kuali.rice.core.api.util.io.SerializationUtils;
import org.kuali.rice.core.web.format.Formatter;
import org.kuali.rice.kew.api.KewApiConstants;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.kns.datadictionary.MaintainableCollectionDefinition;
import org.kuali.rice.kns.datadictionary.MaintenanceDocumentEntry;
import org.kuali.rice.kns.document.MaintenanceDocument;
import org.kuali.rice.kns.document.MaintenanceDocumentBase;
import org.kuali.rice.kns.document.authorization.MaintenanceDocumentRestrictions;
import org.kuali.rice.kns.lookup.LookupResultsService;
import org.kuali.rice.kns.maintenance.Maintainable;
import org.kuali.rice.kns.rule.event.KualiAddLineEvent;
import org.kuali.rice.kns.service.KNSServiceLocator;
import org.kuali.rice.kns.service.MaintenanceDocumentDictionaryService;
import org.kuali.rice.kns.util.KNSGlobalVariables;
import org.kuali.rice.kns.util.MaintenanceUtils;
import org.kuali.rice.kns.util.WebUtils;
import org.kuali.rice.kns.web.struts.form.KualiDocumentFormBase;
import org.kuali.rice.kns.web.struts.form.KualiForm;
import org.kuali.rice.kns.web.struts.form.KualiMaintenanceForm;
import org.kuali.rice.krad.bo.*;
import org.kuali.rice.krad.exception.DocumentTypeAuthorizationException;
import org.kuali.rice.krad.service.KRADServiceLocatorWeb;
import org.kuali.rice.krad.service.LookupService;
import org.kuali.rice.krad.util.GlobalVariables;
import org.kuali.rice.krad.util.KRADConstants;
import org.kuali.rice.krad.util.KRADPropertyConstants;
import org.kuali.rice.krad.util.ObjectUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.InvocationTargetException;
import java.security.GeneralSecurityException;
import java.util.*;

/**
 * This class handles actions for maintenance documents. These include creating new edit, and copying of maintenance records.
 *
 * @deprecated Use {@link org.kuali.rice.krad.maintenance.MaintenanceDocumentController}.
 */
@Deprecated
public class KualiMaintenanceDocumentAction extends KualiDocumentActionBase {
    protected static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(KualiMaintenanceDocumentAction.class);

    private MaintenanceDocumentDictionaryService maintenanceDocumentDictionaryService;
    private LookupService lookupService;
    private LookupResultsService lookupResultsService;

	@Override
	public ActionForward execute(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {	
		request.setAttribute(KRADConstants.PARAM_MAINTENANCE_VIEW_MODE, KRADConstants.PARAM_MAINTENANCE_VIEW_MODE_MAINTENANCE);
		return super.execute(mapping, form, request, response);
	}

	/**
	 * Calls setup Maintenance for new action.
	 */
	public ActionForward start(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		request.setAttribute(KRADConstants.MAINTENANCE_ACTN, KRADConstants.MAINTENANCE_NEW_ACTION);
		return setupMaintenance(mapping, form, request, response, KRADConstants.MAINTENANCE_NEW_ACTION);
	}

	/**
	 * Calls setupMaintenance for copy action.
	 */
	public ActionForward copy(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		// check for copy document number
		if (request.getParameter("document." + KRADPropertyConstants.DOCUMENT_NUMBER) == null) { // object copy
			return setupMaintenance(mapping, form, request, response, KRADConstants.MAINTENANCE_COPY_ACTION);
		}
		else { // document copy
			throw new UnsupportedOperationException("System does not support copying of maintenance documents.");
		}
	}

	/**
	 * Calls setupMaintenance for edit action.
	 */
	public ActionForward edit(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
	
		return setupMaintenance(mapping, form, request, response, KRADConstants.MAINTENANCE_EDIT_ACTION);
	}

	/**
	 * KUALRice 3070 Calls setupMaintenance for delete action.
	 */
	@Override
    public ActionForward delete(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		if (isFormRepresentingLockObject((KualiDocumentFormBase)form)) {
			 return super.delete(mapping, form, request, response);
		}
		KNSGlobalVariables.getMessageList().add(RiceKeyConstants.MESSAGE_DELETE);
		return setupMaintenance(mapping, form, request, response, KRADConstants.MAINTENANCE_DELETE_ACTION);
	}
	
	/**
	 * Calls setupMaintenance for new object that have existing objects attributes.
	 */
	public ActionForward newWithExisting(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		return setupMaintenance(mapping, form, request, response, KRADConstants.MAINTENANCE_NEWWITHEXISTING_ACTION);
	}

	/**
	 * Gets a new document for a maintenance record. The maintainable is specified with the documentTypeName or business object
	 * class request parameter and request parameters are parsed for key values for retrieving the business object. Forward to the
	 * maintenance jsp which renders the page based on the maintainable's field specifications. Retrieves an existing business
	 * object for edit and copy. Checks locking on edit.
	 */
    protected ActionForward setupMaintenance(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response, String maintenanceAction) throws Exception {
		KualiMaintenanceForm maintenanceForm = (KualiMaintenanceForm) form;
		MaintenanceDocument document;

		// create a new document object, if required (on NEW object, or other reasons)
		if (maintenanceForm.getDocument() == null) {
			if (StringUtils.isEmpty(maintenanceForm.getBusinessObjectClassName()) && StringUtils.isEmpty(maintenanceForm.getDocTypeName())) {
				throw new IllegalArgumentException("Document type name or bo class not given!");
			}

			String documentTypeName = maintenanceForm.getDocTypeName();
			// get document type if not passed
			if (StringUtils.isEmpty(documentTypeName)) {
				documentTypeName = getMaintenanceDocumentDictionaryService().getDocumentTypeName(Class.forName(maintenanceForm.getBusinessObjectClassName()));
				maintenanceForm.setDocTypeName(documentTypeName);
			}

			if (StringUtils.isEmpty(documentTypeName)) {
				throw new RuntimeException("documentTypeName is empty; does this Business Object have a maintenance document definition? " + maintenanceForm.getBusinessObjectClassName());
			}

			// check doc type allows new or copy if that action was requested
			if (KRADConstants.MAINTENANCE_NEW_ACTION.equals(maintenanceAction) || KRADConstants.MAINTENANCE_COPY_ACTION.equals(maintenanceAction)) {
				Class<?> boClass = getMaintenanceDocumentDictionaryService().getDataObjectClass(documentTypeName);
				boolean allowsNewOrCopy = getBusinessObjectAuthorizationService().canCreate(boClass, GlobalVariables.getUserSession().getPerson(), documentTypeName);
				if (!allowsNewOrCopy) {
					LOG.error("Document type " + documentTypeName + " does not allow new or copy actions.");
					throw new DocumentTypeAuthorizationException(GlobalVariables.getUserSession().getPerson().getPrincipalId(), "newOrCopy", documentTypeName);
				}
			}

			// get new document from service
			document = (MaintenanceDocument) getDocumentService().getNewDocument(maintenanceForm.getDocTypeName());
			maintenanceForm.setDocument(document);
		}
		else {
			document = (MaintenanceDocument) maintenanceForm.getDocument();
		}

		// retrieve business object from request parameters
		if (!(KRADConstants.MAINTENANCE_NEW_ACTION.equals(maintenanceAction))
                && !(KRADConstants.MAINTENANCE_NEWWITHEXISTING_ACTION.equals(maintenanceAction))) {
			Map<String, String> requestParameters = buildKeyMapFromRequest(document.getNewMaintainableObject(), request);
            PersistableBusinessObject oldBusinessObject = null;
            try {
				Class<?> boClass = Class.forName(maintenanceForm.getBusinessObjectClassName());
            	oldBusinessObject = (PersistableBusinessObject) getLookupService().findObjectBySearch(boClass, requestParameters);
            	if (oldBusinessObject == null && PersistableBusinessObject.class.isAssignableFrom(boClass)) {
					if (KRADConstants.MAINTENANCE_EDIT_ACTION.equals(maintenanceAction) || KRADConstants.MAINTENANCE_COPY_ACTION.equals(maintenanceAction)) {
						oldBusinessObject = document.getNewMaintainableObject().retrieveObjectForEditOrCopy(document, requestParameters);
					}
				}
            } catch ( ClassNotPersistenceCapableException ex ) {
            	if ( !document.getOldMaintainableObject().isExternalBusinessObject() ) {
            		throw new RuntimeException( "BO Class: " + maintenanceForm.getBusinessObjectClassName() + " is not persistable and is not externalizable - configuration error", ex );
            	}
            	// otherwise, let fall through
            }
			if (oldBusinessObject == null && !document.getOldMaintainableObject().isExternalBusinessObject()) {
                throw new RuntimeException("Cannot retrieve old record for maintenance document, incorrect parameters passed on maint url: " + requestParameters );
			} 

			if(document.getOldMaintainableObject().isExternalBusinessObject()){
            	if ( oldBusinessObject == null ) {
            		try {
            			oldBusinessObject = (PersistableBusinessObject)document.getOldMaintainableObject().getBoClass().newInstance();
            		} catch ( Exception ex ) {
            			throw new RuntimeException( "External BO maintainable was null and unable to instantiate for old maintainable object.", ex );
            		}
            	}
				populateBOWithCopyKeyValues(request, oldBusinessObject, document.getOldMaintainableObject());
				document.getOldMaintainableObject().prepareBusinessObject(oldBusinessObject);
			}

			PersistableBusinessObject newBusinessObject = (PersistableBusinessObject) SerializationUtils.deepCopy(
                    oldBusinessObject);

			// set business object instance for editing
			Class<? extends PersistableBusinessObject> businessObjectClass = ClassLoaderUtils.getClass(maintenanceForm.getBusinessObjectClassName(), PersistableBusinessObject.class); 
			document.getOldMaintainableObject().setBusinessObject(oldBusinessObject);
			document.getOldMaintainableObject().setBoClass(businessObjectClass);
			document.getNewMaintainableObject().setBusinessObject(newBusinessObject);
			document.getNewMaintainableObject().setBoClass(businessObjectClass);


			// on a COPY, clear any fields that this user isnt authorized for, and also
			// clear the primary key fields and the version number and objectId
			if (KRADConstants.MAINTENANCE_COPY_ACTION.equals(maintenanceAction)) {
				if (!document.isFieldsClearedOnCopy()) {
					//for issue KULRice 3072
					Class<?> boClass = getMaintenanceDocumentDictionaryService().getDataObjectClass(
                            maintenanceForm.getDocTypeName());
                    if (!getMaintenanceDocumentDictionaryService().getPreserveLockingKeysOnCopy(boClass)) {
                        clearPrimaryKeyFields(document);
                    }

					clearUnauthorizedNewFields(document);

					Maintainable maintainable = document.getNewMaintainableObject();

					maintainable.processAfterCopy( document, request.getParameterMap() );

					// mark so that this clearing doesnt happen again
					document.setFieldsClearedOnCopy(true);

					// mark so that blank required fields will be populated with default values
					maintainable.setGenerateBlankRequiredValues(maintenanceForm.getDocTypeName());
				}
			}
			else if (KRADConstants.MAINTENANCE_EDIT_ACTION.equals(maintenanceAction)) {
				boolean allowsEdit = getBusinessObjectAuthorizationService().canMaintain(oldBusinessObject, GlobalVariables.getUserSession().getPerson(), document.getDocumentHeader().getWorkflowDocument().getDocumentTypeName());
				if (!allowsEdit) {
					LOG.error("Document type " + document.getDocumentHeader().getWorkflowDocument().getDocumentTypeName() + " does not allow edit actions.");
					throw  new DocumentTypeAuthorizationException(GlobalVariables.getUserSession().getPerson().getPrincipalId(), "edit", document.getDocumentHeader().getWorkflowDocument().getDocumentTypeName());
				}
				document.getNewMaintainableObject().processAfterEdit( document, request.getParameterMap() );
			}
			//3070
			else if (KRADConstants.MAINTENANCE_DELETE_ACTION.equals(maintenanceAction)) {
				boolean allowsDelete = getBusinessObjectAuthorizationService().canMaintain(oldBusinessObject, GlobalVariables.getUserSession().getPerson(), document.getDocumentHeader().getWorkflowDocument().getDocumentTypeName());
				if (!allowsDelete) {
					LOG.error("Document type " + document.getDocumentHeader().getWorkflowDocument().getDocumentTypeName() + " does not allow delete actions.");
					throw  new DocumentTypeAuthorizationException(GlobalVariables.getUserSession().getPerson().getPrincipalId(), "delete", document.getDocumentHeader().getWorkflowDocument().getDocumentTypeName());
				}
			}
		}
		// if new with existing we need to populate we need to populate with passed in parameters
		if (KRADConstants.MAINTENANCE_NEWWITHEXISTING_ACTION.equals(maintenanceAction)) {
			PersistableBusinessObject newBO = document.getNewMaintainableObject().getBusinessObject();
			Map<String, String> parameters = buildKeyMapFromRequest(document.getNewMaintainableObject(), request);
			copyParametersToBO(parameters, newBO);
			newBO.refresh();
			document.getNewMaintainableObject().setupNewFromExisting( document, request.getParameterMap() );
		}

		// for new maintainble need to pick up default values
		if (KRADConstants.MAINTENANCE_NEW_ACTION.equals(maintenanceAction)) {
			document.getNewMaintainableObject().setGenerateDefaultValues(maintenanceForm.getDocTypeName());
			document.getNewMaintainableObject().processAfterNew( document, request.getParameterMap() );

			// If a maintenance lock exists, warn the user.
			MaintenanceUtils.checkForLockingDocument(document.getNewMaintainableObject(), false);
		}

		// set maintenance action state
		document.getNewMaintainableObject().setMaintenanceAction(maintenanceAction);
		maintenanceForm.setMaintenanceAction(maintenanceAction);

		// attach any extra JS from the data dictionary
        MaintenanceDocumentEntry entry =  getMaintenanceDocumentDictionaryService().getMaintenanceDocumentEntry(document.getDocumentHeader().getWorkflowDocument().getDocumentTypeName());
		if (LOG.isDebugEnabled()) {
			LOG.debug("maintenanceForm.getAdditionalScriptFiles(): " + maintenanceForm.getAdditionalScriptFiles());
		}
		if (maintenanceForm.getAdditionalScriptFiles().isEmpty()) {
			maintenanceForm.getAdditionalScriptFiles().addAll(entry.getWebScriptFiles());
		}

		// Retrieve notes topic display flag from data dictionary and add to document
		document.setDisplayTopicFieldInNotes(entry.getDisplayTopicFieldInNotes());

		return mapping.findForward(RiceConstants.MAPPING_BASIC);
	}

    protected void populateBOWithCopyKeyValues(HttpServletRequest request, PersistableBusinessObject oldBusinessObject, Maintainable oldMaintainableObject) throws Exception{
		List<String> keyFieldNamesToCopy = new ArrayList<>();
		Map<String, String> parametersToCopy;
		if (!StringUtils.isBlank(request.getParameter(KRADConstants.COPY_KEYS))) {
			String[] copyKeys = request.getParameter(KRADConstants.COPY_KEYS).split(KRADConstants.FIELD_CONVERSIONS_SEPARATOR);
			keyFieldNamesToCopy.addAll(Arrays.asList(copyKeys));
		}
		parametersToCopy = getRequestParameters(keyFieldNamesToCopy, oldMaintainableObject, request);
		if(parametersToCopy!=null && parametersToCopy.size()>0){
			copyParametersToBO(parametersToCopy, oldBusinessObject);
		}
	}

    protected void copyParametersToBO(Map<String, String> parameters, PersistableBusinessObject newBO) throws Exception{
		for (String parmName : parameters.keySet()) {
			String propertyValue = parameters.get(parmName);

			if (StringUtils.isNotBlank(propertyValue)) {
				// set value of property in bo
				if (PropertyUtils.isWriteable(newBO, parmName)) {
					Class<?> type = ObjectUtils.easyGetPropertyType(newBO, parmName);
					if (type != null) {
						Formatter formatter = Formatter.getFormatter(type);
						Object obj = formatter.convertFromPresentationFormat(propertyValue);
						ObjectUtils.setObjectProperty(newBO, parmName, obj.getClass(), obj);
					}
					else {
						ObjectUtils.setObjectProperty(newBO, parmName, String.class, propertyValue);
					}
				}
			}
		}
	}

	/**
	 * Downloads the attachment to the user's browser.
	 */
	public ActionForward downloadAttachment(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		KualiDocumentFormBase documentForm = (KualiDocumentFormBase) form;
		MaintenanceDocumentBase document = (MaintenanceDocumentBase) documentForm.getDocument();

		//for some reason the maintenance framework leaves off ".businessObject." from the property path.  It must be added back on.
		final String brokenDownloadAttachmentPath = request.getParameter("downloadAttachmentPath");
		if (StringUtils.isNotBlank(brokenDownloadAttachmentPath)) {
			final String downloadAttachmentPath = brokenDownloadAttachmentPath.contains(".businessObject.") ? brokenDownloadAttachmentPath : brokenDownloadAttachmentPath
					.replaceAll("document\\.newMaintainableObject\\.", "document.newMaintainableObject.businessObject.")
					.replaceAll("document\\.oldMaintainableObject\\.", "document.oldMaintainableObject.businessObject.");

			final FormFile att = (FormFile) PropertyUtils.getNestedProperty(form, downloadAttachmentPath);
			if (att != null) {
				streamToResponse(att.getFileData(), att.getFileName(), att.getContentType(), response);
				return null;
			}
		}

        int line = getSelectedLine(request);
        if (line < 0) {
            DocumentAttachment documentAttachment = document.getAttachment();
            if (documentAttachment != null
                    && documentAttachment.getAttachmentContent() != null) {

                streamToResponse(documentAttachment.getAttachmentContent(), documentAttachment.getFileName(), documentAttachment.getContentType(), response);
                return null;
            }
            PersistableAttachment attachment = (PersistableAttachment) document.getNewMaintainableObject().getBusinessObject();
            String attachmentPropNm = document.getAttachmentPropertyName();
            FormFile attachmentFromBusinessObject = null;
            byte[] attachmentContent;
            String fileName = attachment.getFileName();
            String contentType = attachment.getContentType();
            if (StringUtils.isNotBlank(attachmentPropNm)) {
                attachmentFromBusinessObject = (FormFile) PropertyUtils.getNestedProperty(attachment, attachmentPropNm);
            }
            if (attachmentFromBusinessObject != null
                    && attachmentFromBusinessObject.getInputStream() != null) {
                attachmentContent = attachmentFromBusinessObject.getFileData();
                fileName = attachmentFromBusinessObject.getFileName();
                contentType = attachmentFromBusinessObject.getContentType();
            } else {
                attachmentContent = attachment.getAttachmentContent();
            }
            if (StringUtils.isNotBlank(fileName)
                    && contentType != null
                    && attachmentContent != null) {
                streamToResponse(attachmentContent, fileName, contentType, response);
            }
        } else {

            // attachment is part of a collection
            PersistableAttachmentList<PersistableAttachment> attachmentsBo = (PersistableAttachmentList<PersistableAttachment>) document.getNewMaintainableObject().getBusinessObject();
            if (CollectionUtils.isEmpty(attachmentsBo.getAttachments())) {
                document.populateAttachmentListForBO();
            }

            List<? extends PersistableAttachment> attachments = attachmentsBo.getAttachments();
            if (CollectionUtils.isNotEmpty(attachments)
                    && attachments.size() > line) {
                PersistableAttachment attachment = attachmentsBo.getAttachments().get(line);

                //it is possible that document hasn't been saved (attachment just added) and the attachment content is still in the FormFile
                //need to grab it if that is the case.
                byte[] attachmentContent; // = attachment.getAttachmentContent();
                String fileName = attachment.getFileName();
                String contentType = attachment.getContentType();
                String attachmentPropNm = document.getAttachmentListPropertyName();
                FormFile attachmentFromBusinessObject = null;
                if (StringUtils.isNotBlank(attachmentPropNm)) {
                    attachmentFromBusinessObject = (FormFile)PropertyUtils.getNestedProperty(attachment, attachmentPropNm);
                }
                //Use form file data if it exists
                //if (attachmentContent == null) {

                if (attachmentFromBusinessObject != null
                    && attachmentFromBusinessObject.getInputStream() != null) {
                    attachmentContent = attachmentFromBusinessObject.getFileData();
                    fileName = attachmentFromBusinessObject.getFileName();
                    contentType = attachmentFromBusinessObject.getContentType();
                } else {
                    attachmentContent = attachment.getAttachmentContent();
                }

                if (attachmentContent != null) {
                    streamToResponse(attachmentContent, fileName, contentType, response);
                } else {
                    // last ditch effort to find the correct attachment
                    //check to see if attachment is populated on document first, so no copying done unless necessary
                    List<MultiDocumentAttachment> multiDocumentAttachs = document.getAttachments();
                    if (CollectionUtils.isNotEmpty(multiDocumentAttachs)) {
                        for (MultiDocumentAttachment multiAttach : multiDocumentAttachs) {
                            if (multiAttach.getFileName().equals(fileName)
                                    && multiAttach.getContentType().equals(contentType)) {
                                streamToResponse(multiAttach.getAttachmentContent(), multiAttach.getFileName(), multiAttach.getContentType(), response);
                                break;
                            }
                        }
                    }
                }
            }
        }
		return null;
	}


	/**
	 * 
	 * This method used to replace the attachment.
	 */
	public ActionForward replaceAttachment(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		KualiDocumentFormBase documentForm = (KualiDocumentFormBase) form;
		MaintenanceDocumentBase document = (MaintenanceDocumentBase) documentForm.getDocument();

		//for some reason the maintenance framework leaves off ".businessObject." from the property path.  It must be added back on.
		final String brokenReplaceAttachmentPath = request.getParameter("replaceAttachmentPath");
		if (StringUtils.isNotBlank(brokenReplaceAttachmentPath)) {
			final String replaceAttachmentPath = brokenReplaceAttachmentPath.contains(".businessObject.") ? brokenReplaceAttachmentPath : brokenReplaceAttachmentPath
					.replaceAll("document\\.newMaintainableObject\\.", "document.newMaintainableObject.businessObject.")
					.replaceAll("document\\.oldMaintainableObject\\.", "document.oldMaintainableObject.businessObject.");

			final Object parent = PropertyUtils.getNestedProperty(form, StringUtils.substringBeforeLast(replaceAttachmentPath, "."));
			if (parent instanceof PersistableAttachment) {
				final PersistableAttachment attachment = (PersistableAttachment) parent;
				attachment.setAttachmentContent(null);
				attachment.setContentType(null);
				attachment.setFileName(null);
			}
			PropertyUtils.setNestedProperty(form, replaceAttachmentPath, null);
			return mapping.findForward(RiceConstants.MAPPING_BASIC);
		}

		int lineNum = getSelectedLine(request);

        if (lineNum < 0) {

            document.refreshReferenceObject("attachment");
            documentForm.setAttachmentFile(null);
            document.setFileAttachment(null);
            getBusinessObjectService().delete(document.getAttachment());
            document.setAttachment(null);

            PersistableAttachment attachment = (PersistableAttachment) document.getNewMaintainableObject().getBusinessObject();

            attachment.setAttachmentContent(null);
            attachment.setContentType(null);
            attachment.setFileName(null);

            if (StringUtils.isNotBlank(document.getAttachmentPropertyName())) {
				PropertyUtils.setProperty(attachment, document.getAttachmentPropertyName(), null);
			}
        } else {
            document.refreshReferenceObject("attachments");
            getBusinessObjectService().delete(document.getAttachment());

            PersistableAttachmentList<PersistableAttachment> attachmentListBo = (PersistableAttachmentList<PersistableAttachment>) document.getNewMaintainableObject().getBusinessObject();

            PersistableAttachment attachment = attachmentListBo.getAttachments().get(lineNum);
            attachment.setAttachmentContent(null);
            attachment.setContentType(null);
            attachment.setFileName(null);

			if (StringUtils.isNotBlank(document.getAttachmentListPropertyName())) {
				PropertyUtils.setProperty(attachment, document.getAttachmentListPropertyName(), null);
			}
        }

	    return mapping.findForward(RiceConstants.MAPPING_BASIC);
	}

	/**
	 * route the document using the document service
	 */
	@Override
	public ActionForward route(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		KualiDocumentFormBase documentForm = (KualiDocumentFormBase) form;
		MaintenanceDocumentBase document = (MaintenanceDocumentBase) documentForm.getDocument();

		ActionForward forward = super.route(mapping, form, request, response);
		PersistableBusinessObject businessObject = document.getNewMaintainableObject().getBusinessObject();
		if(businessObject instanceof PersistableAttachment) {
			document.populateAttachmentForBO();
			String fileName = ((PersistableAttachment) businessObject).getFileName();
			if(StringUtils.isEmpty(fileName)) {
				PersistableAttachment existingBO = (PersistableAttachment) getBusinessObjectService().retrieve(document.getNewMaintainableObject().getBusinessObject());
				if (existingBO == null) {
					if (document.getAttachment() != null) {
						fileName = document.getAttachment().getFileName();
					} else {
						fileName = "";
					}
				} else {
					fileName = existingBO.getFileName();
				}
				request.setAttribute(KRADConstants.BO_ATTACHMENT_FILE_NAME, fileName);
			}
		}
		return forward;
	}

	/**
	 * Handles creating and loading of documents.
	 */
	@Override
	public ActionForward docHandler(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		ActionForward af = super.docHandler(mapping, form, request, response);
        if (af.getName().equals(KRADConstants.KRAD_INITIATED_DOCUMENT_VIEW_NAME))
        {
            return af;
        }
		KualiMaintenanceForm kualiMaintenanceForm = (KualiMaintenanceForm) form;

		if (KewApiConstants.ACTIONLIST_COMMAND.equals(kualiMaintenanceForm.getCommand()) || KewApiConstants.DOCSEARCH_COMMAND.equals(kualiMaintenanceForm.getCommand()) || KewApiConstants.SUPERUSER_COMMAND.equals(kualiMaintenanceForm.getCommand()) || KewApiConstants.HELPDESK_ACTIONLIST_COMMAND.equals(kualiMaintenanceForm.getCommand()) && kualiMaintenanceForm.getDocId() != null) {
			if (kualiMaintenanceForm.getDocument() instanceof MaintenanceDocument) {
				kualiMaintenanceForm.setReadOnly(true);
				kualiMaintenanceForm.setMaintenanceAction(((MaintenanceDocument) kualiMaintenanceForm.getDocument()).getNewMaintainableObject().getMaintenanceAction());

				//Retrieving the FileName from BO table
				Maintainable tmpMaintainable = ((MaintenanceDocument) kualiMaintenanceForm.getDocument()).getNewMaintainableObject();
				if(tmpMaintainable.getBusinessObject() instanceof PersistableAttachment) {
					PersistableAttachment bo = (PersistableAttachment) getBusinessObjectService().retrieve(tmpMaintainable.getBusinessObject());
                    if (bo != null) {
                        request.setAttribute(KRADConstants.BO_ATTACHMENT_FILE_NAME, bo.getFileName());
                    }
				}
			}
			else {
				LOG.error("Illegal State: document is not a maintenance document");
				throw new IllegalArgumentException("Document is not a maintenance document");
			}
		}
		else if (KewApiConstants.INITIATE_COMMAND.equals(kualiMaintenanceForm.getCommand())) {
			kualiMaintenanceForm.setReadOnly(false);
			return setupMaintenance(mapping, form, request, response, KRADConstants.MAINTENANCE_NEW_ACTION);
		}
		else {
			LOG.error("We should never have gotten to here");
			throw new IllegalArgumentException("docHandler called with invalid parameters");
		}
		return mapping.findForward(RiceConstants.MAPPING_BASIC);
	}

	/**
	 * Called on return from a lookup.
	 */
	@Override
	public ActionForward refresh(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		KualiMaintenanceForm maintenanceForm = (KualiMaintenanceForm) form;

		WebUtils.reuseErrorMapFromPreviousRequest(maintenanceForm);
		maintenanceForm.setDerivedValuesOnForm(request);

		refreshAdHocRoutingWorkgroupLookups(request, maintenanceForm);
		MaintenanceDocument document = (MaintenanceDocument) maintenanceForm.getDocument();

		// call refresh on new maintainable
		Map<String, String> requestParams = new HashMap<>();
		for (Enumeration<String> i = request.getParameterNames(); i.hasMoreElements();) {
			String requestKey = i.nextElement();
			String requestValue = request.getParameter(requestKey);
			requestParams.put(requestKey, requestValue);
		}

		// Add multiple values from Lookup
		Collection<PersistableBusinessObject> rawValues = null;
		if (StringUtils.equals(KRADConstants.MULTIPLE_VALUE, maintenanceForm.getRefreshCaller())) {
			String lookupResultsSequenceNumber = maintenanceForm.getLookupResultsSequenceNumber();
			if (StringUtils.isNotBlank(lookupResultsSequenceNumber)) {
				// actually returning from a multiple value lookup
				String lookupResultsBOClassName = maintenanceForm.getLookupResultsBOClassName();
				Class<PersistableBusinessObject> lookupResultsBOClass = (Class<PersistableBusinessObject>) Class.forName(lookupResultsBOClassName);

				rawValues = getLookupResultsService().retrieveSelectedResultBOs(lookupResultsSequenceNumber, lookupResultsBOClass, GlobalVariables.getUserSession().getPerson().getPrincipalId());
			}
		}

		if (rawValues != null) {
			String collectionName = maintenanceForm.getLookedUpCollectionName();
			document.getNewMaintainableObject().addMultipleValueLookupResults(document, collectionName, rawValues, false, document.getNewMaintainableObject().getBusinessObject());
			if (LOG.isInfoEnabled()) {
				LOG.info("********************doing editing 3 in refersh()***********************.");
			}
			boolean isEdit = KRADConstants.MAINTENANCE_EDIT_ACTION.equals(maintenanceForm.getMaintenanceAction());
			boolean isCopy = KRADConstants.MAINTENANCE_COPY_ACTION.equals(maintenanceForm.getMaintenanceAction());

			if (isEdit || isCopy) {
				document.getOldMaintainableObject().addMultipleValueLookupResults(document, collectionName, rawValues, true, document.getOldMaintainableObject().getBusinessObject());
				document.getOldMaintainableObject().refresh(maintenanceForm.getRefreshCaller(), requestParams, document);
			}
		}

		document.getNewMaintainableObject().refresh(maintenanceForm.getRefreshCaller(), requestParams, document);

		//pass out customAction from methodToCall parameter. Call processAfterPost
		String fullParameter = (String) request.getAttribute(KRADConstants.METHOD_TO_CALL_ATTRIBUTE);
		if(StringUtils.contains(fullParameter, KRADConstants.CUSTOM_ACTION)){
			String customAction = StringUtils.substringBetween(fullParameter, KRADConstants.METHOD_TO_CALL_PARM1_LEFT_DEL, KRADConstants.METHOD_TO_CALL_PARM1_RIGHT_DEL);
			String[] actionValue = new String[1];
			actionValue[0]= StringUtils.substringAfter(customAction, ".");
			Map<String,String[]> paramMap = new HashMap<>(request.getParameterMap());
			paramMap.put(KRADConstants.CUSTOM_ACTION, actionValue);
			doProcessingAfterPost( (KualiMaintenanceForm) form, paramMap );
		}

		return mapping.findForward(RiceConstants.MAPPING_BASIC);
	}

	/**
	 * Gets keys for the maintainable business object from the persistence metadata explorer. Checks for existence of key property
	 * names as request parameters, if found adds them to the returned hash map.
	 */
    protected Map<String, String> buildKeyMapFromRequest(Maintainable maintainable, HttpServletRequest request) {
		List<String> keyFieldNames;
		// are override keys listed in the request? If so, then those need to be our keys,
		// not the primary keye fields for the BO
		if (!StringUtils.isBlank(request.getParameter(KRADConstants.OVERRIDE_KEYS))) {
			String[] overrideKeys = request.getParameter(KRADConstants.OVERRIDE_KEYS).split(KRADConstants.FIELD_CONVERSIONS_SEPARATOR);
			keyFieldNames = new ArrayList<>(Arrays.asList(overrideKeys));
		}
		else {
			keyFieldNames = KRADServiceLocatorWeb.getLegacyDataAdapter().listPrimaryKeyFieldNames(maintainable.getBusinessObject().getClass());
		}
		return getRequestParameters(keyFieldNames, maintainable, request);
	}

    protected Map<String, String> getRequestParameters(List<String> keyFieldNames, Maintainable maintainable, HttpServletRequest request){

		Map<String, String> requestParameters = new HashMap<String, String>();


		for (String keyPropertyName : keyFieldNames) {
			if (request.getParameter(keyPropertyName) != null) {
				String keyValue = request.getParameter(keyPropertyName);

				// Check if this element was encrypted, if it was decrypt it
				if (getBusinessObjectAuthorizationService().attributeValueNeedsToBeEncryptedOnFormsAndLinks(maintainable.getBoClass(), keyPropertyName)) {
					try {
						keyValue = StringUtils.removeEnd(keyValue, EncryptionService.ENCRYPTION_POST_PREFIX);
						if (CoreApiServiceLocator.getEncryptionService().isEnabled()) {
							keyValue = getEncryptionService().decrypt(keyValue);
						}
					} catch (GeneralSecurityException e) {
						throw new RuntimeException(e);
					}
				}


				requestParameters.put(keyPropertyName, keyValue);
			}
		}

		return requestParameters;

	}

	/**
	 * Convert a Request into a Map<String,String>. Technically, Request parameters do not neatly translate into a Map of Strings,
	 * because a given parameter may legally appear more than once (so a Map of String[] would be more accurate.) This method should
	 * be safe for business objects, but may not be reliable for more general uses.
	 */
	String extractCollectionName(HttpServletRequest request, String methodToCall) {
		// collection name and underlying object type from request parameter
		String parameterName = (String) request.getAttribute(KRADConstants.METHOD_TO_CALL_ATTRIBUTE);
		String collectionName = null;
		if (StringUtils.isNotBlank(parameterName)) {
			collectionName = StringUtils.substringBetween(parameterName, methodToCall + ".", ".(");
		}
		return collectionName;
	}

	Collection<Object> extractCollection(Object bo, String collectionName) {
		// retrieve the collection from the business object
		Collection<Object> maintCollection = (Collection<Object>) ObjectUtils.getPropertyValue(bo, collectionName);
		return maintCollection;
	}

	Class<?> extractCollectionClass(String docTypeName, String collectionName) {
		return getMaintenanceDocumentDictionaryService().getCollectionBusinessObjectClass(docTypeName, collectionName);
	}

	/**
	 * Adds a line to a collection being maintained in a many section.
	 */
	public ActionForward addLine(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		KualiMaintenanceForm maintenanceForm = (KualiMaintenanceForm) form;
		MaintenanceDocument document = (MaintenanceDocument) maintenanceForm.getDocument();
		Maintainable oldMaintainable = document.getOldMaintainableObject();
		Maintainable newMaintainable = document.getNewMaintainableObject();

		String collectionName = extractCollectionName(request, KRADConstants.ADD_LINE_METHOD);
		if (collectionName == null) {
			LOG.error("Unable to get find collection name and class in request.");
			throw new RuntimeException("Unable to get find collection name and class in request.");
		}

		// if dealing with sub collection it will have a "["
		if ((StringUtils.lastIndexOf(collectionName, "]") + 1) == collectionName.length()) {
			collectionName = StringUtils.substringBeforeLast(collectionName, "[");
		}

		Object bo = newMaintainable.getBusinessObject();
		Collection<Object> maintCollection = extractCollection(bo, collectionName);
		Class<?> collectionClass = extractCollectionClass(maintenanceForm.getDocument().getDocumentHeader().getWorkflowDocument().getDocumentTypeName(), collectionName);
		// get the BO from the new collection line holder
		PersistableBusinessObject addBO = newMaintainable.getNewCollectionLine(collectionName);
		if (LOG.isDebugEnabled()) {
			LOG.debug("obtained addBO from newCollectionLine: " + addBO);
		}

		// link up the user fields, if any
		getBusinessObjectService().linkUserFields(addBO);

		//KULRICE-4264 - a hook to change the state of the business object, which is the "new line" of a collection, before it is validated
		newMaintainable.processBeforeAddLine(collectionName, collectionClass, addBO);
		
		// apply rules to the addBO
		boolean rulePassed;
		if (LOG.isDebugEnabled()) {
			LOG.debug("about to call AddLineEvent applyRules: document=" + document + "\ncollectionName=" + collectionName + "\nBO=" + addBO);
		}
		rulePassed = getKualiRuleService().applyRules(new KualiAddLineEvent(document, collectionName, addBO));

		// if the rule evaluation passed, let's add it
		if (rulePassed) {
			if (LOG.isInfoEnabled()) {
				LOG.info("********************doing editing 4 in addline()***********************.");
			}
			// if edit or copy action, just add empty instance to old maintainable
			boolean isEdit = KRADConstants.MAINTENANCE_EDIT_ACTION.equals(maintenanceForm.getMaintenanceAction());
			boolean isCopy = KRADConstants.MAINTENANCE_COPY_ACTION.equals(maintenanceForm.getMaintenanceAction());


			if (isEdit || isCopy) {
				Object oldBo = oldMaintainable.getBusinessObject();
				Collection<Object> oldMaintCollection = (Collection<Object>) ObjectUtils.getPropertyValue(oldBo, collectionName);

				if (oldMaintCollection == null) {
					oldMaintCollection = new ArrayList<>();
				}
				if (PersistableBusinessObject.class.isAssignableFrom(collectionClass)) {
					PersistableBusinessObject placeholder = (PersistableBusinessObject) collectionClass.newInstance();
					// KULRNE-4538: must set it as a new collection record, because the maintainable will set the BO that gets added
					// to the new maintainable as a new collection record

					// if not set, then the subcollections of the newly added object will appear as read only
					// see FieldUtils.getContainerRows on how the delete button is rendered
					placeholder.setNewCollectionRecord(true);
					oldMaintCollection.add(placeholder);
				}
				else {
					LOG.warn("Should be a instance of PersistableBusinessObject");
					oldMaintCollection.add(collectionClass.newInstance());
				}
				// update collection in maintenance business object
				ObjectUtils.setObjectProperty(oldBo, collectionName, List.class, oldMaintCollection);
			}

			newMaintainable.addNewLineToCollection(collectionName);
			for (Object aSubCollection : maintCollection) {
				getSubCollectionIndex(aSubCollection, maintenanceForm.getDocTypeName());
			}
		}
		doProcessingAfterPost( (KualiMaintenanceForm) form, request );

		return mapping.findForward(RiceConstants.MAPPING_BASIC);
	}

    protected int getSubCollectionIndex(Object object, String documentTypeName) {
		int index = 1;
		MaintainableCollectionDefinition theCollectionDefinition = null;
		for (MaintainableCollectionDefinition maintainableCollectionDefinition : getMaintenanceDocumentDictionaryService().getMaintainableCollections(documentTypeName)) {
			if (maintainableCollectionDefinition.getBusinessObjectClass().equals(object.getClass())) {
				// we've found the collection we were looking for, so let's find all of its subcollections
				theCollectionDefinition = maintainableCollectionDefinition;
				break;
			}
		}
		if (theCollectionDefinition != null) {
			for (MaintainableCollectionDefinition subCollDef : theCollectionDefinition.getMaintainableCollections()) {
				List<?> subCollectionList = new ArrayList<>();
				try {
					subCollectionList = (List<?>) PropertyUtils.getNestedProperty(object, subCollDef.getName());
				} catch (InvocationTargetException | IllegalAccessException | NoSuchMethodException ite) {
					// this shouldn't happen
				}
				index += subCollectionList.size();
			}
		}
		return index;
	}

	/**
	 * Deletes a collection line that is pending by this document. The collection name and the index to delete is embedded into the
	 * delete button name. These parameters are extracted, the collection pulled out of the parent business object, and finally the
	 * collection record at the specified index is removed for the new maintainable, and the old if we are dealing with an edit.
	 */
	public ActionForward deleteLine(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		KualiMaintenanceForm maintenanceForm = (KualiMaintenanceForm) form;
		MaintenanceDocument document = (MaintenanceDocument) maintenanceForm.getDocument();
		Maintainable oldMaintainable = document.getOldMaintainableObject();
		Maintainable newMaintainable = document.getNewMaintainableObject();

		String collectionName = extractCollectionName(request, KRADConstants.DELETE_LINE_METHOD);
		if (collectionName == null) {
			LOG.error("Unable to get find collection name in request.");
			throw new RuntimeException("Unable to get find collection class in request.");
		}

		PersistableBusinessObject bo = newMaintainable.getBusinessObject();
		Collection<?> maintCollection = extractCollection(bo, collectionName);

		int deleteRecordIndex = getLineToDelete(request);

		((List<?>) maintCollection).remove(deleteRecordIndex);

		// if it's either an edit or a copy, need to remove the collection from the old maintainable as well
		if (KRADConstants.MAINTENANCE_EDIT_ACTION.equals(maintenanceForm.getMaintenanceAction()) ||
				KRADConstants.MAINTENANCE_COPY_ACTION.equals(maintenanceForm.getMaintenanceAction())) {
			bo = oldMaintainable.getBusinessObject();
			maintCollection = extractCollection(bo, collectionName);
			((List<?>) maintCollection).remove(deleteRecordIndex);
		}

		doProcessingAfterPost( (KualiMaintenanceForm) form, request );

		return mapping.findForward(RiceConstants.MAPPING_BASIC);
	}

	/**
	 * Turns on (or off) the inactive record display for a maintenance collection.
	 */
	public ActionForward toggleInactiveRecordDisplay(ActionMapping mapping, ActionForm form, HttpServletRequest request, HttpServletResponse response) throws Exception {
		KualiMaintenanceForm maintenanceForm = (KualiMaintenanceForm) form;
		MaintenanceDocument document = (MaintenanceDocument) maintenanceForm.getDocument();
		Maintainable oldMaintainable = document.getOldMaintainableObject();
		Maintainable newMaintainable = document.getNewMaintainableObject();

		String collectionName = extractCollectionName(request, KRADConstants.TOGGLE_INACTIVE_METHOD);
		if (collectionName == null) {
			LOG.error("Unable to get find collection name in request.");
			throw new RuntimeException("Unable to get find collection class in request.");
		}  

		String parameterName = (String) request.getAttribute(KRADConstants.METHOD_TO_CALL_ATTRIBUTE);
		boolean showInactive = Boolean.parseBoolean(StringUtils.substringBetween(parameterName, KRADConstants.METHOD_TO_CALL_BOPARM_LEFT_DEL, "."));

		oldMaintainable.setShowInactiveRecords(collectionName, showInactive);
		newMaintainable.setShowInactiveRecords(collectionName, showInactive);

		return mapping.findForward(RiceConstants.MAPPING_BASIC);
	}

	/**
	 * This method clears the value of the primary key fields on a Business Object.
	 * 
	 * @param document - document to clear the pk fields on
	 */
    protected void clearPrimaryKeyFields(MaintenanceDocument document) {
		// get business object being maintained and its keys
		PersistableBusinessObject bo = document.getNewMaintainableObject().getBusinessObject();
		List<String> keyFieldNames = KRADServiceLocatorWeb.getLegacyDataAdapter().listPrimaryKeyFieldNames(bo.getClass());

		for (String keyFieldName : keyFieldNames) {
			try {
				ObjectUtils.setObjectProperty(bo, keyFieldName, null);
			}
			catch (Exception e) {
				LOG.error("Unable to clear primary key field: " + e.getMessage(), e);
				throw new RuntimeException("Unable to clear primary key field: " + e.getMessage(), e);
			}
		}
        bo.setObjectId(null);
        bo.setVersionNumber(1L);
	}

	/**
	 * This method is used as part of the Copy functionality, to clear any field values that the user making the copy does not have
	 * permissions to modify. This will prevent authorization errors on a copy.
	 * 
	 * @param document - document to be adjusted
	 */
    protected void clearUnauthorizedNewFields(MaintenanceDocument document) {
		// get a reference to the current user
		Person user = GlobalVariables.getUserSession().getPerson();

		// get a new instance of MaintenanceDocumentAuthorizations for this context
		MaintenanceDocumentRestrictions maintenanceDocumentRestrictions = getBusinessObjectAuthorizationService().getMaintenanceDocumentRestrictions(document, user);

		document.getNewMaintainableObject().clearBusinessObjectOfRestrictedValues(maintenanceDocumentRestrictions);
	}

	/**
	 * This method does all special processing on a document that should happen on each HTTP post (ie, save, route, approve, etc).
	 **/
	@Override
	protected void doProcessingAfterPost( KualiForm form, HttpServletRequest request ) {
		MaintenanceDocument document = (MaintenanceDocument) ((KualiMaintenanceForm)form).getDocument();
		Maintainable maintainable = document.getNewMaintainableObject();
		Object bo = maintainable.getBusinessObject();

		getBusinessObjectService().linkUserFields(bo);

		maintainable.processAfterPost(document, request.getParameterMap() );
	}

	protected void doProcessingAfterPost( KualiForm form, Map<String,String[]> parameters ) {
		MaintenanceDocument document = (MaintenanceDocument) ((KualiMaintenanceForm)form).getDocument();
		Maintainable maintainable = document.getNewMaintainableObject();
		Object bo = maintainable.getBusinessObject();

		getBusinessObjectService().linkUserFields(bo);

		maintainable.processAfterPost(document, parameters );
	}

	@Override
    protected void populateAuthorizationFields(KualiDocumentFormBase formBase){
		super.populateAuthorizationFields(formBase);

		KualiMaintenanceForm maintenanceForm = (KualiMaintenanceForm) formBase;
		MaintenanceDocument maintenanceDocument = (MaintenanceDocument) maintenanceForm.getDocument();
		Person user = GlobalVariables.getUserSession().getPerson();
		maintenanceForm.setReadOnly(!formBase.getDocumentActions().containsKey(KRADConstants.KUALI_ACTION_CAN_EDIT));
		MaintenanceDocumentRestrictions maintenanceDocumentAuthorizations = getBusinessObjectAuthorizationService().getMaintenanceDocumentRestrictions(maintenanceDocument, user);
		maintenanceForm.setAuthorizations(maintenanceDocumentAuthorizations);
	}

	public LookupService getLookupService() {
		if ( lookupService == null ) {
			lookupService = KRADServiceLocatorWeb.getLookupService();
		}
		return this.lookupService;
	}

	public LookupResultsService getLookupResultsService() {
		if ( lookupResultsService == null ) {
			lookupResultsService = KNSServiceLocator.getLookupResultsService();
		}
		return this.lookupResultsService;
	}


	public MaintenanceDocumentDictionaryService getMaintenanceDocumentDictionaryService() {
		if (maintenanceDocumentDictionaryService == null) {
			maintenanceDocumentDictionaryService = KNSServiceLocator.getMaintenanceDocumentDictionaryService();
		}

		return maintenanceDocumentDictionaryService;
	}
}
