/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.service;

import org.kuali.rice.core.api.mo.common.GloballyUnique;
import org.kuali.rice.krad.bo.Attachment;
import org.kuali.rice.krad.bo.Note;

import java.io.IOException;
import java.io.InputStream;

/**
 * Defines the methods common to all AttachmentService implementations
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public interface AttachmentService {
    /**
     * Stores the given fileContents and returns referring Attachment object which acts as a momento to the archived object.
     *
     * @param parent - the document to which the attachment belongs
     * @param uploadedFileName - the name of the uploaded file
     * @param mimeType - the uploaded file's mime type
     * @param  fileSize - the size of the uploaded file in bytes
     * @param  fileContents - an input stream used to read the file's contents
     * @param attachmentType -the attachment type code
     *
     * @return Attachment - the attachment, having been written to the file system
     * @throws IOException
     */
    public Attachment createAttachment(GloballyUnique parent, String uploadedFileName, String mimeType, int fileSize, InputStream fileContents, String attachmentType) throws IOException;

    /**
     * Retrieves a given Attachments contents from the corresponding Attachment object
     *
     * @param attachment - the attachment whose contents are to be retrieved
     *
     * @return OutputStream
     * @throws IOException
     */
    public InputStream retrieveAttachmentContents(Attachment attachment) throws IOException;

    /**
     * Deletes a given DocumentAttachment contents from the corresponding Attachment object
     *
     * @param attachment - the attachment whose contents are to be deleted
     */
    public void deleteAttachmentContents(Attachment attachment);
    
    /**
     * 
     * Moves attachments on notes from the pending directory to the real one
     * @param note the Note from which to move attachments.  If this Note does not
     * have an attachment then this method does nothing.
     * 
     * @throws IllegalArgumentException if the given Note is null
     * @throws IllegalArgumentException if the Note does not have a valid object id
     */
    public void moveAttachmentWherePending(Note note);
    
    /**
     * Deletes pending attachments that were last modified before the given time.  Java does not have easy access to a file's creation
     * time, so we use modification time instead.
     * 
     * @param modificationTime the number of milliseconds since "the epoch" (i.e.January 1, 1970, 00:00:00 GMT).  java.util.Date and java.util.Calendar's
     *  methods return time in this format.  If a pending attachment was modified before this time, then it will be deleted (unless an error occurs)
     */
    public void deletePendingAttachmentsModifiedBefore(long modificationTime);
    
    public Attachment getAttachmentByNoteId(Long noteId);
}
