/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad;

import org.kuali.rice.kew.api.KewApiConstants;
import org.kuali.rice.kew.api.WorkflowDocument;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Utility class for working with the UserSession.
 */
public final class UserSessionUtils {

    private UserSessionUtils() {
        throw new IllegalStateException("this class should not be instantiated");
    }

    /**
     * Adds the given {@link org.kuali.rice.kew.api.WorkflowDocument} to the {@link org.kuali.rice.krad.UserSession}.
     * @param userSession the session to add the workflow document to
     * @param workflowDocument the workflow doc to add to the session
     */
    public static void addWorkflowDocument(UserSession userSession, WorkflowDocument workflowDocument) {
        Map<String, WorkflowDocument> workflowDocMap = getWorkflowDocumentMap(userSession);

        workflowDocMap.put(workflowDocument.getDocumentId(), workflowDocument);
    }

    /**
     * Returns the {@link org.kuali.rice.kew.api.WorkflowDocument} with the given ID from the
     * {@link org.kuali.rice.krad.UserSession}.  If there is not one cached in the session with
     * that ID, then null is returned.
     * @param userSession the user session from which to retrieve the workflow document
     * @param workflowDocumentId the ID of the workflow document to get
     * @return the cached workflow document, or null if a document with that ID is not cached in the user session
     */
    public static WorkflowDocument getWorkflowDocument(UserSession userSession, String workflowDocumentId) {
        Map<String, WorkflowDocument> workflowDocMap = getWorkflowDocumentMap(userSession);

        return workflowDocMap.get(workflowDocumentId);
    }

    /**
     * Returns the map of workflow document IDs to {@link org.kuali.rice.kew.api.WorkflowDocument}, making sure to
     * initialize in a thread-safe way if the map does not exist.
     *
     * <p>
     * We assume the {@link org.kuali.rice.krad.UserSession} is not null here.
     * </p>
     * @param userSession the user session from which to retrieve the workflow document
     * @return the map of workflow document IDs to workflow documents
     */
    @SuppressWarnings("unchecked")
    private static Map<String, WorkflowDocument> getWorkflowDocumentMap(UserSession userSession) {
        userSession.addObjectIfAbsent(
                KewApiConstants.WORKFLOW_DOCUMENT_MAP_ATTR_NAME, new ConcurrentHashMap<String, WorkflowDocument>());

        return (ConcurrentMap<String, WorkflowDocument>) userSession.retrieveObject(
                KewApiConstants.WORKFLOW_DOCUMENT_MAP_ATTR_NAME);
    }
}
