/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data.provider;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.krad.data.KradDataServiceLocator;
import org.springframework.beans.factory.InitializingBean;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Provides a mechanism for registering Providers in the {@link ProviderRegistry} using Spring.
 *
 * @see Provider
 * @see ProviderRegistry
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class ProviderRegistrar implements InitializingBean {

    private static final Logger LOG = LogManager.getLogger(ProviderRegistrar.class);

    /**
     * The provider registry.
     */
    protected ProviderRegistry providerRegistry;

    /**
     * The providers currently assigned to the registry.
     */
    protected List<Provider> providers = Collections.unmodifiableList(Collections.<Provider>emptyList());

    /**
     * {@inheritDoc}
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        if ( getProviders() != null ) {
            if ( getProviderRegistry() != null ) {
                for ( Provider provider : getProviders() ) {
                    LOG.info( "Registering data module provider for "+ provider);
                    getProviderRegistry().registerProvider(provider);
                }
            } else {
                LOG.error( "Provider registry not initialized.");
            }
        }
    }

    /**
     * Sets the list of providers for this module.
     *
     * @param providers list of providers
     */
    public void setProviders(List<Provider> providers) {
        this.providers = Collections.unmodifiableList(new ArrayList<Provider>(providers));
    }

    /**
     * Gets the list of providers for this module.
     *
     * @return the list of providers for this module.
     */
    public List<Provider> getProviders() {
        return providers;
    }

    /**
     * Gets the provider registry.
     *
     * @return the provider registry.
     */
    public ProviderRegistry getProviderRegistry() {
        if(this.providerRegistry == null){
            return KradDataServiceLocator.getProviderRegistry();
        }
        return this.providerRegistry;
    }

    /**
     * Setter for the provider registry.
     *
     * @param providerRegistry the provider registry to set.
     */
    public void setProviderRegistry(ProviderRegistry providerRegistry) {
        this.providerRegistry = providerRegistry;
    }

}
