/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.config;

import org.kuali.rice.core.api.config.module.RunMode;
import org.kuali.rice.core.api.config.property.ConfigContext;
import org.kuali.rice.core.api.resourceloader.GlobalResourceLoader;
import org.kuali.rice.core.framework.config.module.ModuleConfigurer;
import org.kuali.rice.krad.service.DataDictionaryService;
import org.kuali.rice.krad.service.KRADServiceLocatorWeb;
import org.kuali.rice.krad.util.KRADConstants;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.SmartApplicationListener;

import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Rice module configurer for KRAD
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class KRADConfigurer extends ModuleConfigurer implements SmartApplicationListener {

    private DataSource applicationDataSource;

    private boolean includeKnsSpringBeans;

    private static final String KRAD_SPRING_BEANS_PATH = "classpath:org/kuali/rice/krad/config/KRADSpringBeans.xml";
    private static final String KNS_SPRING_BEANS_PATH = "classpath:org/kuali/rice/kns/config/KNSSpringBeans.xml";

    public KRADConfigurer() {
        // TODO really the constant value should be "krad" but there's some work to do in order to make
        // that really work, see KULRICE-6532
        super(KRADConstants.KR_MODULE_NAME);
        setValidRunModes(Arrays.asList(RunMode.LOCAL));
        setIncludeKnsSpringBeans(true);
    }

    @Override
    public void addAdditonalToConfig() {
        // export the state of KNS enablement to the global config
        ConfigContext.getCurrentContextConfig().putProperty(KRADConstants.Config.KNS_ENABLED, Boolean.valueOf(isIncludeKnsSpringBeans()).toString());
        configureDataSource();
    }

    @Override
    public List<String> getPrimarySpringFiles() {
        LOG.info("KRADConfigurer:getPrimarySpringFiles: getRunMode => " + getRunMode());
        final List<String> springFileLocations = new ArrayList<String>();
        springFileLocations.add(KRAD_SPRING_BEANS_PATH);

        if (isIncludeKnsSpringBeans()) {
            springFileLocations.add(KNS_SPRING_BEANS_PATH);
        }

        return springFileLocations;
    }

    @Override
    public void onApplicationEvent(ApplicationEvent applicationEvent) {
        if (applicationEvent instanceof ContextRefreshedEvent) {
            loadDataDictionary();
        }
    }

    @Override
    public boolean supportsEventType(Class<? extends ApplicationEvent> aClass) {
        return true;
    }

    @Override
    public boolean supportsSourceType(Class<?> aClass) {
        return true;
    }

    @Override
    public int getOrder() {
        // return a lower value which will give the data dictionary indexing higher precedence since DD indexing should
        // be started as soon as it can be
        return -1000;
    }

    /**
     * Used to "poke" the Data Dictionary again after the Spring Context is initialized.  This is to
     * allow for modules loaded with KualiModule after the KNS has already been initialized to work.
     *
     * Also initializes the DateTimeService
     */
    protected void loadDataDictionary() {
        if (isLoadDataDictionary()) {
            LOG.info("KRAD Configurer - Loading DD");
            DataDictionaryService dds = KRADServiceLocatorWeb.getDataDictionaryService();
            if (dds == null) {
                dds = (DataDictionaryService) GlobalResourceLoader.getService(
                        KRADServiceLocatorWeb.DATA_DICTIONARY_SERVICE);
            }
            dds.getDataDictionary().parseDataDictionaryConfigurationFiles();

            if (isValidateDataDictionary()) {
                LOG.info("KRAD Configurer - Validating DD");
                dds.getDataDictionary().validateDD();
            }

            // KULRICE-4513 After the Data Dictionary is loaded and validated, perform Data Dictionary bean overrides.
            if (isInterfaceBasedOverridesDataDictionary()) {
                dds.getDataDictionary().performBeanOverrides();
            }
        }
    }

    @Override
    public boolean hasWebInterface() {
        return true;
    }

    /**
     * Returns true - KNS UI should always be included.
     *
     * @see org.kuali.rice.core.framework.config.module.ModuleConfigurer#shouldRenderWebInterface()
     */
    @Override
    public boolean shouldRenderWebInterface() {
        return true;
    }

    public boolean isLoadDataDictionary() {
        return ConfigContext.getCurrentContextConfig().getBooleanProperty("load.data.dictionary", true);
    }

    public boolean isValidateDataDictionary() {
        return ConfigContext.getCurrentContextConfig().getBooleanProperty("validate.data.dictionary", false);
    }

    public boolean isInterfaceBasedOverridesDataDictionary() {
        return ConfigContext.getCurrentContextConfig().getBooleanProperty("interface.based.overrides.data.dictionary", false);
    }

    /**
     * Used to "poke" the Data Dictionary again after the Spring Context is initialized.  This is to
     * allow for modules loaded with KualiModule after the KNS has already been initialized to work.
     *
     * Also initializes the DateTimeService
     */
    protected void configureDataSource() {
        if (getApplicationDataSource() != null) {
            ConfigContext.getCurrentContextConfig().putObject(KRADConstants.KRAD_APPLICATION_DATASOURCE,
                    getApplicationDataSource());
        }
    }

    public DataSource getApplicationDataSource() {
        return this.applicationDataSource;
    }

    public void setApplicationDataSource(DataSource applicationDataSource) {
        this.applicationDataSource = applicationDataSource;
    }

    /**
     * Indicates whether the legacy KNS module should be included which will include
     * the KNS spring beans file
     *
     * @return boolean true if kns should be supported, false if not
     */
    public boolean isIncludeKnsSpringBeans() {
        return includeKnsSpringBeans;
    }

    /**
     * Setter for the include kns support indicator
     *
     * @param includeKnsSpringBeans
     */
    public void setIncludeKnsSpringBeans(boolean includeKnsSpringBeans) {
        this.includeKnsSpringBeans = includeKnsSpringBeans;
    }
}
