/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.uif.field;

import org.kuali.rice.krad.uif.component.Component;
import org.kuali.rice.krad.uif.element.Label;

import java.util.List;

/**
 * Component that contains one or more user interface elements and can be placed
 * into a <code>Container</code>
 *
 * <p>
 * Provides a wrapper for various user interface elements so they can be treated
 * uniformly by a container and rendered using a <code>LayoutManager</code>.
 * Implementations exist for various types of elements and properties to
 * configure that element.
 * </p>
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public interface Field extends Component {

	/**
	 * Label text for the field
	 *
	 * <p>
	 * The label is generally used to identify the field in the user interface
	 * </p>
	 *
	 * @return String label text
	 */
	public String getLabel();

	/**
	 * Setter for the field's label text
	 *
	 */
	public void setLabel(String labelText);

	/**
	 * Short label for the field
	 *
	 * <p>
	 * For areas of the user interface that have limited area (such as table
	 * headers), the short label can be used to identify the field
	 * </p>
	 *
	 * @return String short label
	 */
	public String getShortLabel();

	/**
	 * Setter for the field's short label text
	 *
	 * @param shortLabel
	 */
	public void setShortLabel(String shortLabel);

	/**
	 * <code>Label</code> instance for the field
	 *
	 * <p>
	 * The label field contains the labeling text for the field in addition to
	 * configuration for rendering in the user interface (such as the styling
	 * for the label area)
	 * </p>
	 *
	 * @return Label instance
	 */
	public Label getFieldLabel();

	/**
	 * Setter for the field's label field
	 *
	 */
	public void setFieldLabel(Label label);

    /**
     * The label for this field will be positioned to the left of the field content, when true.
     *
     * <p>The usage of this option is for one-off instances in which a label must be positioned to the left of a field
     * for layout purposes. It is recommended that if many fields will have their labels positioned to the left
     * in a group that a CssGridLabelColumnLayoutManager backed group be used instead.</p>
     */
    public boolean isLabelLeft();

    /**
     * @see Field#isLabelLeft()
     */
    public void setLabelLeft(boolean labelLeft);

	/**
	 * Indicates whether the contained <code>Label</code> has been rendered
	 * as part of another field and thus should not be rendered with the
	 * attribute
	 *
	 * @return boolean true if the label field has been rendered, false if it
	 *         should be rendered with the attribute
	 */
	public boolean isLabelRendered();

	/**
	 * Setter for the label field rendered indicator
	 *
	 */
	public void setLabelRendered(boolean labelFieldRendered);

    /**
     * Label style classes for the field
     *
     * @return List&lt;String&gt; label style classes
     */
    public List<String> getLabelStyleClasses();

    /**
     * Setter for the field's label style classes
     *
     */
    public void setLabelStyleClasses(List<String> labelStyleClasses);

    /**
     * Label column span for the field
     *
     * @return int label column span
     */
    public int getLabelColSpan();

    /**
     * Setter for the field's label column span
     *
     */
    public void setLabelColSpan(int labelColSpan);

    /**
     * Field Security object that indicates what authorization (permissions) exist for the field
     *
     * @return FieldSecurity instance
     */
    public FieldSecurity getFieldSecurity();

}
