/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.uif.freemarker;

import java.util.Collections;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.krad.uif.component.Component;
import org.kuali.rice.krad.uif.lifecycle.ViewLifecycle;
import org.kuali.rice.krad.uif.lifecycle.ViewLifecycleTaskBase;

import freemarker.core.Environment;
import freemarker.core.Macro;
import freemarker.template.TemplateModel;

/**
 * Perform actual rendering on a component during the lifecycle.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class RenderComponentTask extends ViewLifecycleTaskBase<Component> {

    private static final Logger LOG = LogManager.getLogger(RenderComponentTask.class);

    /**
     * Constructor.
     * 
     * @param phase The render phase for the component.
     */
    public RenderComponentTask() {
        super(Component.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void performLifecycleTask() {
        Component component = (Component) getElementState().getElement();
        if (!component.isRender() || component.getTemplate() == null) {
            return;
        }
        
        LifecycleRenderingContext renderingContext = ViewLifecycle.getRenderingContext();
        renderingContext.clearRenderingBuffer();

        renderingContext.importTemplate(component.getTemplate());

        for (String additionalTemplate : component.getAdditionalTemplates()) {
            renderingContext.importTemplate(additionalTemplate);
        }

        try {
            Environment env = renderingContext.getEnvironment();

            // Check for a single-arg macro, with the parameter name "component"
            // defer for parent rendering if not found
            Macro fmMacro = (Macro) env.getMainNamespace().get(component.getTemplateName());

            if (fmMacro == null) {
                return;
            }

            String[] args = fmMacro.getArgumentNames();
            if (args == null || args.length != 1 || !component.getComponentTypeName().equals(args[0])) {
                return;
            }

            FreeMarkerInlineRenderUtils.renderTemplate(env, component,
                    null, false, false, Collections.<String, TemplateModel> emptyMap());

            component.setRenderedHtmlOutput(renderingContext.getRenderedOutput());
            component.setSelfRendered(true);
        } catch (Throwable e) {
            if (ViewLifecycle.isStrict()) {
                LOG.warn("Error rendering component during lifecycle phase " + getElementState()
                        + " falling back to higher level rendering", e);
            } else if (ViewLifecycle.isTrace() && LOG.isDebugEnabled()) {
                LOG.debug("component rendering failed during lifecycle phase " + getElementState()
                        + " falling back to higher level rendering", e);
            }
        }

    }
}
