/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krms.framework.engine;

import java.util.Collection;

import org.kuali.rice.krms.api.engine.ExecutionEnvironment;
import org.kuali.rice.krms.api.engine.ResultEvent;
import org.kuali.rice.krms.api.engine.Term;
import org.kuali.rice.krms.api.engine.TermResolutionException;
import org.kuali.rice.krms.framework.engine.expression.ComparisonOperator;
import org.kuali.rice.krms.framework.engine.result.BasicResult;

public class CollectionOfComparablesTermBasedProposition<T> extends ComparableTermBasedProposition<T> {
	private static final ResultLogger LOG = ResultLogger.getInstance();

	private CollectionOperator collectionOper;
	private Term term;

	public CollectionOfComparablesTermBasedProposition(CollectionOperator collectionOper, ComparisonOperator compareOper, Term term, T expectedValue) {
		super(compareOper, term, expectedValue);
		this.term = term;
		this.collectionOper = collectionOper;
	}

	/**
	 * @see org.kuali.rice.krms.framework.engine.ComparableTermBasedProposition#evaluate(org.kuali.rice.krms.api.engine.ExecutionEnvironment)
	 * @throws TermResolutionException if there is a problem resolving a {@link Term}
	 */
	@Override
	public PropositionResult evaluate(ExecutionEnvironment environment) {
		boolean collatedResult = collectionOper.getInitialCollatedResult();

		Collection<? extends Comparable<T>> termValue;

		termValue = environment.resolveTerm(term, this);

		if (termValue != null) {
			for (Comparable<T> item : termValue) {
				collatedResult = collectionOper.reduce(compare(item), collatedResult);
				if (collectionOper.shortCircuit(collatedResult)) break;
			}
		}

		// TODO: log this appropriately
		if (LOG.isEnabled(environment)) {
			LOG.logResult(new BasicResult(ResultEvent.PROPOSITION_EVALUATED, this, environment, collatedResult));
		}

		return new PropositionResult(collatedResult);
	}

	@Override
    public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append(collectionOper.toString());
		sb.append(" "+super.toString());
		return sb.toString();
	}
}
