/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krms.framework.engine;

/**
 * <p>Enumeration for simple collection operators used by {@link CollectionOfComparablesTermBasedProposition}.  The
 * operators encapsulate logic for how to collate results and when to short circuit as a collection is being
 * processed.  Correct usage is best summarized by this code block:</p>
 * <pre>
 * for (Comparable&lt;T&gt; item : comparableItems) {
 *     collatedResult = collectionOper.reduce(compare(item, compareValue), collatedResult);
 *     if (collectionOper.shortCircuit(collatedResult)) break;
 * }
 * </pre>
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public enum CollectionOperator {

	ONE_OR_MORE(false) {
		@Override
		public boolean reduce(boolean elementResult, boolean collatedResult) {
			return elementResult || collatedResult;
		}
		
		@Override
		public boolean shortCircuit(boolean collatedResult) {
			return collatedResult;
		}
	},
	
	ALL(true) {
		@Override
		public boolean reduce(boolean elementResult, boolean collatedResult) {
			return elementResult && collatedResult;
		}

		@Override
		public boolean shortCircuit(boolean collatedResult) {
			return !collatedResult;
		}
	},
	
	NONE(true) {
		@Override
		public boolean reduce(boolean elementResult, boolean collatedResult) {
			return !elementResult && collatedResult;
		}

		@Override
		public boolean shortCircuit(boolean collatedResult) {
			return !collatedResult;
		}
	};
	
	private final boolean initialCollationResult;
	
	private CollectionOperator(boolean initialCollationResult) {
		this.initialCollationResult = initialCollationResult;
	}
	
	/**
	 * This method takes the collated result thus far and the result for the next element,
	 * and produces the next collated result.
	 * 
	 * @return the new collated result
	 */
	public abstract boolean reduce(boolean elementResult, boolean collatedResult);
	
	/**
	 * This method lets the engine know if it can short circuit its iteration through the list based on the 
	 * collated result.  The condition when short circuiting can be done varies with the operator.
	 * 
	 * @param collatedResult
	 * @return true if short circuiting can be done to optimize processing
	 */
	public abstract boolean shortCircuit(boolean collatedResult);
	
	/**
	 * when the result for the first item in the collection is calculated, there isn't yet a collated result 
	 * to use in the {@link #reduce(boolean, boolean)} method.  Different operators require different
	 * initial values to function correctly, so this property holds the correct initial collated value for the 
	 * given operator instance.
	 */
	public boolean getInitialCollatedResult() {
		return initialCollationResult;
	}
	
}
