/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.impl.bus;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.config.CoreConfigHelper;
import org.kuali.rice.core.api.config.property.ConfigContext;
import org.kuali.rice.core.api.util.RiceUtilities;
import org.kuali.rice.ksb.util.KSBConstants;
import org.springframework.beans.factory.config.AbstractFactoryBean;

/**
 * A factory bean which is used to produce an instance id for the service bus.
 * 
 * <p>The value for the instance id is determined in the following order of preferences:
 * 
 * <ol>
 *   <li>If {@code instanceId} is set on this factory bean, that value will be used.</li> 
 *   <li>If {@link KSBConstants.Config#INSTANCE_ID} has been set in the configuration context, that value will be used.</li>
 *   <li>If none of the above, the instance id will be a combination of this application's namespace plus ip address.</li>
 * </ol>
 * 
 * <p>In the case that the instance id is generated, the application id will be pulled
 * from the configuration context using the {@link KSBConstants.Config#INSTANCE_ID} configuration parameter which
 * should always have a value.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public class InstanceIdFactoryBean extends AbstractFactoryBean<String> {

	private String instanceId;
	
	public void setInstanceId(String instanceId) {
		this.instanceId = instanceId;
	}
	
	@Override
	protected String createInstance() throws Exception {
		if (StringUtils.isBlank(instanceId)) {
			this.instanceId = loadInstanceIdFromConfig();
		}
		if (StringUtils.isBlank(instanceId)) {
			String applicationId = CoreConfigHelper.getApplicationId();
			String ipNumber = RiceUtilities.getIpNumber();
			this.instanceId = applicationId + "-" + ipNumber;
		}
        ConfigContext.getCurrentContextConfig().putProperty(KSBConstants.Config.INSTANCE_ID, this.instanceId);
		return this.instanceId;
	}
	
	protected String loadInstanceIdFromConfig() {
		return ConfigContext.getCurrentContextConfig().getProperty(KSBConstants.Config.INSTANCE_ID);
	}

	@Override
	public Class<String> getObjectType() {
		return String.class;
	}

	
	
}
