/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.config.property.ConfigContext;
import org.kuali.rice.ksb.messaging.service.MessageQueueService;
import org.kuali.rice.ksb.service.KSBServiceLocator;
import org.kuali.rice.ksb.util.KSBConstants;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallback;

/**
 * Fetches messages from the db. Marks as 'R'. Gives messages to ThreadPool for execution
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class MessageFetcher implements Runnable {

    private static final Logger LOG = LogManager.getLogger(MessageFetcher.class);

    private Integer maxMessages;
    private Long routeQueueId;

    public MessageFetcher(Integer maxMessages) {
        this.maxMessages = maxMessages;
    }

    public MessageFetcher(Long routeQueueId) {
        this.routeQueueId = routeQueueId;
    }

    @Override
    public void run() {
        if (ConfigContext.getCurrentContextConfig().getBooleanProperty(KSBConstants.Config.MESSAGE_PERSISTENCE,
                false)) {
            try {
                requeueDocument();
                requeueMessages();
            } catch (Throwable t) {
                LOG.error("Failed to fetch messages.", t);
            }
        }
    }

    private void requeueMessages() {
        if (this.routeQueueId == null) {
            try {
                for (PersistedMessageBO message : getRouteQueueService().getNextDocuments(maxMessages)) {
                    message = markEnrouteAndSaveMessage(message);
                    executeMessage(message);
                }
            } catch (Throwable t) {
                LOG.error("Failed to fetch or process some messages during requeueMessages", t);
            }
        }
    }

    private void requeueDocument() {
        try {
            if (this.routeQueueId != null) {
                PersistedMessageBO message = getRouteQueueService().findByRouteQueueId(this.routeQueueId);
                message.setQueueStatus(KSBConstants.ROUTE_QUEUE_ROUTING);
                message = getRouteQueueService().save(message);
                executeMessage(message);
            }
        } catch (Throwable t) {
            LOG.error("Failed to fetch or process some messages during requeueDocument", t);
        }
    }

    private void executeMessage(PersistedMessageBO message) {
        try {
            KSBServiceLocator.getThreadPool().execute(new MessageServiceInvoker(message));
        } catch (Throwable t) {
            LOG.error("Failed to place message " + message + " in thread pool for execution", t);
        }
    }

    private PersistedMessageBO markEnrouteAndSaveMessage(final PersistedMessageBO message) {
        try {
            return KSBServiceLocator.getTransactionTemplate().execute(new TransactionCallback<PersistedMessageBO>() {
                @Override
                public PersistedMessageBO doInTransaction(TransactionStatus status) {
                    message.setQueueStatus(KSBConstants.ROUTE_QUEUE_ROUTING);
                    return getRouteQueueService().save(message);
                }
            });
        } catch (Throwable t) {
            LOG.error("Caught error attempting to mark message " + message + " as R", t);
        }
        return message;
    }

    private MessageQueueService getRouteQueueService() {
        return KSBServiceLocator.getMessageQueueService();
    }

}
