/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging.bam;

import org.kuali.rice.core.api.config.property.Config;
import org.kuali.rice.core.api.config.property.ConfigContext;
import org.kuali.rice.core.api.util.ContextClassLoaderProxy;
import org.kuali.rice.core.api.util.reflect.BaseTargetedInvocationHandler;
import org.kuali.rice.ksb.api.bus.ServiceDefinition;
import org.kuali.rice.ksb.messaging.bam.service.BAMService;
import org.kuali.rice.ksb.service.KSBServiceLocator;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;


/**
 * A service-side proxy for that records an entry in the BAM for invocations
 * on the proxied service endpoint.
 *
 * @see BAMService
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */

public class BAMServerProxy extends BaseTargetedInvocationHandler<Object> {

	private ServiceDefinition serviceDefinition;
	
	private BAMServerProxy(Object target, ServiceDefinition serviceDefinition) {
		super(target);
		this.serviceDefinition = serviceDefinition;
	}
	
	public static boolean isBamSupported() {
		return KSBServiceLocator.getBAMService() != null && Boolean.valueOf(ConfigContext.getCurrentContextConfig().getProperty(Config.BAM_ENABLED));
	}
	
	public static Object wrap(Object target, ServiceDefinition serviceDefinition) {
		if (!isBamSupported()) {
			return target;
		}
		return Proxy.newProxyInstance(target.getClass().getClassLoader(), ContextClassLoaderProxy.getInterfacesToProxy(
                target), new BAMServerProxy(target, serviceDefinition));
	}
	
	@Override
    protected Object invokeInternal(Object proxiedObject, Method method, Object[] arguments) throws Throwable {
		BAMTargetEntry bamTargetEntry = KSBServiceLocator.getBAMService().recordServerInvocation(getTarget(), this.serviceDefinition, method, arguments);
		try {
			return method.invoke(getTarget(), arguments);	
		} catch (Throwable throwable) {
			if (throwable instanceof InvocationTargetException) {
				throwable = throwable.getCause();
			}
			KSBServiceLocator.getBAMService().recordServerInvocationError(throwable, bamTargetEntry);
			throw throwable;
		}
	}
}
