/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging.dao.impl;

import org.kuali.rice.core.api.config.CoreConfigHelper;
import org.kuali.rice.core.api.exception.RiceRuntimeException;
import org.kuali.rice.core.api.util.RiceUtilities;
import org.kuali.rice.ksb.messaging.PersistedMessageBO;
import org.kuali.rice.ksb.messaging.dao.MessageQueueDAO;
import org.kuali.rice.ksb.util.KSBConstants;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceException;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.List;
import java.util.Map;

public class MessageQueueDaoJpa implements MessageQueueDAO {

    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(MessageQueueDaoJpa.class);

    private EntityManager entityManager;

    @Override
    public List<PersistedMessageBO> findAll(int maxRows) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Finding next " + maxRows + " messages");
        }

        TypedQuery<PersistedMessageBO> query = entityManager.createNamedQuery("PersistedMessageBO.FindAll",
                PersistedMessageBO.class);
        query.setMaxResults(maxRows);

        return query.getResultList();
    }

    @Override
    public PersistedMessageBO findByRouteQueueId(Long routeQueueId) {
        return entityManager.find(PersistedMessageBO.class, routeQueueId);
    }

    @Override
    public List<PersistedMessageBO> findByValues(Map<String, String> criteriaValues, int maxRows) {
        CriteriaBuilder builder = entityManager.getCriteriaBuilder();
        CriteriaQuery<PersistedMessageBO> query = builder.createQuery(PersistedMessageBO.class);
        Root<PersistedMessageBO> message = query.from(PersistedMessageBO.class);
        Predicate predicate = builder.conjunction();
        for (Map.Entry<String, String> entry : criteriaValues.entrySet()) {
            predicate = builder.and(predicate, builder.equal(message.get(entry.getKey()), entry.getValue()));
        }
        query.where(predicate);
        TypedQuery<PersistedMessageBO> typedQuery = entityManager.createQuery(query);
        return typedQuery.getResultList();
    }

    @Override
    public List<PersistedMessageBO> getNextDocuments(Integer maxDocuments) {
        String applicationId = CoreConfigHelper.getApplicationId();

        TypedQuery<PersistedMessageBO> query = entityManager.createNamedQuery("PersistedMessageBO.GetNextDocuments",
                PersistedMessageBO.class);
        query.setParameter("applicationId", applicationId);
        query.setParameter("queueStatus", KSBConstants.ROUTE_QUEUE_EXCEPTION);
        query.setParameter("ipNumber", RiceUtilities.getIpNumber());

        if (maxDocuments != null) {
            query.setMaxResults(maxDocuments);
        }

        return query.getResultList();
    }

    @Override
    public void remove(final PersistedMessageBO routeQueue) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Removing message " + routeQueue);
        }
        if (routeQueue.getRouteQueueId() == null) {
            throw new RiceRuntimeException("can't delete a PersistedMessageBO with no id");
        }

        evict(routeQueue);
        entityManager.remove(entityManager.merge(routeQueue));
    }

    @Override
    public PersistedMessageBO save(PersistedMessageBO routeQueue) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Persisting message " + routeQueue);
        }

        routeQueue = entityManager.merge(routeQueue);

        entityManager.flush();
        evict(routeQueue);

        return routeQueue;
    }

    /**
     * similar to JpaPersistenceProvider.
     */
    private void evict(Object entity) {
        if (entityManager.getEntityManagerFactory().getCache() != null) {
            try {
                final Object Key = entityManager.getEntityManagerFactory().getPersistenceUnitUtil().getIdentifier(entity);
                if (Key != null) {
                    entityManager.getEntityManagerFactory().getCache().evict(entity.getClass(), Key);
                }
            } catch (PersistenceException ex) {
                LOG.warn(ex.getMessage(), ex);
                // JPA fails if it can't create the key field classes - we just need to catch and ignore here
            }
        }
    }

    public EntityManager getEntityManager() {
        return this.entityManager;
    }

    public void setEntityManager(EntityManager entityManager) {
        this.entityManager = entityManager;
    }

}
