/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging.exceptionhandling;

import javax.xml.namespace.QName;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.exception.RiceRuntimeException;
import org.kuali.rice.core.api.reflect.ObjectDefinition;
import org.kuali.rice.core.api.resourceloader.GlobalResourceLoader;
import org.kuali.rice.ksb.api.KsbApiServiceLocator;
import org.kuali.rice.ksb.api.bus.Endpoint;
import org.kuali.rice.ksb.api.bus.ServiceBus;
import org.kuali.rice.ksb.messaging.PersistedMessageBO;
import org.kuali.rice.ksb.messaging.quartz.MessageServiceExecutorJob;
import org.kuali.rice.ksb.service.KSBServiceLocator;
import org.quartz.JobDataMap;
import org.quartz.Scheduler;
import org.quartz.impl.JobDetailImpl;
import org.quartz.impl.triggers.SimpleTriggerImpl;


/**
 * Default implementation of {@link ExceptionRoutingService}.  Just saves 
 * the message in the queue as is, which should be marked Exception by the 
 * {@link MessageExceptionHandler}.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public class DefaultExceptionServiceImpl implements ExceptionRoutingService {
	
	private static final Logger LOG = LogManager.getLogger(DefaultExceptionServiceImpl.class);

	@Override
    public void placeInExceptionRouting(Throwable throwable, PersistedMessageBO message) throws Exception {
		LOG.error("Exception caught processing message " + message.getRouteQueueId() + " " + message.getServiceName() + ": " + throwable);

		MessageExceptionHandler exceptionHandler = getMessageExceptionHandler(message.getMethodCall().getServiceConfiguration().getServiceName());
		exceptionHandler.handleException(throwable, message);
	}
	
	@Override
    public void placeInExceptionRoutingLastDitchEffort(Throwable throwable, PersistedMessageBO message) throws Exception {
		LOG.error("Exception caught processing message " + message.getRouteQueueId() + " " + message.getServiceName() + ": " + throwable);
		MessageExceptionHandler exceptionHandler = getMessageExceptionHandler(message.getMethodCall().getServiceConfiguration().getServiceName());
		exceptionHandler.handleExceptionLastDitchEffort(throwable, message);
	}
	
	protected MessageExceptionHandler getMessageExceptionHandler(QName serviceName) {
		ServiceBus serviceBus = KsbApiServiceLocator.getServiceBus();
		Endpoint endpoint = serviceBus.getEndpoint(serviceName);
		if (endpoint == null) {
			throw new RiceRuntimeException("No services found for name " + serviceName);
		}
		String messageExceptionHandlerName = endpoint.getServiceConfiguration().getMessageExceptionHandler();
		if (messageExceptionHandlerName == null) {
			messageExceptionHandlerName = DefaultMessageExceptionHandler.class.getName();
		}
		return (MessageExceptionHandler) GlobalResourceLoader.getObject(new ObjectDefinition(messageExceptionHandlerName));
	}
	
	

	@Override
    public void scheduleExecution(Throwable throwable, PersistedMessageBO message, String description) throws Exception {
		KSBServiceLocator.getMessageQueueService().delete(message);
        PersistedMessageBO messageCopy = message.copy();
		Scheduler scheduler = KSBServiceLocator.getScheduler();
		JobDataMap jobData = new JobDataMap();
		jobData.put(MessageServiceExecutorJob.MESSAGE_KEY, messageCopy);
		JobDetailImpl jobDetail = new JobDetailImpl("Exception_Message_Job " + Math.random(), "Exception Messaging",
			MessageServiceExecutorJob.class);
		jobDetail.setJobDataMap(jobData);
		if (!StringUtils.isBlank(description)) {
		    jobDetail.setDescription(description);
		}

		SimpleTriggerImpl trigger = new SimpleTriggerImpl("Exception_Message_Trigger " + Math.random(), "Exception Messaging", messageCopy
			.getQueueDate());
		trigger.setJobDataMap(jobData);// 1.6 bug required or derby will choke
		scheduler.scheduleJob(jobDetail, trigger);    
	}
		
}
