/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.rice.ksb.messaging.serviceproxies;

import org.apache.commons.lang3.StringUtils;
import org.kuali.rice.core.api.config.property.ConfigurationService;
import org.kuali.rice.ksb.api.bus.Endpoint;
import org.kuali.rice.ksb.api.bus.ServiceBus;
import org.kuali.rice.ksb.api.bus.ServiceConfiguration;
import org.kuali.rice.ksb.api.messaging.AsyncCapableService;
import org.kuali.rice.ksb.api.messaging.AsynchronousCall;
import org.kuali.rice.ksb.api.messaging.CallMetadata;
import org.kuali.rice.ksb.messaging.MessageServiceInvoker;
import org.kuali.rice.ksb.messaging.PersistedMessageBO;
import org.kuali.rice.ksb.messaging.service.MessageQueueService;
import org.kuali.rice.ksb.util.KSBConstants;

import javax.xml.namespace.QName;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class AsyncCapableServiceImpl implements AsyncCapableService {
    private ServiceBus serviceBus;
    private MessageQueueService messageQueueService;
    private ConfigurationService configurationService;

    /**
     * Executes a call.
     *
     * If messaging is on the call will be executed via the ksb messaging framework using value1, value2, and the callMetadata.
     *
     * If messaging is off then the call will be executed immediately by invoking the synchronousCall.
     * It will NOT use the ksb messaging framework.
     *
     * If messaging is on and messaging delivery synchronous is off then the messaging will be persisted and executed asynchronously after the current
     * database transaction is committed.  It will use the ksb messaging framework using value1, value2, and the callMetadata.
     *
     * If messaging is on and messaging delivery synchronous is on then the messaging will be persisted and executed immediately and synchronously.
     * It will use the ksb messaging framework using value1, value2, and the callMetadata.
     */
    @Override
    public void executeCall(QName qname, String applicationId, String value1, String value2, CallMetadata callMetadata, Runnable synchronousCall) {
        if (qname == null) {
            throw new IllegalArgumentException("qname is null");
        }

        if (callMetadata == null) {
            throw new IllegalArgumentException("callMetadata is null");
        }

        if (synchronousCall == null) {
            throw new IllegalArgumentException("synchronousCall is null");
        }


        if (messagingOn()) {
            final List<ServiceConfiguration> configs = findApplicableEndpoints(qname, applicationId)
                    .stream()
                    .map(Endpoint::getServiceConfiguration)
                    .collect(Collectors.toList());

            //assumes they are either all queue or all topic
            boolean queue = configs.stream().anyMatch(ServiceConfiguration::isQueue);

            final Stream<PersistedMessageBO> messages;
            if (queue) {
                messages = configs
                        .stream()
                        .findFirst()
                        .stream()
                        .map(config -> toMessage(config, value1, value2, new AsynchronousCall(callMetadata.getParamTypes(), callMetadata.getArguments(), config, callMetadata.getMethodName())));
            } else {
                messages = configs
                        .stream()
                        .map(config -> toMessage(config, value1, value2, new AsynchronousCall(callMetadata.getParamTypes(), callMetadata.getArguments(), config, callMetadata.getMethodName())));
            }

            final List<PersistedMessageBO> savedMessages = messages
                    .map(message -> getMessageQueueService().save(message))
                    .collect(Collectors.toList());

            savedMessages.forEach(this::sendMessage);
        } else {
            synchronousCall.run();
        }
    }

    private void sendMessage(PersistedMessageBO message) {
        if (asyncModeOn()) {
            sendMessageAsynchronous(message);
        } else {
            sendMessageImmediate(message);
        }
    }

    private void sendMessageAsynchronous(PersistedMessageBO message) {
        MessageSender.sendMessage(message);
    }

    private void sendMessageImmediate(PersistedMessageBO message) {
        new MessageServiceInvoker(message).run();
    }

    private PersistedMessageBO toMessage(ServiceConfiguration serviceConfiguration, String value1, String value2, AsynchronousCall callData) {
        final PersistedMessageBO message = PersistedMessageBO.buildMessage(serviceConfiguration, callData);
        message.setQueueStatus(KSBConstants.ROUTE_QUEUE_ROUTING);
        message.setValue1(StringUtils.substring(value1, 0, 2000));
        message.setValue2(StringUtils.substring(value2, 0, 2000));
        return message;
    }

    private List<Endpoint> findApplicableEndpoints(QName qname, String applicationId) {
        return StringUtils.isBlank(applicationId) ? getAllEndpoints(qname) : getEndpointsForAppId(qname, applicationId);
    }

    private List<Endpoint> getAllEndpoints(QName qname) {
        return getServiceBus().getEndpoints(qname);
    }

    private List<Endpoint> getEndpointsForAppId(QName qname, String applicationId) {
        return getServiceBus().getEndpoints(qname)
                .stream()
                .filter(e -> e.getServiceConfiguration().getApplicationId().equals(applicationId))
                .collect(Collectors.toList());
    }

    private boolean asyncModeOn() {
        return !KSBConstants.MESSAGING_SYNCHRONOUS.equals(getConfigurationService().getPropertyValueAsString(KSBConstants.Config.MESSAGE_DELIVERY));
    }

    private boolean messagingOn() {
        return !getConfigurationService().getPropertyValueAsBoolean(KSBConstants.Config.MESSAGING_OFF);
    }

    public ServiceBus getServiceBus() {
        return serviceBus;
    }

    public void setServiceBus(ServiceBus serviceBus) {
        this.serviceBus = serviceBus;
    }

    public MessageQueueService getMessageQueueService() {
        return messageQueueService;
    }

    public void setMessageQueueService(MessageQueueService messageQueueService) {
        this.messageQueueService = messageQueueService;
    }

    public ConfigurationService getConfigurationService() {
        return configurationService;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }
}
