/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.service;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.InitializingBean;

/**
 * Initializes authenticationService with user credentials.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class BasicAuthenticationCredentials implements InitializingBean {

    private BasicAuthenticationService authenticationService;
    private String serviceNameSpaceURI;
    private String localServiceName;
    private String username;
    private String password;

    /**
     * Register the service credentials for the service
     *
     * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        if (this.validate()) {
            this.getAuthenticationService().registerServiceCredentials(this);
        }
    }

    /**
     * Gets the BasicAuthenticationService to use
     *
     * @return BasicAuthenticationService the BasicAuthenticationService
     */
    public BasicAuthenticationService getAuthenticationService() {
        if (this.authenticationService == null) {
            this.authenticationService = KSBServiceLocator.getBasicAuthenticationService();
        }
        return authenticationService;
    }

    /**
     * Sets the authenticationService
     *
     * @param authenticationService the BasicAthenticationService to set
     */
    public void setAuthenticationService(BasicAuthenticationService authenticationService) {
        this.authenticationService = authenticationService;
    }

    /**
     * Gets the serviceNameSpaceURI
     *
     * @return String serviceNameSpaceURI of this BasicAuthenticationCredentials
     */
    public String getServiceNameSpaceURI() {
        return serviceNameSpaceURI;
    }

    /**
     * Sets the serviceNameSpaceURI
     *
     * @param serviceNameSpaceURI the serviceNameSpaceURI to set
     */
    public void setServiceNameSpaceURI(String serviceNameSpaceURI) {
        this.serviceNameSpaceURI = serviceNameSpaceURI;
    }

    /**
     * Gets the localServiceName
     *
     * @return String localServiceName of this BasicAuthenticationCredentials
     */
    public String getLocalServiceName() {
        return localServiceName;
    }

    /**
     * Sets the localServiceName
     *
     * @param localServiceName the localServiceName to set
     */
    public void setLocalServiceName(String localServiceName) {
        this.localServiceName = localServiceName;
    }

    /**
     * Gets the username
     *
     * @return username the username of this BasicAuthenticationCredentials
     */
    public String getUsername() {
        return username;
    }

    /**
     * Sets the username
     *
     * @param username the username to set
     */
    public void setUsername(String username) {
        this.username = username;
    }

    /**
     * Gets the password
     *
     * @return password the password of this BasicAuthenticationCredentials
     */
    public String getPassword() {
        return password;
    }

    /**
     * Sets the password
     *
     * @param password the password to set
     */
    public void setPassword(String password) {
        this.password = password;
    }

    /**
     * Validates authenticationService, serviceNameSpaceURI, localServiceName, username, and password
     *
     * @return true if the authentictionSerivce is not null and the serviceNameSpaceURI, localServiceName,
     *         username, and password are not blank
     */
    protected boolean validate() {
        return this.getAuthenticationService() != null &&
                StringUtils.isNotBlank(this.getServiceNameSpaceURI()) &&
                StringUtils.isNotBlank(this.getLocalServiceName()) &&
                StringUtils.isNotBlank(this.getUsername()) &&
                StringUtils.isNotBlank(this.getPassword());
    }
}
