/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.location.framework.country;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.util.ConcreteKeyValue;
import org.kuali.rice.core.api.util.KeyValue;
import org.kuali.rice.krad.keyvalues.KeyValuesBase;
import org.kuali.rice.location.api.country.Country;
import org.kuali.rice.location.api.services.LocationApiServiceLocator;

import java.util.ArrayList;
import java.util.List;

/**
 * An abstract KeyValuesBase for defining a values finder which produces a list of Countries.  Sub-classes should
 * extend this class and override {@link #retrieveCountriesForValuesFinder()} in order to produce a list of
 * countries to include.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public abstract class AbstractCountryValuesFinderBase extends KeyValuesBase {

    @Override
    public List<KeyValue> getKeyValues() {
        Country defaultCountry = getDefaultCountry();
        List<Country> countries = new ArrayList<>(retrieveCountriesForValuesFinder());

        List<KeyValue> values = new ArrayList<>(countries.size() + 1);
        values.add(new ConcreteKeyValue("", ""));
        if (defaultCountry != null) {
            values.add(new ConcreteKeyValue(defaultCountry.getCode(), getName(defaultCountry)));
        }

        countries.sort((country1, country2) -> {
            // some institutions may prefix the country name with an asterisk if the country no longer exists
            // the country names will be compared without the asterisk
            String sortValue1 = StringUtils.trim(StringUtils.removeStart(getName(country1), "*"));
            String sortValue2 = StringUtils.trim(StringUtils.removeStart(getName(country2), "*"));
            return sortValue1.compareToIgnoreCase(sortValue2);
        });

        // the default country may show up twice, but that's fine
        for (Country country : countries) {
            if (country.isActive()) {
                values.add(new ConcreteKeyValue(country.getCode(), getName(country)));
            }
        }
        return values;
    }

    private String getName(Country country) {
        return StringUtils.isNotBlank(country.getNameV3()) ? country.getNameV3() : country.getName();
    }

    /**
     * Returns a list of countries that will be added to the result of {@link #getKeyValues()}.  Note that the result
     * may be filtered by active status
     *
     * @return a List of countries to include in the values returned by this finder
     */
    protected abstract List<Country> retrieveCountriesForValuesFinder();

    /**
     * Returns the default country to use for this values finder.  If no default country is returned, none will be
     * used.  The default implementation of this method will defer to {@link org.kuali.rice.location.api.country.CountryService#getDefaultCountry()}.
     *
     * @return the default country to use for this values finder, or null if no default country should be used
     */
    protected Country getDefaultCountry() {
        return LocationApiServiceLocator.getCountryService().getDefaultCountry();
    }

}
