/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package software.amazon.awssdk.utils;

import static software.amazon.awssdk.utils.Validate.paramNotNull;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import software.amazon.awssdk.annotations.SdkProtectedApi;
import software.amazon.awssdk.annotations.SdkTestInternalApi;

/**
 * Utilities that make it easier to create, use and destroy
 * {@link ScheduledExecutor}s.
 */
@SdkProtectedApi
public final class ScheduledExecutorUtils {
    private ScheduledExecutorUtils() {
    }

    /**
     * Wrap a scheduled executor in a type that cannot be closed, or shut down.
     */
    public static ScheduledExecutorService unmanagedScheduledExecutor(ScheduledExecutorService executor) {
        if (executor instanceof UnmanagedScheduledExecutorService) {
            return executor;
        }
        if (executor == null) {
            return null;
        }
        return new UnmanagedScheduledExecutorService(executor);
    }

    /**
     * Unwrap a scheduled executor. Requires the UnmanagedScheduledExecutorService to be the "outer" type.
     */
    public static ScheduledExecutorService unwrapUnmanagedScheduledExecutor(ScheduledExecutorService executor) {
        if (executor instanceof UnmanagedScheduledExecutorService) {
            return ((UnmanagedScheduledExecutorService) executor).delegate;
        }
        return executor;
    }

    /**
     * Wrapper around {@link ScheduledExecutorService} to prevent it from being
     * closed. Used when the customer provides
     * a custom scheduled executor service in which case they are responsible for
     * the lifecycle of it.
     */
    @SdkTestInternalApi
    public static final class UnmanagedScheduledExecutorService implements ScheduledExecutorService {

        private final ScheduledExecutorService delegate;

        UnmanagedScheduledExecutorService(ScheduledExecutorService delegate) {
            this.delegate = paramNotNull(delegate, "ScheduledExecutorService");
        }

        public ScheduledExecutorService scheduledExecutorService() {
            return delegate;
        }

        @Override
        public void shutdown() {
            // Do nothing, this executor service is managed by the customer.
        }

        @Override
        public List<Runnable> shutdownNow() {
            return new ArrayList<>();
        }

        @Override
        public boolean isShutdown() {
            return delegate.isShutdown();
        }

        @Override
        public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
            return delegate.awaitTermination(timeout, unit);
        }

        @Override
        public ScheduledFuture<?> schedule(Runnable command, long delay, TimeUnit unit) {
            return delegate.schedule(command, delay, unit);
        }

        @Override
        public <V> ScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit unit) {
            return delegate.schedule(callable, delay, unit);
        }

        @Override
        public ScheduledFuture<?> scheduleAtFixedRate(Runnable command, long initialDelay, long period, TimeUnit unit) {
            return delegate.scheduleAtFixedRate(command, initialDelay, period, unit);
        }

        @Override
        public ScheduledFuture<?> scheduleWithFixedDelay(Runnable command, long initialDelay, long delay,
                TimeUnit unit) {
            return delegate.scheduleWithFixedDelay(command, initialDelay, delay, unit);
        }

        @Override
        public boolean isTerminated() {
            return delegate.isTerminated();
        }

        @Override
        public <T> Future<T> submit(Callable<T> task) {
            return delegate.submit(task);
        }

        @Override
        public <T> Future<T> submit(Runnable task, T result) {
            return delegate.submit(task, result);
        }

        @Override
        public Future<?> submit(Runnable task) {
            return delegate.submit(task);
        }

        @Override
        public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException {
            return delegate.invokeAll(tasks);
        }

        @Override
        public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
                throws InterruptedException {
            return delegate.invokeAll(tasks, timeout, unit);
        }

        @Override
        public <T> T invokeAny(Collection<? extends Callable<T>> tasks)
                throws InterruptedException, ExecutionException {
            return delegate.invokeAny(tasks);
        }

        @Override
        public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
                throws InterruptedException, ExecutionException, TimeoutException {
            return delegate.invokeAny(tasks, timeout, unit);
        }

        @Override
        public void execute(Runnable command) {
            delegate.execute(command);
        }
    }
}
