//[okhttp](../../../index.md)/[okhttp3](../index.md)/[HttpUrl](index.md)

# HttpUrl

[jvm]\
class [HttpUrl](index.md)

A uniform resource locator (URL) with a scheme of either `http` or `https`. Use this class to compose and decompose Internet addresses. For example, this code will compose and print a URL for Google search:

```java
HttpUrl url = new HttpUrl.Builder()
    .scheme("https")
    .host("www.google.com")
    .addPathSegment("search")
    .addQueryParameter("q", "polar bears")
    .build();
System.out.println(url);
```

which prints:

```kotlin
https://www.google.com/search?q=polar%20bears
```

As another example, this code prints the human-readable query parameters of a Twitter search:

```java
HttpUrl url = HttpUrl.parse("https://twitter.com/search?q=cute%20%23puppies&f=images");
for (int i = 0, size = url.querySize(); i < size; i++) {
  System.out.println(url.queryParameterName(i) + ": " + url.queryParameterValue(i));
}
```

which prints:

```kotlin
q: cute #puppies
f: images
```

In addition to composing URLs from their component parts and decomposing URLs into their component parts, this class implements relative URL resolution: what address you'd reach by clicking a relative link on a specified page. For example:

```java
HttpUrl base = HttpUrl.parse("https://www.youtube.com/user/WatchTheDaily/videos");
HttpUrl link = base.resolve("../../watch?v=cbP2N1BQdYc");
System.out.println(link);
```

which prints:

```kotlin
https://www.youtube.com/watch?v=cbP2N1BQdYc
```

##  What's in a URL?

A URL has several components.

###  Scheme

Sometimes referred to as *protocol*, A URL's scheme describes what mechanism should be used to retrieve the resource. Although URLs have many schemes (`mailto`, `file`, `ftp`), this class only supports `http` and `https`. Use [java.net.URI](https://docs.oracle.com/javase/8/docs/api/java/net/URI.html) for URLs with arbitrary schemes.

###  Username and Password

Username and password are either present, or the empty string `""` if absent. This class offers no mechanism to differentiate empty from absent. Neither of these components are popular in practice. Typically HTTP applications use other mechanisms for user identification and authentication.

###  Host

The host identifies the webserver that serves the URL's resource. It is either a hostname like `square.com` or `localhost`, an IPv4 address like `192.168.0.1`, or an IPv6 address like `::1`.

Usually a webserver is reachable with multiple identifiers: its IP addresses, registered domain names, and even `localhost` when connecting from the server itself. Each of a web server's names is a distinct URL and they are not interchangeable. For example, even if `http://square.github.io/dagger` and `http://google.github.io/dagger` are served by the same IP address, the two URLs identify different resources.

###  Port

The port used to connect to the web server. By default this is 80 for HTTP and 443 for HTTPS. This class never returns -1 for the port: if no port is explicitly specified in the URL then the scheme's default is used.

###  Path

The path identifies a specific resource on the host. Paths have a hierarchical structure like &quot;/square/okhttp/issues/1486&quot; and decompose into a list of segments like `["square", "okhttp", "issues", "1486"]`.

This class offers methods to compose and decompose paths by segment. It composes each path from a list of segments by alternating between &quot;/&quot; and the encoded segment. For example the segments `["a", "b"]` build &quot;/a/b&quot; and the segments `["a", "b", ""]` build &quot;/a/b/&quot;.

If a path's last segment is the empty string then the path ends with &quot;/&quot;. This class always builds non-empty paths: if the path is omitted it defaults to &quot;/&quot;. The default path's segment list is a single empty string: `[""]`.

###  Query

The query is optional: it can be null, empty, or non-empty. For many HTTP URLs the query string is subdivided into a collection of name-value parameters. This class offers methods to set the query as the single string, or as individual name-value parameters. With name-value parameters the values are optional and names may be repeated.

###  Fragment

The fragment is optional: it can be null, empty, or non-empty. Unlike host, port, path, and query the fragment is not sent to the webserver: it's private to the client.

##  Encoding

Each component must be encoded before it is embedded in the complete URL. As we saw above, the string `cute #puppies` is encoded as `cute%20%23puppies` when used as a query parameter value.

###  Percent encoding

Percent encoding replaces a character (like `\ud83c\udf69`) with its UTF-8 hex bytes (like `%F0%9F%8D%A9`). This approach works for whitespace characters, control characters, non-ASCII characters, and characters that already have another meaning in a particular context.

Percent encoding is used in every URL component except for the hostname. But the set of characters that need to be encoded is different for each component. For example, the path component must escape all of its `?` characters, otherwise it could be interpreted as the start of the URL's query. But within the query and fragment components, the `?` character doesn't delimit anything and doesn't need to be escaped.

```java
HttpUrl url = HttpUrl.parse("http://who-let-the-dogs.out").newBuilder()
    .addPathSegment("_Who?_")
    .query("_Who?_")
    .fragment("_Who?_")
    .build();
System.out.println(url);
```

This prints:

```kotlin
http://who-let-the-dogs.out/_Who%3F_?_Who?_#_Who?_
```

When parsing URLs that lack percent encoding where it is required, this class will percent encode the offending characters.

###  IDNA Mapping and Punycode encoding

Hostnames have different requirements and use a different encoding scheme. It consists of IDNA mapping and Punycode encoding.

In order to avoid confusion and discourage phishing attacks, [IDNA Mapping](http://www.unicode.org/reports/tr46/#ToASCII) transforms names to avoid confusing characters. This includes basic case folding: transforming shouting `SQUARE.COM` into cool and casual `square.com`. It also handles more exotic characters. For example, the Unicode trademark sign (™) could be confused for the letters &quot;TM&quot; in `http://ho™ail.com`. To mitigate this, the single character (™) maps to the string (tm). There is similar policy for all of the 1.1 million Unicode code points. Note that some code points such as &quot;\ud83c\udf69&quot; are not mapped and cannot be used in a hostname.

[Punycode](http://ietf.org/rfc/rfc3492.txt) converts a Unicode string to an ASCII string to make international domain names work everywhere. For example, &quot;σ&quot; encodes as &quot;xn--4xa&quot;. The encoded string is not human readable, but can be used with classes like [InetAddress](https://docs.oracle.com/javase/8/docs/api/java/net/InetAddress.html) to establish connections.

##  Why another URL model?

Java includes both [java.net.URL](https://docs.oracle.com/javase/8/docs/api/java/net/URL.html) and [java.net.URI](https://docs.oracle.com/javase/8/docs/api/java/net/URI.html). We offer a new URL model to address problems that the others don't.

###  Different URLs should be different

Although they have different content, `java.net.URL` considers the following two URLs equal, and the [equals()](https://docs.oracle.com/javase/8/docs/api/java/lang/Object.html#equals-kotlin.Any?-) method between them returns true:

- 
   https://example.net/
- 
   https://example.com/

This is because those two hosts share the same IP address. This is an old, bad design decision that makes `java.net.URL` unusable for many things. It shouldn't be used as a [Map](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.collections/-map/index.html) key or in a [Set](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.collections/-set/index.html). Doing so is both inefficient because equality may require a DNS lookup, and incorrect because unequal URLs may be equal because of how they are hosted.

###  Equal URLs should be equal

These two URLs are semantically identical, but `java.net.URI` disagrees:

- 
   http://host:80/
- 
   http://host

Both the unnecessary port specification (`:80`) and the absent trailing slash (`/`) cause URI to bucket the two URLs separately. This harms URI's usefulness in collections. Any application that stores information-per-URL will need to either canonicalize manually, or suffer unnecessary redundancy for such URLs.

Because they don't attempt canonical form, these classes are surprisingly difficult to use securely. Suppose you're building a webservice that checks that incoming paths are prefixed &quot;/static/images/&quot; before serving the corresponding assets from the filesystem.

```java
String attack = "http://example.com/static/images/../../../../../etc/passwd";
System.out.println(new URL(attack).getPath());
System.out.println(new URI(attack).getPath());
System.out.println(HttpUrl.parse(attack).encodedPath());
```

By canonicalizing the input paths, they are complicit in directory traversal attacks. Code that checks only the path prefix may suffer!

```kotlin
/static/images/../../../../../etc/passwd
/static/images/../../../../../etc/passwd
/etc/passwd
```

###  If it works on the web, it should work in your application

The `java.net.URI` class is strict around what URLs it accepts. It rejects URLs like `http://example.com/abc|def` because the `|` character is unsupported. This class is more forgiving: it will automatically percent-encode the `|'` yielding `http://example.com/abc%7Cdef`. This kind behavior is consistent with web browsers. `HttpUrl` prefers consistency with major web browsers over consistency with obsolete specifications.

###  Paths and Queries should decompose

Neither of the built-in URL models offer direct access to path segments or query parameters. Manually using `StringBuilder` to assemble these components is cumbersome: do '+' characters get silently replaced with spaces? If a query parameter contains a '&', does that get escaped? By offering methods to read and write individual query parameters directly, application developers are saved from the hassles of encoding and decoding.

###  Plus a modern API

The URL (JDK1.0) and URI (Java 1.4) classes predate builders and instead use telescoping constructors. For example, there's no API to compose a URI with a custom port without also providing a query and fragment.

Instances of [HttpUrl](index.md) are well-formed and always have a scheme, host, and path. With `java.net.URL` it's possible to create an awkward URL like `http:/` with scheme and path but no hostname. Building APIs that consume such malformed values is difficult!

This class has a modern API. It avoids punitive checked exceptions: [toHttpUrl](-companion/to-http-url.md) throws [IllegalArgumentException](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-illegal-argument-exception/index.html) on invalid input or [toHttpUrlOrNull](-companion/to-http-url-or-null.md) returns null if the input is an invalid URL. You can even be explicit about whether each component has been encoded already.

## Types

| Name | Summary |
|---|---|
| [Builder](-builder/index.md) | [jvm]<br>class [Builder](-builder/index.md) |
| [Companion](-companion/index.md) | [jvm]<br>object [Companion](-companion/index.md) |

## Functions

| Name | Summary |
|---|---|
| [equals](equals.md) | [jvm]<br>open operator override fun [equals](equals.md)(other: [Any](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-any/index.html)?): [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html) |
| [hashCode](hash-code.md) | [jvm]<br>open override fun [hashCode](hash-code.md)(): [Int](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-int/index.html) |
| [newBuilder](new-builder.md) | [jvm]<br>fun [newBuilder](new-builder.md)(): [HttpUrl.Builder](-builder/index.md)<br>Returns a builder based on this URL.<br>[jvm]<br>fun [newBuilder](new-builder.md)(link: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)): [HttpUrl.Builder](-builder/index.md)?<br>Returns a builder for the URL that would be retrieved by following `link` from this URL, or null if the resulting URL is not well-formed. |
| [queryParameter](query-parameter.md) | [jvm]<br>fun [queryParameter](query-parameter.md)(name: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)?<br>The first query parameter named `name` decoded using UTF-8, or null if there is no such query parameter. |
| [queryParameterName](query-parameter-name.md) | [jvm]<br>fun [queryParameterName](query-parameter-name.md)(index: [Int](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-int/index.html)): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>Returns the name of the query parameter at `index`. For example this returns `"a"` for `queryParameterName(0)` on `http://host/?a=apple&b=banana`. This throws if `index` is not less than the [query size](query-size.md). |
| [queryParameterValue](query-parameter-value.md) | [jvm]<br>fun [queryParameterValue](query-parameter-value.md)(index: [Int](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-int/index.html)): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)?<br>Returns the value of the query parameter at `index`. For example this returns `"apple"` for `queryParameterName(0)` on `http://host/?a=apple&b=banana`. This throws if `index` is not less than the [query size](query-size.md). |
| [queryParameterValues](query-parameter-values.md) | [jvm]<br>fun [queryParameterValues](query-parameter-values.md)(name: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)): [List](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.collections/-list/index.html)&lt;[String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)?&gt;<br>Returns all values for the query parameter `name` ordered by their appearance in this URL. For example this returns `["banana"]` for `queryParameterValue("b")` on `http://host/?a=apple&b=banana`. |
| [redact](redact.md) | [jvm]<br>fun [redact](redact.md)(): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>Returns a string with containing this URL with its username, password, query, and fragment stripped, and its path replaced with `/...`. For example, redacting `http://username:password@example.com/path` returns `http://example.com/...`. |
| [resolve](resolve.md) | [jvm]<br>fun [resolve](resolve.md)(link: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)): [HttpUrl](index.md)?<br>Returns the URL that would be retrieved by following `link` from this URL, or null if the resulting URL is not well-formed. |
| [topPrivateDomain](top-private-domain.md) | [jvm]<br>fun [topPrivateDomain](top-private-domain.md)(): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)?<br>Returns the domain name of this URL's [host](host.md) that is one level beneath the public suffix by consulting the [public suffix list](https://publicsuffix.org). Returns null if this URL's [host](host.md) is an IP address or is considered a public suffix by the public suffix list. |
| [toString](to-string.md) | [jvm]<br>open override fun [toString](to-string.md)(): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html) |
| [toUri](to-uri.md) | [jvm]<br>@[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;uri&quot;)<br>fun [toUri](to-uri.md)(): [URI](https://docs.oracle.com/javase/8/docs/api/java/net/URI.html)<br>Returns this URL as a [java.net.URI](https://docs.oracle.com/javase/8/docs/api/java/net/URI.html). Because `URI` is more strict than this class, the returned URI may be semantically different from this URL: |
| [toUrl](to-url.md) | [jvm]<br>@[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;url&quot;)<br>fun [toUrl](to-url.md)(): [URL](https://docs.oracle.com/javase/8/docs/api/java/net/URL.html)<br>Returns this URL as a [java.net.URL](https://docs.oracle.com/javase/8/docs/api/java/net/URL.html). |

## Properties

| Name | Summary |
|---|---|
| [encodedFragment](encoded-fragment.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;encodedFragment&quot;)<br>val [encodedFragment](encoded-fragment.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)?<br>This URL's encoded fragment, like `"abc"` for `http://host/#abc`. This is null if the URL has no fragment. |
| [encodedPassword](encoded-password.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;encodedPassword&quot;)<br>val [encodedPassword](encoded-password.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>The password, or an empty string if none is set. |
| [encodedPath](encoded-path.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;encodedPath&quot;)<br>val [encodedPath](encoded-path.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>The entire path of this URL encoded for use in HTTP resource resolution. The returned path will start with `"/"`. |
| [encodedPathSegments](encoded-path-segments.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;encodedPathSegments&quot;)<br>val [encodedPathSegments](encoded-path-segments.md): [List](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.collections/-list/index.html)&lt;[String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)&gt;<br>A list of encoded path segments like `["a", "b", "c"]` for the URL `http://host/a/b/c`. This list is never empty though it may contain a single empty string. |
| [encodedQuery](encoded-query.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;encodedQuery&quot;)<br>val [encodedQuery](encoded-query.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)?<br>The query of this URL, encoded for use in HTTP resource resolution. This string may be null (for URLs with no query), empty (for URLs with an empty query) or non-empty (all other URLs). |
| [encodedUsername](encoded-username.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;encodedUsername&quot;)<br>val [encodedUsername](encoded-username.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>The username, or an empty string if none is set. |
| [fragment](fragment.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;fragment&quot;)<br>val [fragment](fragment.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)?<br>This URL's fragment, like `"abc"` for `http://host/#abc`. This is null if the URL has no fragment. |
| [host](host.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;host&quot;)<br>val [host](host.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>The host address suitable for use with [InetAddress.getAllByName](https://docs.oracle.com/javase/8/docs/api/java/net/InetAddress.html#getAllByName-kotlin.String-). May be: |
| [isHttps](is-https.md) | [jvm]<br>val [isHttps](is-https.md): [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html) |
| [password](password.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;password&quot;)<br>val [password](password.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>Returns the decoded password, or an empty string if none is present. |
| [pathSegments](path-segments.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;pathSegments&quot;)<br>val [pathSegments](path-segments.md): [List](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.collections/-list/index.html)&lt;[String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)&gt;<br>A list of path segments like `["a", "b", "c"]` for the URL `http://host/a/b/c`. This list is never empty though it may contain a single empty string. |
| [pathSize](path-size.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;pathSize&quot;)<br>val [pathSize](path-size.md): [Int](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-int/index.html)<br>The number of segments in this URL's path. This is also the number of slashes in this URL's path, like 3 in `http://host/a/b/c`. This is always at least 1. |
| [port](port.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;port&quot;)<br>val [port](port.md): [Int](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-int/index.html)<br>The explicitly-specified port if one was provided, or the default port for this URL's scheme. For example, this returns 8443 for `https://square.com:8443/` and 443 for `https://square.com/`. The result is in `[1..65535]`. |
| [query](query.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;query&quot;)<br>val [query](query.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)?<br>This URL's query, like `"abc"` for `http://host/?abc`. Most callers should prefer [queryParameterName](query-parameter-name.md) and [queryParameterValue](query-parameter-value.md) because these methods offer direct access to individual query parameters. |
| [queryParameterNames](query-parameter-names.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;queryParameterNames&quot;)<br>val [queryParameterNames](query-parameter-names.md): [Set](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.collections/-set/index.html)&lt;[String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)&gt;<br>The distinct query parameter names in this URL, like `["a", "b"]` for `http://host/?a=apple&b=banana`. If this URL has no query this is the empty set. |
| [querySize](query-size.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;querySize&quot;)<br>val [querySize](query-size.md): [Int](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-int/index.html)<br>The number of query parameters in this URL, like 2 for `http://host/?a=apple&b=banana`. If this URL has no query this is 0. Otherwise it is one more than the number of `"&"` separators in the query. |
| [scheme](scheme.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;scheme&quot;)<br>val [scheme](scheme.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>Either &quot;http&quot; or &quot;https&quot;. |
| [username](username.md) | [jvm]<br>@get:[JvmName](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;username&quot;)<br>val [username](username.md): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>The decoded username, or an empty string if none is present. |
