/*
 * Copyright 2010-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package software.amazon.awssdk.core.client.builder;

import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.http.SdkHttpClient;

/**
 * This includes required and optional override configuration required by every sync client builder. An instance can be acquired
 * by calling the static "builder" method on the type of sync client you wish to create.
 *
 * <p>Implementations of this interface are mutable and not thread-safe.</p>
 *
 * @param <B> The type of builder that should be returned by the fluent builder methods in this interface.
 * @param <C> The type of client generated by this builder.
 */
@SdkPublicApi
public interface SdkSyncClientBuilder<B extends SdkSyncClientBuilder<B, C>, C> {
    /**
     * Sets the {@link SdkHttpClient} that the SDK service client will use to make HTTP calls. This HTTP client may be
     * shared between multiple SDK service clients to share a common connection pool. To create a client you must use an
     * implementation-specific builder. Note that this method is only recommended when you wish to share an HTTP client across
     * multiple SDK service clients. If you do not wish to share HTTP clients, it is recommended to use
     * {@link #httpClientBuilder(SdkHttpClient.Builder)} so that service-specific default configuration may be applied.
     *
     * <p>
     * <b>This client must be closed by the user when it is ready to be disposed. The SDK will not close the HTTP client
     * when the service client is closed.</b>
     * </p>
     */
    B httpClient(SdkHttpClient httpClient);

    /**
     * Sets a {@link SdkHttpClient.Builder} that will be used to obtain a configured instance of {@link SdkHttpClient}. Any
     * service-specific HTTP configuration will be merged with the builder's configuration prior to creating the client. When
     * there is no desire to share HTTP clients across multiple service clients, the client builder is the preferred way to
     * customize the HTTP client as it benefits from service-specific default configuration.
     *
     * <p>
     * <b>Clients created by the builder are managed by the SDK and will be closed when the service client is closed.</b>
     * </p>
     */
    B httpClientBuilder(SdkHttpClient.Builder httpClientBuilder);
}
